/* ****************************************************************************
 *
 *	File: CosStream.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2003-2006 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/

package com.adobe.internal.pdftoolkit.core.cos;

import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.zip.ZipException;

import com.adobe.internal.io.ByteWriterFactory.Fixed;
import com.adobe.internal.io.stream.IO;
import com.adobe.internal.io.stream.InputByteStream;
import com.adobe.internal.io.stream.OutputByteStream;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFCosParseException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFIOException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFSecurityException;
import com.adobe.internal.pdftoolkit.core.filter.CustomFilterRegistry;
import com.adobe.internal.pdftoolkit.core.filter.FilterParams;
import com.adobe.internal.pdftoolkit.core.filter.FilterStream;
import com.adobe.internal.pdftoolkit.core.securityframework.EncryptionHandlerState;
import com.adobe.internal.pdftoolkit.core.types.ASName;
import com.adobe.internal.pdftoolkit.core.util.BooleanHolder;
import com.adobe.internal.pdftoolkit.core.util.StringOps;

/**
 * Represents a COS stream as defined in section 3.2.7 of the PDF Reference
 * Manual version 1.4.
 * The stream data can be represented in encoded or decoded form.
 * In a PDF file the stream data are encoded according to the value of the
 * /Filters entry in the stream dictionary [if one exists].
 * For those rare occasions when an application processes encoded stream data
 * (mostly when it copies the stream from one external location to another)
 * the CosStream class contains getStreamEncoded(...) and newDataEncoded(...) methods.
 * These methods deliver/accept data in the encoded form.
 * When the data are placed in the CosStream object with newDataEncoded(...)
 * methods, the write() method does not encode the stream data before writing them
 * to the PDF file.
 * The consumers of the CosStream class should not mix CosStream methods that
 * deliver/accept the stream data in the decoded and encoded form for the same
 * CosStream object.
 */
public class CosStream extends CosDictionary
{
	private boolean mDataInPDF;		// True if data from stream is in main PDF file (at position mPos)
	private long mPos;			// Starting position in original file of the stream data
	private InputByteStream mDataStream;	// Cached stream data
	private boolean mDataEncoded;		// Cached stream data is encoded
	private boolean mToEncrypt;		// Encrypt only when true
	private boolean mIsEncrypted;		// Decrypt only when true
	private CosArray mOutputFilters;	// New filters to use at save time. This is a CosArray of CosArrays each of size 2 having filter name (CosName) and
										// filter parameters.(CosDictionary or CosNull)
	private ASName mCryptFilter;		// Cached crypt filter name
	private boolean mCryptFilterSet;	// Cached crypt filter name has been set

	/**
	 * Constructs a stream object.
	 *
	 * @param doc		Document containing the stream
	 * @param map		Map for the dictionary (ASName key - CosObject value pairs)
	 * @param info		Object info for the stream
	 * @param pos		Starting position in the stream
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	CosStream(CosDocument doc, Map map, CosObjectInfo info, long pos)
		throws PDFCosParseException, PDFIOException, IOException, PDFSecurityException
	{
		super(doc, map, info);
		init(true, pos, null, true, true);
	}

	/**
	 * Construct a stream object that can be written into. The repository
	 * passed in should be one that allows appending.
	 * @param doc
	 * @param info
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	CosStream(CosDocument doc, CosObjectInfo info)
		throws PDFCosParseException, IOException, PDFIOException, PDFSecurityException
	{
		super(doc, new LinkedHashMap(), info);
		init(false, 0, null, true, true);
	}

	/**
	 * Construct a stream object that can be written into. The repository
	 * passed in should be one that allows appending.
	 * @param info
	 * @param rep
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	CosStream(CosDocument doc, CosObjectInfo info, InputByteStream rep)
		throws PDFCosParseException, IOException, PDFIOException, PDFSecurityException
	{
		super(doc, new LinkedHashMap(), info);
		init(false, 0, rep, true, true);
	}

	private void init(boolean dataInPDF, long pos, InputByteStream decoded, boolean toEncrypt, boolean isEncrypted)
		throws PDFCosParseException, IOException, PDFIOException, PDFSecurityException
	{
		mDataInPDF = dataInPDF;
		mPos = pos;
		if (!dataInPDF) {
			setCachedStream(false, decoded);
			put(ASName.k_Length, 0);
		} else {
			// Length does not exist or is not a numeric, and we need to detect it in some way
			CosObject lenObj = get(ASName.k_Length);
			if(!(lenObj instanceof CosNumeric)) {
				long length = CosRepairUtils.getStreamLength(getDocument(), pos);
				if(length == -1)
					throw new PDFCosParseException("Expected 'endstream' for stream at pos " + pos);
				setRepairedValue(ASName.k_Length, getDocument().createCosNumeric(length));
			}
		}
		mToEncrypt = toEncrypt;
		mIsEncrypted = isEncrypted;
	}

	/**
	 * Close the CosObject and release any and all resources associated with it.
	 * Any use of this CosObject after this call is considered an error.
	 * @throws IOException
	 */
	@Override
	void close()
		throws IOException
	{
		releaseStreams();
		mOutputFilters = null;
		super.close();
	}

	/**
	 * Release resources used by the CosObject that can be recreated later.
	 * @throws IOException
	 */
	@Override
	void release()
		throws IOException
	{
		releaseStreams();
		mDataInPDF = true;
		mIsEncrypted = mToEncrypt;
		super.release();
	}

	/**
	 * Release the InputByteStream used.
	 * @throws IOException
	 */
	private void releaseStreams()
		throws IOException
	{
		setCachedStream(false, null);
	}

	private InputByteStream getCachedStream(boolean encoded)
	{
		if (encoded != mDataEncoded)
			return null;
		return mDataStream;
	}

	private void setCachedStream(boolean encoded, InputByteStream newStream)
		throws IOException
	{
		if (mDataStream != null)
			mDataStream.close();
		mDataStream = newStream;
		mDataEncoded = encoded;
	}

	/**
	 * 
	 * This method is overridden from CosDictionary in order to catch any
	 * changes that might affect the ability to decode the stream body
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	@Override
	public CosObject put(ASName key, CosObject cosObject)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		try {
			if ((mDataInPDF || mDataEncoded) && (key == ASName.k_Filter || key == ASName.k_DecodeParms)) {
				CosObject oldValue = get(key);
				if (!cosObject.equals(oldValue)) {
					InputByteStream stm = getStream(false, false, true);
					mCryptFilterSet = false;
					setCachedStream(false, stm);
					mDataInPDF = false;
				}
			}
			return super.put(key, cosObject);
		} catch (IOException e) {
			throw new PDFIOException(e);
		}
	}

	@Override
	public CosObject remove(ASName key)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		try  {
			if ((mDataInPDF || mDataEncoded) && (key == ASName.k_Filter || key == ASName.k_DecodeParms)) {
				CosObject oldValue = get(key);
				if (oldValue != null) {
					InputByteStream stm = getStream(false, false, true);
					mCryptFilterSet = false;
					setCachedStream(false, stm);
					mDataInPDF = false;
				}
			}
			return super.remove(key);
		} catch (IOException e) {
			throw new PDFIOException(e);
		}
	}

	/**
	 * return the type of this CosObject
	 */
	@Override
	public int getType()
	{
		return t_Stream;
	}
	
	
	/** 
	 * return the length of the CosStream.
	 * @throws PDFSecurityException 
	 * @throws PDFIOException 
	 * @throws PDFCosParseException 
	 */
	public long getLength() throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		return getLong(ASName.k_Length);
	}

	/**
	 * Obtains a slice of the underlying InputByteStream. The caller is responsible
	 * for closing this InputByteStream when they are finished with it.
	 *
	 * @return InputByteStream for reading the COS stream's data. The stream is passed through any registered filters.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public InputByteStream getStreamDecoded()
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		try {
			return getStream(false, true, true);
		} catch (IOException e) {
			throw new PDFIOException(e);
		}
	}
	
	/**
	 * Returns underlying stream as {@link InputStream}. If the stream is encrypted, then a copy of
	 * decrypted stream is returned. In case of filters, filter stream is returned instead of copying it.
	 * The caller is responsible for closing this InputStream when they are finished with it.
	 * @return {@link InputStream}
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public InputStream getStreamDecodedNoCopying()
			throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		InputByteStream ibs = null;
		OutputByteStream os = null;
		try{
			try {
				BooleanHolder wasDecrypted = new BooleanHolder(false);
				InputStream is = this.getStreamForCopying(false, wasDecrypted);
				if(wasDecrypted.getValue()){
					os = getStreamManager().getUnregisteredOutputByteStream(Fixed.GROWABLE, is.available());
					copyStream(os, is, false);
					os.flush();
					ibs = os.closeAndConvert();
					os = null;
					return ibs.toInputStream();
				}else
					return is;
			} finally{
				if(os != null)
					os.close();
				if(ibs != null)
					ibs.close();
			}
		}catch (IOException e) {
			throw new PDFIOException(e);
		}
	}

	/**
	 * Obtains a slice of the underlying InputByteStream. The caller is responsible
	 * for closing this InputByteStream when they are finished with it.
	 *
	 * @return ByteStream for reading the COS stream's data without passing them through any registered filters.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public InputByteStream getStreamEncoded()
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		try {
			return getStream(true, true, true);
		} catch (IOException e) {
			throw new PDFIOException(e);
		}
	}

	/**
	 * Copy the stream data into the given OutputStream.  This is intended for use
	 * in exporting data for use outside of the PDF Toolkit.
	 * @param destStm the destination for the stream data
	 * @param encoded copy the data as encoded if true; decoded if false
	 * @return true if data was copied into the destination; false otherwise
	 */
	public<T> boolean copyStream(T destStm, boolean encoded)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		try{
			return copyStream(destStm, this.getStreamForCopying(encoded, new BooleanHolder(false)), encoded);
		}catch (IOException e) {
			throw new PDFIOException(e);
		}
	}

	private<T> boolean copyStream(T destStm, InputStream srcStm, boolean encoded)
			throws PDFCosParseException, PDFIOException, PDFSecurityException{
		InputStream is = null;
		try {
			is = srcStm != null ? srcStm : this.getStreamForCopying(encoded, new BooleanHolder(false));
			if (is != null) {
				long bytesCopied = 0;
				try {
					if(destStm instanceof OutputStream)
						bytesCopied = IO.copy(is, (OutputStream)destStm);
					else if(destStm instanceof OutputByteStream)
						bytesCopied = IO.copy(is, (OutputByteStream)destStm);
					else
						throw new PDFIOException("Destination stream must be either OutputStream or OutputByteStream.");
				} catch (EOFException e) {
					bytesCopied = 1;	// An empty stream won't generate this
				} catch (ZipException e) {
					throw new PDFCosParseException(e);
				}
				return bytesCopied != 0;
			}
			return false;
		} catch (IOException e) {
			throw new PDFIOException(e);
		}finally{
			try {
				if(is != null)
					is.close();
			} catch (IOException e) {
				throw new PDFIOException(e);
			}
		}
	}
	/**
	 * Returns encoded/decoded stream based on the boolean passed here.
	 */
	private InputStream getStreamForCopying(boolean encoded, BooleanHolder wasDecrypted)
		throws PDFCosParseException, IOException, PDFIOException, PDFSecurityException
	{
		InputStream is = null;
		InputByteStream ibs = getCachedStream(encoded);
			if (ibs == null) {
				if (mDataInPDF) {
					ibs = getStreamDataFromPDF();
					is = ibs.toInputStream();
					boolean hasAcroBug = false;
					if (needsDecryption(ibs)) {
						hasAcroBug = checkForAcroBug();
						if (!hasAcroBug) {
							EncryptionHandlerState encryptHandle = getDocument().getEncryption().getStreamDecryptionStateHandler(this, ibs);
							is = new DecryptingInputStream(this, is, encryptHandle);
							wasDecrypted.setValue(true);
						}
					}
					if (needsDecoding(encoded)) {
						is = buildInputFilterStream(is);
						if (hasAcroBug) {
							EncryptionHandlerState encryptHandle = getDocument().getEncryption().getStreamDecryptionStateHandler(this, ibs);
							is = new DecryptingInputStream(this, is, encryptHandle);
							wasDecrypted.setValue(true);
						}
					}
				} else {
					if (!encoded) {
						ibs = getCachedStream(true);
						if (ibs != null)
							is = buildInputFilterStream(ibs.toInputStream());
					}
				}
			}
			if (is != null)
				return is;
			else if (ibs != null)
				return ibs.toInputStream();
			return null;
	}
	
	InputByteStream getStream(boolean encoded, boolean slice, boolean register)
		throws PDFCosParseException, IOException, PDFIOException, PDFSecurityException
	{
		InputByteStream ibs = getCachedStream(encoded);
		if (ibs == null) {
			if (mDataInPDF) {
				ibs = getStreamDataFromPDF();
				if (needsDecryption(ibs) || needsDecoding(encoded)) {
					ibs.close();
					ibs = null;
				}
			}
			if (ibs == null) {
				OutputByteStream obs;
				if (register)
					obs = getStreamManager().getOutputByteStreamDecryptedDocument(Fixed.GROWABLE, getLong(ASName.k_Length));
				else
					obs = getStreamManager().getUnregisteredOutputByteStream(Fixed.GROWABLE, getInt(ASName.k_Length));
				copyStream(obs, encoded);
				ibs = obs.closeAndConvert();
			} else {
				if (slice) {
					ibs = ibs.slice();
				}
			}
		} else {
			if (slice) {
				ibs = ibs.slice();
			}
		}
		return ibs;
	}
	
	private InputByteStream getStreamDataFromPDF() throws PDFIOException, PDFCosParseException, PDFSecurityException{
		CosDocument doc = getDocument();
		Long length = getLong(ASName.k_Length);
		InputByteStream ibs = null;
		if(length == null)
		{
			// The stream dictionary does not have a length entry. If the document uses
			// repair list, repair the document. Else throw an PDFCosParseEx
			if(doc.getUseRepairList())
			{
				return setRepairedStreamLength(doc, mPos);
			}
			else
			{
				throw new PDFCosParseException("CosStream with object number -  " + getObjNum() + " does not have a length entry.");
			}
		}
		try {
			ibs = doc.getStream(mPos, length);
			if(doc.getUseRepairList())
			{
				// If repair is enabled on the doc, check if the stream end is valid, as 
				// parsing invalid streams can end up in lot of trouble.
				boolean isStreamLengthOk = CosRepairUtils.isStreamEndValid(doc, mPos + length); 
				if(! isStreamLengthOk)
				{
					// If stream length is not ok, fix it.
					ibs = setRepairedStreamLength(doc, mPos);
				}
			}
			
		} catch (PDFIOException e) {
			if(doc.getUseRepairList())
			{
				// Some exception occurred while loading the stream. This is known to happen
				// is length entry in stream is huge and we end up reading data beyond EOF.
				// Calculate and set correct stream length.
				ibs = setRepairedStreamLength(doc, mPos);
			}
			else
			{
				// Propagate the exception
				throw new PDFCosParseException(e);
			}
		}
		return ibs;
	}
	// Sets repaired stream length and gets the underlying InputByteStream corresponding to this length
	private InputByteStream setRepairedStreamLength(CosDocument doc, long currentPos)
	throws PDFIOException
	{
		long actuallength = CosRepairUtils.getStreamLength(getDocument(), mPos);		
		InputByteStream ibs = getDocument().getStream(mPos, actuallength);		
		setRepairedValue(ASName.k_Length, getDocument().createCosNumeric(actuallength));
		
		return ibs;
	}

	protected void adjustPos(long offset)
	{
		mPos += offset;
	}

	ASName getCryptFilter()
		throws PDFCosParseException, IOException, PDFIOException, PDFSecurityException
	{
		if (mCryptFilterSet)
			return mCryptFilter;
		ASName filterName = null;
		CosObject filter = get(ASName.k_Filter);
		if (filter != null) {
			if (filter.getType() == CosObject.t_Name) {
				if (((CosName)filter).nameValue().equals(ASName.k_Crypt)) {
					CosDictionary params = getCosDictionary(ASName.k_DecodeParms);
					filterName = params != null ? params.getName(ASName.k_Name) : ASName.k_Identity;
				}
			}
			else if (filter.getType() == CosObject.t_Array) {
				int cryptInd = ((CosArray)filter).findName(ASName.k_Crypt);
				if (cryptInd >= 0) {
					CosArray params = getCosArray(ASName.k_DecodeParms);
					if (params != null) {
						filterName = params.getCosDictionary(cryptInd).getName(ASName.k_Name);
						if (filterName == null) {
							filterName = ASName.k_Identity;
						}
					}
					else { // use Identity by default
						filterName = ASName.k_Identity;
					}
				}
			}
		}
		mCryptFilterSet = true;
		mCryptFilter = filterName;
		return filterName;
	}

	private boolean checkForAcroBug()
		throws PDFCosParseException, IOException, PDFIOException, PDFSecurityException
	{
		CosObject filter = get(ASName.k_Filter);
		if (filter instanceof CosArray && ((CosArray)filter).findName(ASName.k_Crypt) > 0)
			return true;
		return false;
	}
	
	private ASName getCryptFilter(CosArray filterList)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		ASName filterName = null;
		if (filterList != null) {
			Iterator<CosObject> filterIter = filterList.iterator();
			while (filterIter.hasNext()) {
				CosArray filterItem = (CosArray) filterIter.next();
				if (((CosName)filterItem.get(0)).nameValue() != ASName.k_Crypt)
					continue;
				CosObject params = filterItem.get(1);
				filterName = (params instanceof CosDictionary) ? ((CosDictionary)params).getName(ASName.k_Name) : ASName.k_Identity;
				break;
			}
			mCryptFilterSet = true;
			mCryptFilter = filterName;
		}
		return filterName;
	}
	
	private boolean needsDecoding(boolean encoded)
		throws PDFCosParseException, IOException, PDFIOException, PDFSecurityException
	{
		if (encoded)
			return false;
		CosObject filter = get(ASName.k_Filter);
		if (filter instanceof CosArray) {
			if (((CosArray)filter).isEmpty())
				return false;
			if (((CosArray)filter).size() > 1)
				return true;
			filter = ((CosArray)filter).get(0);
		}
		if (filter instanceof CosName && ((CosName)filter).nameValue() != ASName.k_Crypt)
			return true;
		return false;
	}

	private boolean needsEncoding()
		throws PDFCosParseException, IOException, PDFIOException, PDFSecurityException
	{
		if (mOutputFilters == null)
			return needsDecoding(false);
		if (mOutputFilters.isEmpty())
			return false;
		if (mOutputFilters.size() == 1 && ((CosName)(((CosArray)(mOutputFilters.get(0))).get(0))).nameValue() == ASName.k_Crypt)
			return false;
		return true;
	}

	/**
	 * Set new unencrypted and decoded data for this CosStream.
	 * This method takes the InputByteStream from the caller. After this
	 * method returns the InputByteStream belongs to this CosStream
	 * and should NOT be used by the caller ever again. If the caller needs to continue to
	 * use the InputByteStream then it should make a slice from
	 * the InputByteStream before calling this method.
	 * @param byteStream the new data stream to give to this object
	 * @throws PDFSecurityException
	 * @throws PDFIOException
	 * @throws PDFCosParseException
	 */
	public void newDataDecoded(InputByteStream byteStream)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		newData(byteStream, false);
	}

	/**
	 * Set new unencrypted but already encoded data for this CosStream.
	 * This method takes the InputByteStream from the caller. After this
	 * method returns the InputByteStream belongs to this CosStream
	 * and should NOT be used by the caller ever again. If the caller needs to continue to
	 * use the InputByteStream then it should make a slice from
	 * the InputByteStream before calling this method.
	 * @param byteStream the new data stream to give to this object
	 * @throws PDFSecurityException
	 * @throws PDFIOException
	 * @throws PDFCosParseException
	 */
	public void newDataEncoded(InputByteStream byteStream)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		newData(byteStream, true);
	}

	/**
	 * Set new data for this CosStream.
	 * This method takes the InputByteStream from the caller. After this
	 * method returns the InputByteStream belongs to this CosStream
	 * and should NOT be used by the caller ever again. If the caller needs to continue to
	 * use the InputByteStream then it should make a slice from
	 * the InputByteStream before calling this method.
	 * @param byteStream the new data stream to give to this object
	 * @param encoded whether the InputByteStream contains encoded data or not
	 * @throws PDFSecurityException
	 * @throws PDFIOException
	 * @throws PDFCosParseException
	 */
	private void newData(InputByteStream byteStream, boolean encoded)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		try {
			mDataInPDF = false;
			getInfo().markDirty();
			setCachedStream(encoded, byteStream);
		} catch (IOException e) {
			throw new PDFIOException(e);
		}
	}

	/**
	 * @throws PDFSecurityException
	 * @throws IOException
	 * @throws PDFCosParseException
	 */
	void setToEncrypt(boolean encrypted)
		throws PDFCosParseException, PDFIOException, IOException, PDFSecurityException
	{
		if (mToEncrypt == encrypted)
			return;
		getInfo().markDirty();
		mToEncrypt = encrypted;
	}

	/**
	 * @throws PDFSecurityException
	 * @throws IOException
	 * @throws PDFCosParseException
	 */
	void setIsEncrypted(boolean encrypted)
		throws PDFCosParseException, PDFIOException, IOException, PDFSecurityException
	{
		if (mIsEncrypted == encrypted)
			return;
		getInfo().markDirty();
		mIsEncrypted = encrypted;
	}
	
	/**
	 * This method sets the filter list which are used while writing this stream.
	 * @param filters
	 * @throws PDFCosParseException
	 * @throws IOException
	 * @throws PDFSecurityException
	 */
	public void setOutputFiltersList(CosArray filters)
		throws PDFCosParseException, PDFIOException, IOException, PDFSecurityException
	{
		mOutputFilters = filters;
		getInfo().markDirty();
	}

	/**
	 * Returns filter list that shall be used when this stream is written.
	 */
	public CosArray getOutputFiltersList()
	{
		return mOutputFilters;
	}

	/**
	 * This method returns the filter list with which this stream is encoded currently.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	
	public CosArray getInputFiltersList()
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		try {
			return getInputFilters(false);
		} catch (IOException e) {
			throw new PDFIOException(e);
		}
	}
	
	CosArray getInputFilters(boolean removeCryptFilter)
		throws PDFCosParseException, IOException, PDFIOException, PDFSecurityException
	{
		CosArray filterList = getDocument().createCosArray();
		CosArray filter = null;
		CosObject filterObj = get(ASName.k_Filter);
		CosObject paramsObj = get(ASName.k_DecodeParms);
		if (filterObj instanceof CosName && (!removeCryptFilter || ((CosName)filterObj).nameValue() != ASName.k_Crypt)) {
			filter = getDocument().createCosArray();
			filter.add(filterObj);
			filter.add(paramsObj!= null ? paramsObj : getDocument().createCosNull());
			filterList.add(filter);
		}
		else if (filterObj instanceof CosArray) {
			CosObject filterName = null, filterParams = null;
			for (int i = 0; i < ((CosArray)filterObj).size(); i++) {
				filter = getDocument().createCosArray();
				filterName = ((CosArray)filterObj).get(i);
				if (removeCryptFilter && filterName instanceof CosName && ((CosName)filterName).nameValue() == ASName.k_Crypt)
					continue;
				if (paramsObj != null)
					filterParams = ((CosArray)paramsObj).get(i);
				filter.add(filterName);
				filter.add(filterParams!= null ? filterParams : getDocument().createCosNull());
				filterList.add(filter);
			}
		}
		return filterList;
	}

	/**
	 * Compare the old filters to the new ones to see if anything actually changed
	 */
	private boolean filtersChanged(CosArray filterList, boolean ignoreCrypt)
		throws PDFCosParseException, PDFIOException, IOException, PDFSecurityException
	{
		CosObject oldFilterList = get(ASName.k_Filter);
		CosObject oldParamsList = get(ASName.k_DecodeParms);
		ASName oldFilter = null;
		ASName newFilter = null;
		int oldIndex = 0;
		int newIndex = 0;
		while (true) {
			oldFilter = getFilter(oldFilterList, oldIndex);
			if (ignoreCrypt && oldFilter == ASName.k_Crypt)
				oldFilter = getFilter(oldFilterList, ++oldIndex);
			newFilter = getFilter(filterList, newIndex);
			if (ignoreCrypt && newFilter == ASName.k_Crypt)
				newFilter = getFilter(filterList, ++newIndex);
			if (oldFilter == null && newFilter == null)
				return false;
			if (oldFilter != newFilter)
				return true;
			if (!getParams(oldParamsList, oldIndex).equals(getParams(filterList, newIndex)))
				return true;
			oldIndex++;
			newIndex++;
		}
	}

	/**
	 * Get one filter from a filter object which can be any of null,
	 * CosName, CosArray or ArrayList of CosObject[].
	 */
	private ASName getFilter(CosObject filterObject, int index)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		if (filterObject instanceof CosName) {
			if (index != 0)
				return null;
			return ((CosName)filterObject).nameValue();
		}
		if (filterObject instanceof CosArray) {
			if (index >= ((CosArray)filterObject).size())
				return null;
			CosObject objectAtIndex = ((CosArray)filterObject).get(index);
			if(objectAtIndex instanceof CosName)
				return ((CosName)(objectAtIndex)).nameValue();
			else if(objectAtIndex instanceof CosArray)
				return ((CosName)((CosArray)objectAtIndex).get(0)).nameValue();
		}
		return null;
	}

	/**
	 * Get one filter params from a params object which can be any of null,
	 * CosDictionary, CosArray or ArrayList of CosObject[].
	 */
	private CosObject getParams(Object paramsObject, int index)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		if (paramsObject instanceof CosDictionary) {
			if (index != 0)
				paramsObject = null;
		}
		else if (paramsObject instanceof CosArray) {
			if (index >= ((CosArray)paramsObject).size())
				paramsObject = null;
			else
				paramsObject = ((CosArray)paramsObject).get(index);
		}
		if (!(paramsObject instanceof CosDictionary))
			return getDocument().createCosNull();
		return (CosDictionary)paramsObject;
	}

	boolean needsEncryption()
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		return needsDecryptionOrEncryption(null, true);
	}

	boolean needsDecryption(InputByteStream stream)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		return needsDecryptionOrEncryption(stream, false);
	}

	private boolean needsDecryptionOrEncryption(InputByteStream stream, boolean encrypting)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		try {
			CosEncryption encryptionHandler = getDocument().getEncryption();
			boolean encrypted = encrypting ? mToEncrypt : mIsEncrypted;
			boolean needsIt = encrypting ? encryptionHandler.needsEncryption() : encryptionHandler.needsDecryption();
			if (!encrypted || ((stream != null) && (stream.length() == 0)))
				return false;
			CosObject cosType = get(ASName.k_Type);
			ASName type = (cosType instanceof CosName) ? ((CosName) cosType).nameValue() : null;
			if (!encryptionHandler.getEncryptionState() || !needsIt || (ASName.k_XRef.equals(type)))
				return false;
			int shouldDoIt = getDocument().getEncryption().shouldDecryptOrEncrypt(type, getCryptFilter(), encrypting);
			if (shouldDoIt != 0)
				return shouldDoIt > 0 ? true : false;
			if (isDirty() || ((this instanceof CosObjectStream)))
				return true;
			if ((getCryptFilter() != null) || (getDocument().getLinearization() == null))
				return false;
			return true;
		} catch (IOException e) {
			throw new PDFIOException(e);
		}
	}

	private InputStream buildInputFilterStream(InputStream inputStream)
		throws PDFCosParseException, IOException, PDFIOException, PDFSecurityException
	{
		CosObject rawFilter = get(ASName.k_Filter);
		Iterator filterIter;
		FilterParams params = FilterStream.buildFilterParams(get(ASName.k_DecodeParms));
		if (rawFilter instanceof CosName) {
			List filterList = new ArrayList();
			filterList.add(rawFilter);
			filterIter = filterList.iterator();
		} else if (rawFilter == null)
			filterIter = getDocument().createCosArray().iterator();
		else {
			filterIter = ((CosArray)rawFilter).iterator();
		}
		while (filterIter.hasNext()) {
			ASName curFilter = ((CosName)filterIter.next()).nameValue();
			if (curFilter != ASName.k_Crypt)
				inputStream = FilterStream.applyFilter(inputStream, curFilter, params, getCustomFilterRegistry());
		}
		return inputStream;
	}

	/*
	 * Currently this just checks the CCITTFaxDecode, if present, to make
	 * sure it has no parameters other than "Columns" and "K=-1" but it
	 * could do other work as well.
	 */
	private void checkOutputFilters()
		throws PDFCosParseException, PDFIOException, IOException, PDFSecurityException
	{
		CosArray filterList = mOutputFilters;
		if (mOutputFilters == null)
			filterList = getInputFiltersList();
		CosArray filterAtIndex;
		for(int i=0; i<filterList.size(); i++){
			filterAtIndex = (CosArray) filterList.get(i);
			if (filterAtIndex.get(0) instanceof CosName && ((CosName)filterAtIndex.get(0)).nameValue() == ASName.k_CCITTFaxDecode) {
				int cols = 1728;
				if (filterAtIndex.get(1) instanceof CosDictionary) {
					CosDictionary paramsDict = (CosDictionary)filterAtIndex.get(1);
					CosObject cosCols = paramsDict.get(ASName.k_Columns);
					if (cosCols instanceof CosNumeric) {
						cols = cosCols.intValue();
						CosObject cosMode = paramsDict.get(ASName.k_K);
						CosObject cosRows = paramsDict.get(ASName.k_Rows);
						if (cosMode instanceof CosNumeric) {
							int mode = cosMode.intValue();
							if (mode == -1 && paramsDict.size() == ((cosRows == null) ? 2 : 3)) {
								return;
							}
						}
					}
				}
				CosDictionary newParams = getDocument().createDirectCosDictionary();
				newParams.put(ASName.k_Columns, cols);
				newParams.put(ASName.k_K, -1);
				filterAtIndex.add(1, newParams != null ? newParams : getDocument().createCosNull());
				setOutputFiltersList(filterList);
				return;
			}
		}
	}

	
	@Override
	public String toString()
	{
		// Never output stream data with "toString". DO NOT set "debug" to "false", ever.
		return toString(true);
	}

	/**
	 * Return stream data offset or -1 if object has never been written.
	 */
	public long getStreamDataOffset()
	{
		long objPos = getObjPos();
		if (objPos == 0 || mPos == 0)
			return -1;
		return mPos - objPos;
	}

	/*
	 * New stream output code from here on down to the end
	 */
	@Override
	void writeOut(OutputByteStream dstByteStm, boolean inString, boolean inDebugMode)
		throws PDFCosParseException, PDFIOException, IOException, PDFSecurityException
	{
		
		writeOut(dstByteStm, inString, inDebugMode, false);
	}
	
	private void setFlateOuputFilter() throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosObject cosType = get(ASName.k_Type);
		ASName type = (cosType instanceof CosName) ? ((CosName)cosType).nameValue() : null;
		if (type != ASName.k_Metadata) {
			put(ASName.k_Filter, getDocument().createCosName(ASName.k_FlateDecode));
			remove(ASName.k_DecodeParms);
		}
	}
	
	//Determines if filter object represents FLATE filter
	private boolean isFlateFilterObject(CosObject filterObj)
			throws PDFCosParseException, PDFIOException, PDFSecurityException {
		if (filterObj instanceof CosName
				&& filterObj.nameValue() == ASName.k_FlateDecode) {
			return true;
		} else if (filterObj instanceof CosArray
				&& ((CosArray) filterObj).size() == 1
				&& ((CosArray) filterObj).get(0).nameValue() == ASName.k_FlateDecode) {
			return true;
		} else {
			return false;
		}
	}
	

	/*
	 * New stream output code from here on down to the end : Overloaded method
	 */
	void writeOut(OutputByteStream dstByteStm, boolean inString, boolean inDebugMode, boolean saveToCopy)
		throws PDFCosParseException, PDFIOException, IOException, PDFSecurityException
	{
		long initPos = mPos;
		boolean doDecoding = false;
		boolean doEncoding = false;
		boolean doDecryption = false;
		boolean hasAcroBug = false;
		boolean doEncryption = false;
		InputByteStream srcByteStm = null;
		InputStream srcStm = null;
		OutputStream dstStm = null;
		long initialLength = 0;
		long lengthPosition = 0;
		long streamLength = 0;
		boolean emptyStream = false;
		if (inDebugMode) {
			super.writeOut(dstByteStm, inString, inDebugMode);
		} else {
			try{
				if (!mDataInPDF && !mDataEncoded)
					checkOutputFilters();
				CosObject inputFilterObj = get(ASName.k_Filter);
				
				//Determining if default FlateDecode filter should be set
				if(mOutputFilters == null)
				{
					if(getDocument().forceCompress())
					{
						if(inputFilterObj == null || (!isFlateFilterObject(inputFilterObj) && !(mDataInPDF || mDataEncoded)))
						{
							//Setting Flate since:
							//-> Output filter has not been specified and compression is forced
							//-> Input filter is not FLATE and data is in decoded form
							setFlateOuputFilter();
						}
					}
					else
					{
						if(inputFilterObj != null && !isFlateFilterObject(inputFilterObj) && !(mDataInPDF || mDataEncoded))
						{
							//Setting Flate since:
							//-> Output filter has not been specified and input filter was not null and it was not FLATE and data is in decoded form
							setFlateOuputFilter();
						}
					}
				}


				mOutputFilters = getDocument().getEncryption().checkEFFOutputFilter(this);
				if (mToEncrypt)
					mOutputFilters = getDocument().getEncryption().checkMetadataStream(this);
				if (mDataInPDF) {
					initialLength = getLong(ASName.k_Length);
					try {
						srcByteStm = getDocument().getStream(mPos, initialLength);
					} catch (PDFIOException e) {
						// TODO or should we just change the stream length in the object
						// and dirty the document, for errors which occur while saving an object ? 
						initialLength = CosRepairUtils.getStreamLength(getDocument(), mPos);
						srcByteStm = getDocument().getStream(mPos, initialLength);
						setRepairedValue(ASName.k_Length, getDocument().createCosNumeric(initialLength));
					}
					doDecryption = (initialLength == 0) ? false : needsDecryption(srcByteStm);
					if (mOutputFilters != null/*if output filters are null then no point of checking for filters change.*/ && filtersChanged(mOutputFilters, true)) {
						doDecoding = true;
						doEncoding = true;
					}
				} else {
					srcByteStm = getCachedStream(mDataEncoded);
					if (srcByteStm == null)
						srcByteStm = getStreamManager().getInputByteStream(new byte[0]);
					else
						srcByteStm = srcByteStm.slice();
					initialLength = srcByteStm.length();
					doEncoding = !mDataEncoded && needsEncoding();
				}
				srcStm = srcByteStm.toInputStream();
				if (doDecryption) {
					hasAcroBug = checkForAcroBug();
					if (!hasAcroBug) {
						EncryptionHandlerState encryptHandle = getDocument().getEncryption().getStreamDecryptionStateHandler(this, srcByteStm);
						srcStm = new DecryptingInputStream(this, srcStm, encryptHandle);
					}
				}
				getCryptFilter(mOutputFilters);
				doEncryption = needsEncryption();
				if (doEncryption) {
					EncryptionHandlerState encryptHandle = getDocument().getEncryption().getStreamEncryptionStateHandler(this);
					dstStm = new EncryptingOutputStream(this, dstByteStm, encryptHandle);
				} else {
					dstStm = dstByteStm.toOutputStream();
				}
				if (initialLength == 0)
					doDecoding = doEncoding = false;
				if (doDecoding || doEncoding) {
					CosArray inFilters = (doDecoding) ? getInputFilters(true) : getDocument().createCosArray();
					if (inFilters == null)
						inFilters = getDocument().createCosArray();
					CosArray outFilters = (doEncoding) ? ((getOutputFiltersList() != null) ? getOutputFiltersList() : getInputFilters(true)) : getDocument().createCosArray();
					if (outFilters == null)
						outFilters = getDocument().createCosArray();
					CosArray inFiltersTrimmed = getDocument().createCosArray();
					CosArray outFiltersTrimmed = getDocument().createCosArray();
					trimFilterLists(inFilters, outFilters, inFiltersTrimmed, outFiltersTrimmed, hasAcroBug);
					Iterator<CosObject> filterIter = inFiltersTrimmed.iterator();
					while (filterIter.hasNext()) {
						CosArray filterItem = (CosArray) filterIter.next();
						ASName curFilter = ((CosName)filterItem.get(0)).nameValue();
						FilterParams params = FilterStream.buildFilterParams(filterItem.get(1));
						srcStm = FilterStream.applyFilter(srcStm, curFilter, params, getCustomFilterRegistry());
					}
					if (hasAcroBug) {
						EncryptionHandlerState encryptHandle = getDocument().getEncryption().getStreamDecryptionStateHandler(this, srcByteStm);
						srcStm = new DecryptingInputStream(this, srcStm, encryptHandle);
					}
					filterIter = outFiltersTrimmed.iterator();
					while (filterIter.hasNext()) {
						CosArray filterItem = (CosArray) filterIter.next();
						ASName curFilter = ((CosName)filterItem.get(0)).nameValue();
						FilterParams params = FilterStream.buildFilterParams(filterItem.get(1));
						dstStm = FilterStream.applyFilter(dstStm, curFilter, params, getCustomFilterRegistry());
					}
				}
				CosArray newFilters = mOutputFilters;
				mOutputFilters = null;
				if (newFilters == null && doDecryption && !doEncryption)
					newFilters = getInputFilters(true);
				setInputFilters(newFilters);
				if (doDecoding || doEncoding || doDecryption || doEncryption) {
					if(doEncryption && initialLength == 0){
						initialLength = 1;
						emptyStream = true;
					}
					super.put(ASName.k_Length, initialLength * 1000);
					lengthPosition = writeStreamDict(dstByteStm, inString, inDebugMode);
				} else {
					super.put(ASName.k_Length, initialLength);
					super.writeOut(dstByteStm, inString, inDebugMode);
				}
				dstByteStm.write("stream\n".getBytes());
				mPos = dstByteStm.getPosition();
				try {
					if(emptyStream){
						byte[] buf = new byte[0];
						dstStm.write(buf, 0, 0);
					}else
					{
						IO.copy(srcStm, dstStm);
					}
				} catch (EOFException e) {
				} catch (ZipException e) {
					throw new PDFCosParseException(e);
				}
				if(dstStm != null)
					dstStm.flush();
			}finally{
				try{
					try{
						if(srcStm != null) srcStm.close();
					}finally{
						try{
							if(dstStm != null) dstStm.flush();
						}finally{
							try{
								if(dstStm != null) dstStm.close();
							}finally{
								if(srcByteStm != null) srcByteStm.close();
							}
						}
					}
				}catch(IOException e){
					//exception eaten.
				}
			}
			streamLength = dstByteStm.getPosition() - mPos;
			dstByteStm.write("\nendstream".getBytes());
			if (lengthPosition != 0) {
				super.put(ASName.k_Length, streamLength);
				long savePosition = dstByteStm.getPosition();
				dstByteStm.seek(lengthPosition);
				String initialLengthString = Long.toString(initialLength * 1000);
				String finalLengthString = Long.toString(streamLength);
				int padding = initialLengthString.length() - finalLengthString.length();
				if (padding < 0)
					throw new PDFIOException("CosStream length field string overflow.");
				while (padding-- > 0)
					dstByteStm.write(' ');
				dstByteStm.write(StringOps.toByteArray(finalLengthString));
				dstByteStm.seek(savePosition);
			}
		}
		
		// If document is saved to a copy, reset mPos as underlying buffer will not be changed.
		if(saveToCopy)
			mPos = initPos;
	}

	/*
	 * Set the stream dictionary Filter and DecodeParms values from the passed array
	 */
	private void setInputFilters(CosArray filterList)
		throws PDFCosParseException, PDFIOException, PDFSecurityException, IOException
	{
		if (filterList != null && filtersChanged(filterList, false)) {
			CosObject outputFilter = null;
			CosObject outputParams = null;
			if (filterList.size() == 1) {
				CosArray filter = (CosArray)filterList.get(0);
				outputFilter = filter.get(0);
				outputParams = FilterStream.updateCustomFilterParams(((CosName)outputFilter).nameValue(), filter.get(1), getCustomFilterRegistry());
			} else if (filterList.size() != 0) {
				boolean outputParamsNull = true;
				outputFilter = getDocument().createCosArray();
				outputParams = getDocument().createCosArray();
				Iterator<CosObject> iter = filterList.iterator();
				while (iter.hasNext()) {
					CosArray listItem = (CosArray) iter.next();
					CosObject filter = listItem.get(0);
					CosObject params = listItem.get(1);
					((CosArray)outputFilter).add(filter);
					params = FilterStream.updateCustomFilterParams(((CosName)filter).nameValue(), params, getCustomFilterRegistry());
					if (params == null)
						((CosArray)outputParams).add(getDocument().createCosNull());
					else {
						((CosArray)outputParams).add(params);
						outputParamsNull = false;
					}
				}
				if (outputParamsNull)
					outputParams = null;
			}
			super.remove(ASName.k_Filter);
			if (outputFilter != null)
				super.put(ASName.k_Filter, outputFilter);
			super.remove(ASName.k_DecodeParms);
			if (outputParams != null && !(outputParams instanceof CosNull))
				super.put(ASName.k_DecodeParms, outputParams);
		}
	}

	/*
	 * Remove complementary and/or unneeded filters from the input and output filter lists
	 */
	private void trimFilterLists(CosArray inFilters, CosArray outFilters, CosArray inFiltersTrimmed, CosArray outFiltersTrimmed, boolean hasAcroBug)
		throws PDFCosParseException, IOException, PDFSecurityException, PDFIOException
	{
		Iterator<CosObject> filterIter = inFilters.iterator();
		while (filterIter.hasNext()) {
			CosArray filterItem = (CosArray) filterIter.next();
			if (((CosName)filterItem.get(0)).nameValue() == ASName.k_Crypt)
				continue;
			inFiltersTrimmed.add(filterItem);
		}
		filterIter = outFilters.iterator();
		while (filterIter.hasNext()) {
			CosArray filterItem = (CosArray) filterIter.next();
			if (((CosName)filterItem.get(0)).nameValue() == ASName.k_Crypt)
				continue;
			outFiltersTrimmed.add(filterItem);
		}
		if (hasAcroBug)
			return;
		while (!(inFiltersTrimmed.isEmpty() || outFiltersTrimmed.isEmpty())) {
			CosArray inFilterItemLast = (CosArray)inFiltersTrimmed.get(inFiltersTrimmed.size() - 1);
			CosArray outFilterItemLast = (CosArray)outFiltersTrimmed.get(outFiltersTrimmed.size() - 1);
			if (((CosName)inFilterItemLast.get(0)).nameValue() != ((CosName)outFilterItemLast.get(0)).nameValue())
				break;
			if (!inFilterItemLast.get(1).equals(outFilterItemLast.get(1)))
				break;
			inFiltersTrimmed.remove(inFiltersTrimmed.size() - 1);
			outFiltersTrimmed.remove(outFiltersTrimmed.size() - 1);
		}
	}

	/*
	 * Write out the streams dictionary and remember where the length went
	 */
	private long writeStreamDict(OutputByteStream outStream, boolean inString, boolean inDebugMode)
		throws PDFCosParseException, IOException, PDFSecurityException, PDFIOException
	{
		long lengthPosition = 0;
		outStream.write('<');
		outStream.write('<');
		Iterator iter = mData.entrySet().iterator();
		while (iter.hasNext()) {
			Map.Entry entry = (Map.Entry)iter.next();
			ASName key = (ASName)entry.getKey();
			key.write(outStream);
			CosObject value = (CosObject) entry.getValue();
			if (value.isIndirect() || value instanceof CosBoolean || value instanceof CosNumeric || value instanceof CosNull)
				outStream.write(' ');
			if (key == ASName.k_Length)
				lengthPosition = outStream.getPosition();
			value.writeOut(outStream, inString, inDebugMode);
		}
		outStream.write('>');
		outStream.write('>');
		return lengthPosition;
	}

	/**
	 * Return the filter registry built from filters supplied during doc open.
	 * @return filter registry
	 */
	private CustomFilterRegistry getCustomFilterRegistry()
	{
		return getDocument().getOptions().getCustomFilterRegistry();
	}
	
	/**
	 * This method follows three criteria to check:
	 *  1) both should be instance of CosStream.
	 *  2) SHA-1 compressed byte Stream of one should be equal to other one.
	 *  3) their dictionary should be equal.
	 *  Returns false if they are not on same document.
	 *  This maintains a continuously growing list of indirect CosObject pairs, which have already been compared, to get rid of infinite recursion.
	 *  This method can take some time if called on large Cos Objects.
	 *  @param value
	 *  @return boolean
	 */
	@Override
	public boolean equals(CosObject value){
		HashMap<Integer, HashSet<Integer>> alreadyComparedCosObjectPairsList = new HashMap<Integer, HashSet<Integer>>();
		return this.safeEquals(value, alreadyComparedCosObjectPairsList);		
	}
	
	/**
	 *	This method will be used for internal communication only after CosObject.equals() is called.
	 */
	@Override
	boolean safeEquals(CosObject value, HashMap<Integer, HashSet<Integer>> alreadyComparedCosObjectPairsList){
		if(!(value instanceof CosStream) || value.getDocument() != this.getDocument())
			return false;
		if(value == this)
			return true;
		
		///// we will return from here only if these CosObjects have already been compared.
		///// This check will again be executed in it's super class (CosDictionary) if this returns false.
		//// But if we don't check it here then super class will return false but here hash comparison will be done which
		/// will be useless in this case.
		if(cosObjectPairAlreadyInList(new Integer[]{this.getObjNum(), value.getObjNum()}
																, alreadyComparedCosObjectPairsList) == 0)
			return true;
		
		CosStream valueStream = (CosStream) value;
		///comparing underlying dictionaries.
		if(!((super.safeEquals(valueStream, alreadyComparedCosObjectPairsList))))
			return false;
		try{
			///comparing SHA-1 hash of both streams.
			return Arrays.equals(CosUtils.digestStream(valueStream.getStreamDecoded(), "SHA-1"),
									CosUtils.digestStream(this.getStreamDecoded(), "SHA-1"));
		}catch(Exception e){
			throw new RuntimeException("problem occured while equating " + valueStream.getObjNum() + " with " + this.getObjNum(), e);
		}
	}

	/**
	 * Returns true if this stream is encoded else false.
	 * @return boolean
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 * @throws IOException
	 */
	public boolean isEncoded() throws PDFCosParseException, PDFIOException, PDFSecurityException, IOException{
		if(mDataInPDF)
			return needsDecoding(false);
		else
			return mDataEncoded;
	}
}
















