/*
 * File: CredentialFactory.java
 *  
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/

package com.adobe.internal.pdftoolkit.core.credentials;

import java.io.IOException;
import java.security.Key;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.UnrecoverableKeyException;
import java.security.cert.Certificate;
import java.security.cert.X509Certificate;
import java.security.spec.InvalidKeySpecException;

import com.adobe.internal.pdftoolkit.core.credentials.impl.ByteArrayKeyHolder;
import com.adobe.internal.pdftoolkit.core.credentials.impl.JCECredentials;
import com.adobe.internal.pdftoolkit.core.credentials.impl.JCEKeyHolder;
import com.adobe.internal.pdftoolkit.core.credentials.impl.utils.CertUtils;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFInvalidParameterException;


/**
 * The Credential Factory allows us to create credentials that can be used for encryption/digsig related functions.
 *  
 * @author mdharan
 *
 */
public class CredentialFactory 
{

	
	private CredentialFactory () {}
	
	
	
	public static CredentialFactory newInstance()
	{
		return new CredentialFactory();
	}
	
	
	/**
	 *Create a new instance of a <code>Credentials</code> object from a PrivateKeyHolder, an explicit certificate
	 * and certificate chain.
	 * 
	 * @param privateKeyHolder  - A container for a private key. It can be be generated using the <code>PrivateKeyHolderFactory</code>.
	 * @param cert - An X509 certificate
	 * @param certChain - An array of certificates in the certificate chain.
	 * 
	 * @return <code>Credentials</code>
	 
	 * @throws PDFInvalidParameterException
	 */
	public Credentials createCredentials(PrivateKeyHolder privateKeyHolder,X509Certificate cert, Certificate[] certChain) throws PDFInvalidParameterException
	{
		
		try 
		{
			privateKeyHolder = computeJCEPrivateKey(privateKeyHolder);
			return new JCECredentials(privateKeyHolder, cert, certChain);
		} 
		catch (NoSuchAlgorithmException e) 
		{
			throw new PDFInvalidParameterException("Could not create the private key using the data in the private key holder",e);
		} 
		catch (InvalidKeySpecException e) 
		{
			throw new PDFInvalidParameterException("Invalid key specification in the private key holder",e);
		} 
		
	    
		
	}

	
	/**
	 * Create a new instance of a <code>Credentials</code> object from a java private key, cert and certificate chain.
	 * @param privateKey - JCE private key
	 * @param cert - an X509 Certificate
	 * @param certChain - an array of certificates
	 * @return <code>Credentials</code>
	 * @throws PDFInvalidParameterException
	 */
	public Credentials createCredentials(PrivateKey privateKey, X509Certificate cert, Certificate[] certChain) throws PDFInvalidParameterException
	{
		return  new JCECredentials(privateKey, cert, certChain);								
	}
	
	
	/**
	 * Create a new instance of a <code>Credentials</code> object from der encoded bytes representing a private key, certificate, certificate chain.
	 * You cannot use this method if your signature algorithm is DSA.
	 * If signature algorithm is DSA use - 
	 * public Credentials createCredentials(PrivateKeyHolder privateKeyHolder, byte[] derEncodedCert, byte[][] derEncodedCertChain)
	 * 
	 * @param derEncodedPrivateKey - byte array representing a DER encoded PKCS8 unencrypted private key
	 * @param derEncodedCert - byte array representing a DER encoded X509 certificate
	 * @param derEncodedCertChain - array of byte arrays representing a DER encoded certificate chain.
	 * @return <code>Credentials</code>
	 * @throws PDFInvalidParameterException
	 */
	public Credentials createCredentials(byte[] derEncodedPrivateKey, byte[] derEncodedCert, byte[][] derEncodedCertChain) throws PDFInvalidParameterException
	{
		try 
		{
			X509Certificate jceCert = (X509Certificate) CertUtils.importCertificate(derEncodedCert);
			Certificate[] jceCertChain = CertUtils.importCertificateChain(derEncodedCertChain);	
			PrivateKeyHolder privateKeyHolder = CertUtils.createJCEPrivateKeyHolder(derEncodedPrivateKey ,"RSA");
			return new JCECredentials(privateKeyHolder, jceCert, jceCertChain);
		} 
		catch (java.security.cert.CertificateException e) 
		{					
			throw new PDFInvalidParameterException("Certificate Exception when importing JCE keys and certs - ",e);
		} 
		catch (IOException e) 
		{					
			throw new PDFInvalidParameterException("IO Exception when importing JCE keys and certs - ",e);
		} 
		catch (NoSuchAlgorithmException e) 
		{					
			throw new PDFInvalidParameterException("Exception when importing JCE keys and certs - ",e);
		} 
		catch (InvalidKeySpecException e) 
		{					
			throw new PDFInvalidParameterException("Invalid key Exception when importing JCE keys and certs - ",e);
		}				
	}
	
	
	/**
	 * Create a new instance of a <code>Credentials</code> object from a private key holder and bytes representing a DER encoded certificate
	 * and certificate chain.
	 * 
	 * @param privateKeyHolder - A container for a private key. It can be be generated using the <code>PrivateKeyHolderFactory</code>.
	 * @param derEncodedCert - byte array representing a DER encoded X509 certificate
	 * @param derEncodedCertChain - array of byte arrays representing a DER encoded certificate chain.
	 * @return <code>Credentials</code>
	 * @throws PDFInvalidParameterException
	 */
	public Credentials createCredentials(PrivateKeyHolder privateKeyHolder, byte[] derEncodedCert, byte[][] derEncodedCertChain) throws PDFInvalidParameterException
	{
		//THIS IS Untested. Only for DSS.			
		if (privateKeyHolder instanceof JCEKeyHolder)
			return new JCECredentials((privateKeyHolder),derEncodedCert, derEncodedCertChain);			
		//End of Untested Code.		
		try 
			{
      privateKeyHolder = computeJCEPrivateKey(privateKeyHolder);  // Turns non-JCE private keys into JCE private key
				X509Certificate jceCert = (X509Certificate) CertUtils.importCertificate(derEncodedCert);
				Certificate[] jceCertChain = CertUtils.importCertificateChain(derEncodedCertChain);	
				
      JCECredentials credentials =  new JCECredentials(privateKeyHolder, jceCert, jceCertChain);
      return credentials;

    } 
    catch (java.security.cert.CertificateException e) 
    {         
      throw new PDFInvalidParameterException("Certificate Exception when importing JCE keys and certs - ",e);
    } 
    catch (IOException e) 
    {
      throw new PDFInvalidParameterException("IO Exception when importing JCE keys and certs - ",e);
    } 
    catch (NoSuchAlgorithmException e) 
    {
      throw new PDFInvalidParameterException("Exception when importing JCE keys and certs - ",e);
    } 
    catch (InvalidKeySpecException e) 
    {
      throw new PDFInvalidParameterException("Invalid key Exception when importing JCE keys and certs - ",e);
    } 

	}
	
	/**
	 * Create a new instance of a <code>Credentials</code> object for Encryption 
	 * from the bytes representing a DER encoded certificate
	 * and certificate chain.
	 * This <code>Credentials</code> object may be used for encryption only. It may not be used for
	 * decryption or Digital Signatures processing.
	 * <p>This method may not be used on JDK 1.4.2 with public key length greater than 2048.</p>
	 * 
	 * @param derEncodedCert - byte array representing a DER encoded X509 certificate
	 * @param derEncodedCertChain - array of byte arrays representing a DER encoded certificate chain.
	 * @return <code>Credentials</code>
	 * @throws PDFInvalidParameterException
	 */
	public Credentials createEncryptionCredentials(byte[] derEncodedCert, byte[][] derEncodedCertChain) 
	throws PDFInvalidParameterException
	{
		try 
		{
			X509Certificate jceCert = (X509Certificate) CertUtils.importCertificate(derEncodedCert);
			Certificate[] jceCertChain = CertUtils.importCertificateChain(derEncodedCertChain);	
			PrivateKey privateKey = null;
			return new JCECredentials(privateKey, jceCert, jceCertChain);

		} 
		catch (java.security.cert.CertificateException e) 
		{					
			throw new PDFInvalidParameterException("Certificate Exception when importing JCE keys and certs - ",e);
		} 
		catch (IOException e) 
		{
			throw new PDFInvalidParameterException("IO Exception when importing JCE keys and certs - ",e);
		} 
	}

	
	/**
	 * 
	 * This method should not be used except for internal testing. It will not work for HSM signing
	 * or when using the IBMJCE provider.
	 * public Credentials createCredentials(PrivateKeyHolder privateKeyHolder, byte[] derEncodedCert, byte[][] derEncodedCertChain) 
	 * should be used instead.
	 *
	 * Create a new instance of a <code>Credentials</code> object from a keystore entry.
	 * 
	 * This method is synchronized because the underlying KeyStore may not be thread safe.
	 * 
	 * @param keyStore the keystore containing the certificate and certificate chain.
	 * @param alias the alias (or name) of the entry within the keystore containing the 
	 * certificate and certificate chain.
	 * @param password the password for recovering the key.
	 * 
	 * @return <code>Credentials</code>
	 * @throws PDFInvalidParameterException if any of the input parameters specify an invalid or 
	 * inaccessible entity.
	 */
	public  synchronized Credentials createCredentials(KeyStore keyStore, String alias, char[] password) 
    throws PDFInvalidParameterException
	{
		
		PrivateKey privateKey;
		Key key;
		try
		{
			key = keyStore.getKey(alias, password);			
		}
		catch (NoSuchAlgorithmException e)
		{
			throw new PDFInvalidParameterException("could not retrieve the key from the key store", e);
		}
		catch (UnrecoverableKeyException e)
		{
			throw new PDFInvalidParameterException("could not retrieve the key from the key store", e);
		}
		catch (KeyStoreException e)
		{
			throw new PDFInvalidParameterException("could not retrieve the key from the key store", e);
		}

		if (key instanceof PrivateKey)
		{
			privateKey = (PrivateKey)key;
		}
		else
			throw new PDFInvalidParameterException("key store did not contain a private key");
		
		
		Certificate cert;
		try
		{
			cert = keyStore.getCertificate(alias);
		}
		catch (KeyStoreException e)
		{
			throw new PDFInvalidParameterException("could not retrieve the certificate from the key store", e);
		}
		if (!(cert instanceof X509Certificate))
			throw new PDFInvalidParameterException("key store did not contain an X509 certificate.");
			
		// Get the certificate chain.
		Certificate[] certChain;
		try
		{
			certChain = keyStore.getCertificateChain(alias);
		}
		catch (KeyStoreException e)
		{
			throw new PDFInvalidParameterException("could not retrieve the certificate chain from the key store", e);
		}
    return new JCECredentials(privateKey, (X509Certificate)cert, certChain);
		
	}
	
	  public static PrivateKeyHolder computeJCEPrivateKey(PrivateKeyHolder privateKeyHolder) throws NoSuchAlgorithmException, InvalidKeySpecException, PDFInvalidParameterException
		{
			
			if (privateKeyHolder instanceof JCEKeyHolder)
			{
	      return privateKeyHolder;
			}
			else if (privateKeyHolder instanceof ByteArrayKeyHolder)
			{
	      return CertUtils.createJCEPrivateKeyHolder(((ByteArrayKeyHolder)privateKeyHolder).getDerEncodedKey(), ((ByteArrayKeyHolder)privateKeyHolder).getAlgorithm());
			}
			else
			{
				throw new PDFInvalidParameterException("The PrivateKeyHolder did not contain any valid private keys");
			}
		}
	  public static String getJCEProviderName(PrivateKeyHolder privateKeyHolder)
	  {
	    if (privateKeyHolder instanceof JCEKeyHolder)
	      return ((JCEKeyHolder)privateKeyHolder).getProviderName();
	    else
	      return null;
	  }

	 

}
