/*
 * File: ASCIIHexOutputStream.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2003-2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/

package com.adobe.internal.pdftoolkit.core.filter;

import java.io.FilterOutputStream;
import java.io.IOException;
import java.io.OutputStream;

/**
 * ASCIIHexOutputStream
 *  Writes data coded as Hex (2 hex bytes per byte of data)
 *
 *	Important: This filter assumes the output stream is an ISO/Latin-1
 *	stream, of 1-byte (not Unicode) characters!
 *
 * Copyright (C) 1996-2005 Adobe Systems Incorporated
 */
public class ASCIIHexOutputStream extends FilterOutputStream
{
	// Variables
	private int NEWLINE_EVERY = 64;	/* maximum line length */
	private int lineLength;
	private byte toHex[];
	private long totalOut;

	// Constructors
	/**
	 * Creates an output stream filter.
	 * @param out	the output stream
	 */
	public ASCIIHexOutputStream(OutputStream  out, FilterParams p)
	{
		super(out);
		initOutput();
		if (p != null) {
			if (p.containsKey(FilterParams.NewlineEvery_K))
				NEWLINE_EVERY = ((Integer)p.get(FilterParams.NewlineEvery_K)).intValue();
			}
	}

	public ASCIIHexOutputStream(OutputStream  out)
	{
		this(out, null);
	}

	// Methods
	private void initOutput()
	{
		int i;
		lineLength = NEWLINE_EVERY;
		totalOut = 0;

		toHex = new byte[16];
		for (i = 0; i <= 9; i++)
			toHex[i] = (byte)(i + '0');
		for (i = 10; i <= 15; i++)
			toHex[i] = (byte)(i - 10 + 'A');
	}

	private void putEOL()
		throws IOException
	{
		out.write((byte)'\r');	/* The most general form of end of line */
		out.write((byte)'\n');
		lineLength = 0;
		totalOut += 2;
	}

	/**
	 * Writes a byte. Will block until the byte is actually
	 * written.
	 * @param b the byte
	 * @exception IOException If an I/O error has occurred.
	 */
	@Override
	public void write(int b)
		throws IOException
	{
		if (lineLength > NEWLINE_EVERY - 2)
			putEOL();
		out.write(toHex[(b >>> 4) & 0x0f]);
		out.write(toHex[b & 0x0f]);
		lineLength += 2;
		totalOut += 2;
	}

	/**
	 * Writes a subarray of bytes.
	 * @param b	the data to be written
	 * @param off	the start offset in the data
	 * @param len	the number of bytes that are written
	 * @exception IOException If an I/O error has occurred.
	 */
	@Override
	public void write(byte  b[], int  off, int  len)
		throws IOException
	{
		int maxWrite = b.length-off;
		if (maxWrite > len)
			maxWrite = len;
		while (maxWrite-- > 0) {
			write(b[off++] & 0x0ff);
		}
	}

	/**
	 * Writes an array of bytes. Will block until the bytes
	 * are actually written.
	 * @param b	the data to be written
	 * @exception IOException If an I/O error has occurred.
	 */
	@Override
	public void write(byte  b[])
		throws IOException
	{
		write(b, 0, b.length);
	}

	/**
	 * Closes the stream and writes the stream trailer ">".
	 * This method must be called to release any resources
	 * associated with the stream.
	 * @exception IOException If an I/O error has occurred.
	 */
	@Override
	public void close()
		throws IOException
	{
		out.write((byte)'>');		/* End of stream marker (may go 1 past NEWLINE_EVERY) */
		++totalOut;
		super.close();
	}

	/**
	 * Counts the number of bytes written by this filter.
	 * @return	actual number of bytes written
	 */
	public long getTotalOut()
	{
		return totalOut;
	}
}
