/*
 * File: FilterParams.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2003-2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/

package com.adobe.internal.pdftoolkit.core.filter;

import java.util.Enumeration;
import java.util.HashMap;
import java.util.Hashtable;

import com.adobe.internal.pdftoolkit.core.cos.CosBoolean;
import com.adobe.internal.pdftoolkit.core.cos.CosDictionary;
import com.adobe.internal.pdftoolkit.core.cos.CosDocument;
import com.adobe.internal.pdftoolkit.core.cos.CosNumeric;
import com.adobe.internal.pdftoolkit.core.cos.CosObject;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFCosParseException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFIOException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFSecurityException;
import com.adobe.internal.pdftoolkit.core.types.ASBoolean;
import com.adobe.internal.pdftoolkit.core.types.ASDictionary;
import com.adobe.internal.pdftoolkit.core.types.ASName;
import com.adobe.internal.pdftoolkit.core.types.ASNumber;

/**
 * A class to hold parameters for Stream filters
 * This is just a HashTable, but all the keywords are defined here
 * Modification log:
 *  6/17/98 Marc Kaufman - new Class, Hashtable for parameters
 */
public class FilterParams extends Hashtable<String, Object> {
	static final long serialVersionUID = 0;

	//Images
	public static final String Height_K = "Height";                     //Integer
	public static final String Width_K = "Width";                       //Integer

	//All input streams
	public static final String CloseSource_K = "CloseSource";           //Boolean
	public static final String AvoidInOverrun_K = "AvoidInOverrun";     //Boolean


	//Filter specific
	public static final String Predictor_K = "Predictor";               //Integer  TIFF
	public static final String Columns_K = "Columns";                         //Integer  TIFF CCITT
	public static final String Components_K = "Components";             //Integer  TIFF
	public static final String BitsPerComponent_K = "BitsPerComponent"; //Integer  TIFF

	public static final String EarlyChange_K = "EarlyChange";           //Integer  LZW

	public static final String Rows_K = "Rows";                         //Integer  CCITT
	public static final String K_K = "K";                               //Integer  CCITT
	public static final String DamagedRowsBeforeError_K = "DamagedRowsBeforeError"; //Integer  CCITT
	public static final String ByteAlign_K = "ByteAlign";               //Boolean  CCITT
	public static final String DoRTC_K = "DoRTC";                       //Boolean  CCITT
	public static final String DoEOL_K = "DoEOL";                       //Boolean  CCITT
	public static final String BlackIs1_K = "BlackIs1";                 //Boolean  CCITT
	public static final String UncompressedMode_K = "UncompressedMode"; //Boolean  CCITT
	public static final String ColorSpace_K = "ColorSpace"; // PDFColorspace cos object.

	public static final String NewlineEvery_K = "NewlineEvery" ;        //Integer  ASCIIHex ASCII85
	
	public static final String JBIG2Globals_K = "JBIG2Globals" ;        //Stream  JBIG2Decode
	/**
	 * Sets the compression level to use:
	 *  	FlateEncode.Z_NO_COMPRESSION =		 0;
	 *  	FlateEncode.Z_BEST_SPEED =			 1;
	 *  	FlateEncode.Z_BEST_COMPRESSION =	 9;
	 *  	FlateEncode.Z_DEFAULT_COMPRESSION =	-1;
	 */
	public static final String CompressionLevel_K = "CompressionLevel";       //Integer  FLATE

	/**
	 * Sets the compression strategy to use:
	 *  	FlateEncode.Z_FILTERED =			1;
	 *  	FlateEncode.Z_HUFFMAN_ONLY =		2;
	 *  	FlateEncode.Z_DEFAULT_STRATEGY =	0;
	 */
	public static final String CompressionStrategy_K = "CompressionStrategy"; //Integer  FLATE

	/**
	 * Sets the amount of memory to use:
	 *  	FlateEncode.MAX_MEM_LEVEL = 9;
	 *  	FlateEncode.DEF_MEM_LEVEL = 8;
	 */
	public static final String MemoryLevel_K = "MemoryLevel";                 //Integer  FLATE

	/**
	 * Sets the size of the comparison window (2^bits)
	 */
	public static final String WindowBits_K = "WindowBits";                   //Integer  FLATE

	public FilterParams()
	{
		super(20);  /* Make the default size suitably small */
	}
	
	/**
	 * 	 
	 * @return ASDIctionary representation of the FilterParams
	 */
	public ASDictionary toASDictionary() {
		
		ASDictionary filtDict = new ASDictionary();
		
		for (Enumeration e = this.keys() ; e.hasMoreElements() ;) {
			 String key = (String) e.nextElement();
			 ASName asKey = ASName.create(key);
			 Object value = this.get(key);
			 if (value instanceof Integer) {
				 ASNumber num = new ASNumber(((Integer) value).intValue());
				 filtDict.put(asKey, num);				 
			 } else if (value instanceof Boolean) {
				 ASBoolean bool = new ASBoolean(((Boolean)value).booleanValue());
				 filtDict.put(asKey, bool);
			 } else if (value instanceof Double) {
				 ASNumber num = new ASNumber(((Double) value).doubleValue());
				 filtDict.put(asKey, num);				 
			 }
		}

		return filtDict;
		
	}
	
	/***
	 * COnverts 
	 * @param doc
	 * @return CosDIctionary representation of the FilterParams
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public CosObject toCosObject(CosDocument doc) 
	throws PDFCosParseException, PDFIOException, PDFSecurityException {
		
		HashMap map = new HashMap();
		
		for (Enumeration e = this.keys() ; e.hasMoreElements() ;) {
			 String key = (String) e.nextElement();
			 ASName asKey = ASName.create(key);
			 Object value = this.get(key);
			 if (value instanceof Number) {
				 Number num = (Number) value;
				 CosNumeric cosNum = doc.createCosNumeric(num);
				 map.put(asKey, cosNum);				 
			 } else if (value instanceof Boolean) {
				 Boolean bool = (Boolean) value;
				 CosBoolean cosBool = doc.createCosBoolean(bool);
				 map.put(asKey, cosBool);
			 } else if (value instanceof CosObject)
				 map.put(asKey, value);
		}

		CosDictionary cosDict = doc.createCosDictionary(map, CosObject.DIRECT);
		
		return cosDict;
		
	}
	
}
