/*
 * File: FilterStream.java
 *  
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2003-2006 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/

package com.adobe.internal.pdftoolkit.core.filter;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.zip.Deflater;
import java.util.zip.DeflaterOutputStream;
import java.util.zip.Inflater;
import java.util.zip.InflaterInputStream;

import com.adobe.internal.io.stream.InputByteStream;
import com.adobe.internal.pdftoolkit.core.cos.CosArray;
import com.adobe.internal.pdftoolkit.core.cos.CosBoolean;
import com.adobe.internal.pdftoolkit.core.cos.CosDictionary;
import com.adobe.internal.pdftoolkit.core.cos.CosNumeric;
import com.adobe.internal.pdftoolkit.core.cos.CosObject;
import com.adobe.internal.pdftoolkit.core.cos.CosStream;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFCosParseException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFIOException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFInvalidParameterException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFSecurityException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFUnableToCompleteOperationException;
import com.adobe.internal.pdftoolkit.core.filter.spi.CustomFilterException;
import com.adobe.internal.pdftoolkit.core.types.ASArray;
import com.adobe.internal.pdftoolkit.core.types.ASBoolean;
import com.adobe.internal.pdftoolkit.core.types.ASDictionary;
import com.adobe.internal.pdftoolkit.core.types.ASName;
import com.adobe.internal.pdftoolkit.core.types.ASNumber;
import com.adobe.internal.pdftoolkit.core.types.ASObject;

/**
 * @author sweet
 *
 * Some utility static methods to help with filtered streams.
 */
public class FilterStream {
	
	/**
	 * This implementation of {@link OutputStream} is used only in case of FlateDecode filter.
	 * This was implemented to override the close() method of {@link DeflaterOutputStream} which doesn't close
	 * underlying {@link Deflater} if it's non-default.
	 * @author hkansal
	 *
	 */
	static class FilterOutputStream extends DeflaterOutputStream{

		FilterOutputStream(OutputStream out, Deflater def) {
			super(out, def);
		}
		
		@Override
		public void close() throws IOException {
			try{
				super.close();
			}finally{
				if(def != null)
					def.end();
			}
	    }
	}
	
	/**
	 * Build a suitable FilterParams class from the contents of the 
	 * DecodeParms field of CosStream.  The collection of all parameters 
	 * from the dictionary or array of dictionaries is merged into the
	 * FilterParams object.
	 * @param parmsObj  
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException 
	 */
	public static FilterParams buildFilterParams(CosObject parmsObj) 
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		FilterParams result = null;
		if (parmsObj instanceof CosDictionary) {
			result = addToParms(new FilterParams(), (CosDictionary)parmsObj);
		} else if (parmsObj instanceof CosArray) {
			result = new FilterParams();
			for (int i = 0; i < ((CosArray)parmsObj).size(); i++) {
				CosObject arrayItem = ((CosArray)parmsObj).get(i);
				if (arrayItem instanceof CosDictionary) {
					result = addToParms(result, (CosDictionary)arrayItem);
				}
			}
		}
		return result;
	}

	/**
	 * Build a suitable FilterParams class from the contents of the 
	 * DecodeParms field of an inline image's image dict. The collection
	 * of all parameters from the dictionary or array of dictionaries
	 * is merged into the FilterParams object.
	 * @param parmsObj  
	 * @throws PDFInvalidParameterException 
	 */
	public static FilterParams buildFilterParams(ASObject parmsObj) throws PDFInvalidParameterException 
	{
		FilterParams result = null;
		if (parmsObj instanceof ASDictionary) {
			result = addToParms(new FilterParams(), (ASDictionary)parmsObj);
		} else if (parmsObj instanceof ASArray) {
			result = new FilterParams();
			for (int i = 0; i < ((ASArray)parmsObj).size(); i++) {
				ASObject arrayItem = ((ASArray)parmsObj).get(i);
				if (arrayItem instanceof ASDictionary) {
					result = addToParms(result, (ASDictionary)arrayItem);
				}
			}
		}
		return result;
	}

	/**
	 * Take the information in a DecodeParms dictionary and populate the proper
	 * fields of a FilterParams object.  Since the DecodeParms entry can also be
	 * an array of dictionaries, we pass in the result to have additional fields
	 * populated.
	 * @param result
	 * @param parmsDict
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException 
	 */
	private static FilterParams addToParms(FilterParams result, CosDictionary parmsDict) 
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosObject predictor = parmsDict.get(ASName.k_Predictor);
		if (predictor != null)
			result.put(FilterParams.Predictor_K, ((CosNumeric)predictor).getValue());
		//			result.put(FilterParams.Predictor_K, new Integer(predictor.intValue()));

		CosObject columns = parmsDict.get(ASName.k_Columns);
		if (columns != null)
			result.put(FilterParams.Columns_K, ((CosNumeric)columns).getValue());
		//			result.put(FilterParams.Cols_K, new Integer(columns.intValue()));

		CosObject colors = parmsDict.get(ASName.k_Colors);
		if (colors != null)
			result.put(FilterParams.Components_K, ((CosNumeric)colors).getValue());
		//			result.put(FilterParams.Components_K, new Integer(colors.intValue()));

		CosObject earlyChange = parmsDict.get(ASName.k_EarlyChange);
		if (earlyChange != null)
			result.put(FilterParams.EarlyChange_K, ((CosNumeric)earlyChange).getValue());
		//			result.put(FilterParams.EarlyChange_K, new Integer(earlyChange.intValue()));

		CosObject k = parmsDict.get(ASName.k_K);
		if (k != null)
			result.put(FilterParams.K_K, ((CosNumeric)k).getValue());
		//			result.put(FilterParams.K_K, new Integer(k.intValue()));

		CosObject eol = parmsDict.get(ASName.k_EndOfLine);
		if (eol != null)
			result.put(FilterParams.DoEOL_K, ((CosBoolean)eol).getValue());
		//			result.put(FilterParams.DoEOL_K, new Boolean(eol.booleanValue()));

		CosObject eba = parmsDict.get(ASName.k_EncodedByteAlign);
		if (eba != null)
			result.put(FilterParams.ByteAlign_K, ((CosBoolean)eba).getValue());
		//			result.put(FilterParams.ByteAlign_K, new Boolean(eba.booleanValue()));

		CosObject rows = parmsDict.get(ASName.k_Rows);
		if (rows != null)
			result.put(FilterParams.Rows_K, ((CosNumeric)rows).getValue());
		//			result.put(FilterParams.Rows_K, new Integer(rows.intValue()));

		CosObject eob = parmsDict.get(ASName.k_EndOfBlock);
		if (eob != null)
			result.put(FilterParams.DoRTC_K, ((CosBoolean)eob).getValue());
		//			result.put(FilterParams.DoRTC_K, new Boolean(eob.booleanValue()));

		CosObject blackIs1 = parmsDict.get(ASName.k_BlackIs1);
		if (blackIs1 != null)
			result.put(FilterParams.BlackIs1_K, ((CosBoolean)blackIs1).getValue());
		//			result.put(FilterParams.BlackIs1_K, new Boolean(blackIs1.booleanValue()));

		CosObject drbe = parmsDict.get(ASName.k_DamagedRowsBeforeError);
		if (drbe != null)
			result.put(FilterParams.DamagedRowsBeforeError_K, ((CosNumeric)drbe).getValue());
		//			result.put(FilterParams.DamagedRowsBeforeError_K, new Integer(drbe.intValue()));

		CosObject bitsPerComponent = parmsDict.get(ASName.k_BitsPerComponent);
		if (bitsPerComponent != null)
			result.put(FilterParams.BitsPerComponent_K, ((CosNumeric)bitsPerComponent).getValue());
		
		CosObject colorspace = parmsDict.get(ASName.k_ColorSpace);
		if (colorspace != null)
			result.put(FilterParams.ColorSpace_K, colorspace);
		
		CosObject jbig2Globals = parmsDict.get(ASName.k_JBIG2Globals);
		if (jbig2Globals instanceof CosStream){
			InputByteStream ibs = null;
			try{
				try{
					ibs = ((CosStream)jbig2Globals).getStreamDecoded();
					byte[] data = new byte[(int) ibs.length()];
					ibs.read(data);
					result.put(FilterParams.JBIG2Globals_K, data);
				}finally{
					if(ibs != null) ibs.close();
				}
			}catch(IOException e){
				throw new PDFIOException(e);
			}
			
			
		}
		return result;
	}

	/**
	 * Take the information in a DecodeParms dictionary and populate the proper
	 * fields of a FilterParams object.  Since the DecodeParms entry can also be
	 * an array of dictionaries, we pass in the result to have additional fields
	 * populated.
	 * @param result
	 * @param parmsDict
	 * @throws PDFInvalidParameterException 
	 */
	private static FilterParams addToParms(FilterParams result, ASDictionary parmsDict) throws PDFInvalidParameterException 
	{
		try
		{
		ASNumber predictor = parmsDict.getNumber(ASName.k_Predictor);
		if (predictor != null)
			result.put(FilterParams.Predictor_K, Integer.valueOf(predictor.intValue()));

		ASNumber columns = parmsDict.getNumber(ASName.k_Columns);
		if (columns != null)
			result.put(FilterParams.Columns_K, Integer.valueOf(columns.intValue()));

		ASNumber colors = parmsDict.getNumber(ASName.k_Colors);
		if (colors != null)
			result.put(FilterParams.Components_K, Integer.valueOf(colors.intValue()));

		ASNumber earlyChange = parmsDict.getNumber(ASName.k_EarlyChange);
		if (earlyChange != null)
			result.put(FilterParams.EarlyChange_K, Integer.valueOf(earlyChange.intValue()));

		ASNumber k = parmsDict.getNumber(ASName.k_K);
		if (k != null)
			result.put(FilterParams.K_K, Integer.valueOf(k.intValue()));

		ASBoolean eol = parmsDict.getBoolean(ASName.k_EndOfLine);
		if (eol != null)
			result.put(FilterParams.DoEOL_K, Boolean.valueOf(eol.isTrue()));

		ASBoolean eba = parmsDict.getBoolean(ASName.k_EncodedByteAlign);
		if (eba != null)
			result.put(FilterParams.ByteAlign_K, Boolean.valueOf(eba.isTrue()));

		ASNumber rows = parmsDict.getNumber(ASName.k_Rows);
		if (rows != null)
			result.put(FilterParams.Rows_K, Integer.valueOf(rows.intValue()));

		ASBoolean eob = parmsDict.getBoolean(ASName.k_EndOfBlock);
		if (eob != null)
			result.put(FilterParams.DoRTC_K, Boolean.valueOf(eob.isTrue()));

		ASBoolean blackIs1 = parmsDict.getBoolean(ASName.k_BlackIs1);
		if (blackIs1 != null)
			result.put(FilterParams.BlackIs1_K, Boolean.valueOf(blackIs1.isTrue()));

		ASNumber drbe = parmsDict.getNumber(ASName.k_DamagedRowsBeforeError);
		if (drbe != null)
			result.put(FilterParams.DamagedRowsBeforeError_K, Integer.valueOf(drbe.intValue()));

		ASNumber bitsPerComponent = parmsDict.getNumber(ASName.k_BitsPerComponent);
		if (bitsPerComponent != null)
			result.put(FilterParams.BitsPerComponent_K, Integer.valueOf(bitsPerComponent.intValue()));
		
		return result;
		}
		catch(PDFUnableToCompleteOperationException e)
		{
			throw new PDFInvalidParameterException("Invalid decode paramaters dictionary", e);
		}
	}


	/**
	 * Apply a named input filter to the stream.A PDFCosParseException is thrown for
	 * filter names that are not supported or incorrect.
	 * To support valid filters, the user can register their filter encoders/decoders with 
	 * the PDFFilterRegistry as a document open option and they will be used or the client
	 * can extract the stream content directly.
	 * 
	 * @param inStm
	 * @param filterName
	 * @param params
	 * @param filterRegistry
	 * @return decoded bytes in an inputStream 
	 * @throws PDFCosParseException
	 * @throws IOException 
	 */
	static public InputStream applyFilter(InputStream inStm, ASName filterName, FilterParams params, CustomFilterRegistry filterRegistry) 
		throws PDFCosParseException, IOException
	{
		
		//Check the filter registry for a match
		if (filterRegistry != null)
		{
			try 
			{
				if (filterRegistry.isDecodeFilterRegistered(filterName))
					return filterRegistry.decode(filterName, inStm, params);
			} 
			catch (CustomFilterException e) 
			{
				throw new PDFCosParseException("Unable to use filter registry to decode stream that uses " + filterName,e);
			}
		}
		
		if (filterName.equals(ASName.k_FlateDecode) || filterName.equals(ASName.k_Fl)) {

			//The first two bytes in a compressed raw file corresponds to Zlib header
			//Hence for streams throwing "Incorrect data check", we would by pass adler 32 check by not computing it at the end of inflate process.
			//This is done by skipping first two bytes in the stream and turning on the "nowrap" option to true in Inflater object.
			//For more reference:- 
			//"http://stackoverflow.com/questions/33348192/attached-code-throws-java-util-zip-zipexception-incorrect-data-check-for-given"
			inStm.skip(2);
			InputStream result = new InflaterInputStream(inStm, new Inflater(true));

			if (params != null &&  params.containsKey(FilterParams.Predictor_K) &&
				((Integer)params.get(FilterParams.Predictor_K)).intValue() != 1 )
				return new TIFFInputStream(result, params);

			return result;
		}

		if (filterName.equals(ASName.k_ASCII85Decode) || filterName.equals(ASName.k_A85))
			return new ASCII85InputStream(inStm, params);

		if (filterName.equals(ASName.k_ASCIIHexDecode) || filterName.equals(ASName.k_AHx))
			return new ASCIIHexInputStream(inStm, params);

		if (filterName.equals(ASName.k_LZWDecode) || filterName.equals(ASName.k_LZW))
			return new LZWInputStream(inStm, params);

		if (filterName.equals(ASName.k_Crypt))
			return inStm; // Crypt filter is handled by the encryption code
		
		if (filterName.equals(ASName.k_CCITTFaxDecode) || filterName.equals(ASName.k_CCF))
			return new CCITTFaxInputStream(inStm, params);
		
		if (filterName.equals(ASName.k_DCTDecode) || filterName.equals(ASName.k_DCT))
			return new DCTInputStream(inStm, params);

		if (filterName.equals(ASName.k_RunLengthDecode) || filterName.equals(ASName.k_RL))
			return new RunLengthInputStream(inStm, params);
	
		throw new PDFCosParseException("Undefined filter - " + filterName.asString(true));
		
	}

	/**
	 * Apply a named output filter to the stream.
	 * @throws PDFCosParseException if we get a junk filter name.
	 * There are valid filters (like DCTDecode for images) that we really
	 * don't support, but there's no way to get here with those unless
	 * the client tries to extract the stream content directly. 
	 */
	static public OutputStream applyFilter(OutputStream inStm, ASName filterName, FilterParams params, CustomFilterRegistry filterRegistry) 
		throws PDFCosParseException
	{
		//Check the filter registry for a match
		if (filterRegistry != null)
		{
			try 
			{
				if (filterRegistry.isEncodeFilterRegistered(filterName))
					return filterRegistry.encode(filterName, inStm, params);
			} 
			catch (CustomFilterException e) 
			{
				throw new PDFCosParseException("Unable to use filter registry to encode stream that uses " + filterName,e);
			}
		}
		
		if (filterName.equals(ASName.k_FlateDecode) || filterName.equals(ASName.k_Fl)) {
			
			OutputStream result = new FilterOutputStream(inStm, new Deflater(Deflater.BEST_COMPRESSION));
			
			if (params != null &&  params.containsKey(FilterParams.Predictor_K) &&
				((Integer)params.get(FilterParams.Predictor_K)).intValue() != 1 )
				return new TIFFOutputStream(result, params);

			return result;
		}

		if (filterName.equals(ASName.k_ASCII85Decode) || filterName.equals(ASName.k_A85))
			return new ASCII85OutputStream(inStm, params);

		if (filterName.equals(ASName.k_ASCIIHexDecode) || filterName.equals(ASName.k_AHx))
			return new ASCIIHexOutputStream(inStm, params);

		if (filterName.equals(ASName.k_LZWDecode) || filterName.equals(ASName.k_LZW))
			return new LZWOutputStream(inStm, params);

		if (filterName.equals(ASName.k_RunLengthDecode) || filterName.equals(ASName.k_RL))
			return new RunLengthOutputStream(inStm);
		
		if (filterName.equals(ASName.k_CCITTFaxDecode) || filterName.equals(ASName.k_CCF))
			return new CCITTFaxOutputStream(inStm, params);
			
		if (filterName.equals(ASName.k_Crypt))
			return inStm; // Crypt filter is handled by the encryption code

		throw new PDFCosParseException("Undefined filter - " + filterName.asString(true));
	}

	/**
	 * Apply a named input filter to the stream.
	 * @throws PDFCosParseException if we get a junk filter name.
	 * There are valid filters that we really
	 * don't support, but there's no way to get here with those unless
	 * the client tries to extract the stream content directly. 
	 * @throws IOException 
	 */
	public static InputStream applyFilter(InputStream srcStm, ASName curFilter,
			FilterParams params) throws PDFCosParseException, IOException 
	{
		return applyFilter(srcStm, curFilter, params, null);
	}
	
	/**
	 * Apply a named output filter to the stream.
	 * @throws PDFCosParseException if we get a junk filter name.
	 * There are valid filters (like DCTDecode for images) that we really
	 * don't support, but there's no way to get here with those unless
	 * the client tries to extract the stream content directly. 
	 */
	static public OutputStream applyFilter(OutputStream inStm, ASName filterName, FilterParams params) 
	throws PDFCosParseException
	{
		return applyFilter(inStm, filterName, params, null);
	}
	
	/**
	 * Checks if the filter is a custom encode filter and then updates the filterParams based on 
	 * functionality available to the custom encode filter 
	 * 
	 * @param filterName
	 * @param params
	 * @param filterRegistry
	 * @return a CosObject representing an new updated filterParam
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	static public CosObject updateCustomFilterParams(ASName filterName, CosObject params, CustomFilterRegistry filterRegistry) 
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		//Check the filter registry for a match
		if (filterRegistry != null)
		{
			if (filterRegistry.isEncodeFilterRegistered(filterName)) {
				FilterParams fParams = FilterStream.buildFilterParams(params); 
				fParams = filterRegistry.updateFilterParams(filterName, fParams);
				if (fParams == null) return null;
				return fParams.toCosObject(params.getDocument());
			}
		} 
		return params;
	}

	/**
	 * Checks if the filter is a custom encode filter and then updates the filterParams based on 
	 * functionality available to the custom encode filter 
	 * 
	 * @param filterName
	 * @param params
	 * @param filterRegistry
	 * @return an ASDictionary representing an new updated filterParam
	 * @throws PDFInvalidParameterException 
	 */
	static public ASDictionary updateCustomFilterParams(ASName filterName, ASDictionary params, CustomFilterRegistry filterRegistry) throws PDFInvalidParameterException 
	{
	//Check the filter registry for a match
		if (filterRegistry != null)
		{
			if (filterRegistry.isEncodeFilterRegistered(filterName)) {
				FilterParams fParams = FilterStream.buildFilterParams(params); 
				fParams = filterRegistry.updateFilterParams(filterName, fParams);
				if (fParams == null) return null;
				return fParams.toASDictionary();
			}
		} 
		return params;
	}
}
