/*
 * File: FlateInputStream.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2003-2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/
package com.adobe.internal.pdftoolkit.core.filter;

import java.io.InputStream;

/**
 * FlateInputStream
 *  Implements the GZIP decompression algorithm and ZLIB format
 *  RFCs 1950 and 1951
 *
 *	Important: This filter assumes the input stream is an ISO/Latin-1
 *	stream, of 1-byte (not Unicode) characters!
 *
 * Copyright (C) 1995-1996 Jean-loup Gailly
 * Copyright (C) 1996-2005 Adobe Systems Incorporated
 */
class HuffTree
{
	//Constants
	static final int	BMAX =	15;			/* maximum bit length of any code */
	static final int	N_MAX =	288;		/* maximum number of codes in any set */

	static final int	EXOP_TYPE_MASK	  = 0xE0;
	static final int	EXOP_UNIBYTE_LITERAL    = 0;
	static final int	EXOP_SUBTREE	    = 0;
	static final int	EXOP_LEN_DIST	   = 0x40;
	static final int	EXOP_END_OF_BLOCK       = 0x60;
	static final int	EXOP_ILLEGAL_CODE       = 0xC0;
	static final int	EXOP_MULTIBYTE_LITERAL  = 0xA0;	/* Adobe invention -- not used (at this time) */

	// Class variables
	protected byte	Exop[];
	protected byte	Bits[];
	protected int	Base[];
	protected int	maxLook;

	// Huffman tree creation
	public HuffTree (
					 int		b[],				/* code lengths in bits (all assumed <= BMAX) */
					 int		n,		  /* number of codes (assumed <= N_MAX) */
					 int		s,		  /* number of simple-valued codes (0..s-1) */
					 short	d[],				/* list of base values for non-simple codes */
					 byte	e[],				/* list of extra bits for non-simple codes */
					 int		m					/* Maximum lookup bits */
					 )
	{
		int		t_needed;
		int		t_avail;
		int		codeCount[] = new int[BMAX+1];	/* number of codes for each bit length */
		int		minCodeLength, maxCodeLength;
		int		h;						/* table level */
		int		i;						/* counter, current code */
		int		j;						/* counter */
		int		k;						/* number of bits in current code, starting shortest */
		int		p;						/* index into c[], b[], or v[] */
		int		q;						/* points to current table */
		byte	r_exop;
		byte	r_bits;
		int		r_base;					/* table entry for structure assignment */
		int		u[] = new int [BMAX];	/* table stack */
		int		v[] = new int [N_MAX];	/* values in order of bit length */
		int		w;						/* bits before this table == (l * h) */
		int		x[] = new int [BMAX+1];	/* bit offsets, then code stack */
		int		xp;						/* pointer into x */
		int		nt_carry;
		int		z;						/* number of entries in current table,
																		   must have lifetime covering
																		   "for (; k<=g; k++)" loop
																		*/

		Exop = null;
		Bits = null;
		Base = null;

		/* Generate counts (number of codes) for each bit length */
		for (i = 0; i <= BMAX; i++)
			codeCount[i] = 0;

		for (i = 0; i < n; i++) {
			codeCount[b[i]]++;
		}

		if (codeCount[0] == n)
			return;						/* no positive-length codes */

		/* Find minimum and maximum length, bound maxLook by those */
		for (minCodeLength = 1; minCodeLength <= BMAX; minCodeLength++ )	/* set minimum code length */
			if (codeCount[minCodeLength] > 0)
				break;

		for (maxCodeLength = BMAX; maxCodeLength > 0; maxCodeLength--)		/* set maximum code length */
			if (codeCount[maxCodeLength] > 0)
				break;

		maxLook = m;					/* ensure minCodeLength <= maxLook <= maxCodeLength */
		if (maxLook < minCodeLength)
			maxLook = minCodeLength;
		if (maxLook > maxCodeLength)
			maxLook = maxCodeLength;

		/* Adjust last length count to fill out codes, if needed */
		for (i = minCodeLength, nt_carry = (1 << (minCodeLength - 1)); i <= maxCodeLength; i++) {
			nt_carry <<= 1;
			if ( nt_carry < codeCount[i] )
				return;					/* code space overcommitted */
			nt_carry -= codeCount[i];
		}
		if ( (1 << maxLook) < nt_carry )
			return;
		/* In a worst case, nt_carry could be nearly
		   as large as (1 << BMAX).  This
		   could result in a huge decoding tree when reading an
		   erroneous input.  The 1 << maxLook limit is somewhat arbitrary.
		*/
		codeCount[maxCodeLength] += nt_carry;

		{
			/* Pre-calculate the space that will be required for all of the
			   decode tables.  The major invariant in tree construction is
			   that all EXOP_SUBTREE links occur in tables that are at
			   levels that are integer multiples of maxLook.  Subject to that
			   invariant, the tables are as small as possible.
			*/
			int	n_terms = 0;
			int	n_links = 1;	/* at next_lower multiple of maxLook (== 0) */
			t_needed = t_avail = 0;

			for ( i = minCodeLength; i <= maxCodeLength; i++ )
				{
					int	lev_diff = ((i - 1) % maxLook) + 1;  /* 1..maxLook */
					int	n_tables = 0;

					n_terms = (2 * n_terms) + codeCount[i]; /* number of terminals needing placement at level i */
					if ( lev_diff == maxLook )
						{
							/* Complete set of tables at this level: terminals (and links if i < g). */
							n_tables = n_links;
							n_links = (n_tables << maxLook) - n_terms;
							n_terms = 0;
						}
					else if ( (maxLook < i) || (i == maxCodeLength) )
						{
							n_tables = n_terms >> lev_diff;
							/* Any tables at this level must be completely full of terminals. */
							n_links -= n_tables;
							n_terms -= (n_tables << lev_diff);
						}
					t_needed += (n_tables << lev_diff);
				}

			/* Now allocate the arrays */
			Exop = new byte[t_needed];
			Bits = new byte[t_needed];
			Base = new int [t_needed];
		}

		/* Generate starting offsets into the value table for each length */
		for ( i = 1, x[0] = 0, x[1] = 0; i < maxCodeLength; i++ )
			x[i + 1] = x[i] + codeCount[i];

		/* Make a table of values in order of bit lengths */
		for ( i = 0; i < n; i++ )
			{
				if (b[i] != 0 )
					v[x[b[i]]++] = i;
			}
		n = x[maxCodeLength];

		/* Generate the Huffman codes and for each, make the table entries */
		/* NOTE that the Huffman codes are effectively high bit on the right,
		 *	so that consecutive codes are bit reversed in k-bits
		 */
		i = 0;							/* first Huffman code is zero */
		p = 0;							/* grab values in bit order */
		h = -1;							/* no tables yet--level -1 */
		w = -maxLook;					/* bits decoded == (maxLook * h) */
		u[0] = 0;						/* just to keep compilers happy */
		q = 0;							/* ditto */
		z = 0;							/* ditto */

		/* go through the bit lengths (k already is bits in shortest code) */
		for (k = minCodeLength; k <= maxCodeLength; k++)
			{
				for (int a = codeCount[k]; a-- > 0; )		/* unprocessed codes of length k */
					{
						int		f;				/* i repeats in table every f entries (because of bit reversal) */
						/* here i is the Huffman code of length k bits for value *p */
						/* make tables up to required level */
						while (k > w + maxLook)
							{
								h++;
								w += maxLook;							/* previous table always maxLook bits */

								/* compute minimum size table less than or equal to maxLook bits */
								z = maxCodeLength - w;
								if ( maxLook < z )
									z = maxLook;

								if ((f = 1 << (j = k - w)) > a + 1)		/* try a k-w bit table */
									{										/* too few codes for k-w bit table */
										f -= a + 1;							/* deduct codes from patterns left */
										xp = k;
										if (j < z)
											while (++j < z)					/* try smaller tables up to z bits */
												{
													if ((f <<= 1) <= codeCount[++xp])
														break;					/* enough codes to use up j bits */
													f -= codeCount[xp];			/* else deduct codes from patterns */
												}
									}
								z = 1 << j;					/* table entries for j-bit table */

								/* allocate and link in new table */
								q = t_avail;
								t_avail += z;

								u[h] = q;

								/* connect to last table, if there is one */
								if (h > 0)
									{
										x[h] = i;							/* save pattern for backing up */
										r_bits = (byte)maxLook;				/* bits to dump before this table */
										r_exop = (byte)(EXOP_SUBTREE + j);	/* bits in this table */
										j = i >> (w - maxLook);				/* (get around Turbo C bug) */
										r_base = q;
										/* byte offset to this table */
										Exop[u[h-1]+j] = r_exop;			/* connect to last table */
										Bits[u[h-1]+j] = r_bits;
										Base[u[h-1]+j] = r_base;
									}
							}

						/* set up table entry in r */
						r_bits = (byte)(k - w);
						if (p >= n)
							{
								r_exop = (byte)EXOP_ILLEGAL_CODE;		/* out of values--invalid code */
								r_base = 0;
							}
						else if (v[p] < s)
							{
								r_exop = (byte)(v[p] < 256 ? EXOP_UNIBYTE_LITERAL : EXOP_END_OF_BLOCK);
								r_base = v[p++];	  /* simple code is just the value */
							}
						else
							{
								int index = v[p++] - s;
								r_exop = (byte)(EXOP_LEN_DIST + e[index]); /* non-simple--look up in lists */
								r_base = d[index];
							}

						/* fill code-like entries with r */
						f = 1 << (k - w);
						for (j = i >> w; j < z; j += f) {
							Exop[q+j] = r_exop;
							Bits[q+j] = r_bits;
							Base[q+j] = r_base;
						}

						/* backwards increment the k-bit (bit reversed) code i */
						for (j = 1 << (k - 1); (i & j) != 0; j >>= 1)
							i ^= j;
						i ^= j;

						/* backup over finished tables */
						while ((i & ((1 << w) - 1)) != x[h])
							{
								h--;		    /* don't need to update q */
								w -= maxLook;
							}
					}
			}
	}
}

public class FlateInputStream extends DecodeInputStream
{
	// Variables
	private boolean isInit;

	// Constructors
	/**
	 * Creates an input stream filter.
	 * @param in	the input stream
	 */
	public FlateInputStream(
							InputStream in,
							int	 inSize,
							int	 outSize,
							FilterParams diparams )
	{
		super(in, inSize, outSize, 0, diparams);
		initInput();
	}

	public FlateInputStream(InputStream  in, FilterParams p)
	{
		super(in, 0, p);
		initInput();
	}

	public FlateInputStream(InputStream  in)
	{
		this(in, null);
	}

	// Methods
	private void initInput()
	{
		isInit = false;
	}

	@Override
	public void fill()
	{
		if (!isInit) {
			inflateInit();
			isInit = true;
		}
		inflate_blocks();
	}

	//Methods from the Flate code base
	//Constants
	/* Table for deflate from PKZIP's appnote.txt. */
	private static final byte border[] = { /* Order of the bit length code lengths */
		16, 17, 18, 0, 8, 7, 9, 6, 10, 5, 11, 4, 12, 3, 13, 2, 14, 1, 15};

	private static final short cplens[] = { /* Copy lengths for literal codes 257..285 */
		3, 4, 5, 6, 7, 8, 9, 10, 11, 13, 15, 17, 19, 23, 27, 31,
		35, 43, 51, 59, 67, 83, 99, 115, 131, 163, 195, 227, 258, 0, 0};

	private static final byte cplext[] = { /* Extra bits for literal codes 257..285 */
		0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 2, 2, 2, 2,
		3, 3, 3, 3, 4, 4, 4, 4, 5, 5, 5, 5, 0,
		(byte)0x80 /*EXOP_ILLEGAL_CODE - EXOP_LEN_DIST*/,
		(byte)0x80 /*EXOP_ILLEGAL_CODE - EXOP_LEN_DIST*/};

	private static final short cpdist[] = { /* Copy offsets for distance codes 0..29 */
		1, 2, 3, 4, 5, 7, 9, 13, 17, 25, 33, 49, 65, 97, 129, 193,
		257, 385, 513, 769, 1025, 1537, 2049, 3073, 4097, 6145,
		8193, 12289, 16385, 24577};

	private static final byte cpdext[] = { /* Extra bits for distance codes */
		0, 0, 0, 0, 1, 1, 2, 2, 3, 3, 4, 4, 5, 5, 6, 6,
		7, 7, 8, 8, 9, 9, 10, 10, 11, 11,
		12, 12, 13, 13};

	/* Block input states */
	private static final int TYPE =		0;	/* get type bits (3, including end bit) */
	private static final int LENS =		1;	/* get lengths for stored */
	private static final int STORED =	2;	/* processing stored block */
	private static final int TABLE =	3;	/* get table lengths */
//	private static final int BTREE =	4;	/* get bit lengths tree for a dynamic block */
//	private static final int DTREE =	5;	/* get length, distance trees for a dynamic block */
	private static final int CODES =	6;	/* processing fixed or dynamic block */
	private static final int COPY =		7;	/* copy previous data in window */
	private static final int DONE =		8;	/* finished last block, done */
	private static final int BAD =		9;	/* got a data error--stuck here */
	private static final int FINAL =	10;	/* final idle state */

	private int			blockMode;
	private HuffTree	blockLengthTree;
	private HuffTree	fixedLengthTree;
	private HuffTree	blockDistTree;
	private HuffTree	fixedDistTree;
	private boolean		lastBlock;
	private boolean     localEOF;

	private int		wbits;			/* log2(window size)  (8..15, defaults to 15) */
	private int		wmask;			/* mask for window index */
	private byte	window[];
	private int		windex;			/* index into look-back buffer */

	// BitStream handling
	private int	    rawBits, rawBitsLeft;

	/* Get some bits, but leave them in case we didn't really need all of them */
	private int GetBits(int t)
	{
		int		inChar;

		fillLoop:
		while (t > rawBitsLeft &&
			   !pendingEOF && pendingException == null)
			{
				if ( inCount <= inPos ) {
					if (!fillInputBuffer()) {
						pendingException =
							new FlateFilterException("Flate: missing EOF");
						break fillLoop;
					}
				}
				inChar = inBuf[inPos++] & 0xFF;
				rawBits += inChar << rawBitsLeft;
				rawBitsLeft += 8;
			}
		if (t > rawBitsLeft)
			t = rawBitsLeft;

		return rawBits & ((1 << t) - 1);
	}

	/* Use up some bits */
	private void DumpBits(int t)
	{
		if (t > rawBitsLeft) {		/* should never happen */
			GetBits(t);
			if (t > rawBitsLeft)	/* only happens at EOF */
				t = rawBitsLeft;
		}
		rawBits >>>= t;
		rawBitsLeft -= t;
	}

	/* Synchronize to a byte boundary */
	private void DumpBitsToByte()
	{
		DumpBits(rawBitsLeft & 7);
	}

	/* Check for data available */
	private boolean AtEOF()
	{
		return localEOF && rawBitsLeft == 0;
	}

	// Adler32 computation
	private static final int	Adler_BASE = 65521;	/* largest prime smaller than 65536 */
	private int					adler_s1, adler_s2;

	private void Adler32(int b)
	{
		adler_s1 += b & 0x0ff;
		if (adler_s1 >= Adler_BASE)
			adler_s1 -= Adler_BASE;

		adler_s2 += adler_s1;
		if (adler_s2 >= Adler_BASE)
			adler_s2 -= Adler_BASE;
	}

	//TODO: Something is wrong here, all files fail. Check out and fix.
	private void checkAdler()
	{
		int		a1, a2, s1, s2;

		DumpBitsToByte();
		a1 = adler_s1;      /* Getting bits will change the Adler sum */
		a2 = adler_s2;
		s2 =  GetBits(8) << 8;	DumpBits(8);
		s2 += GetBits(8);		DumpBits(8);
		s1 =  GetBits(8) << 8;	DumpBits(8);
		s1 += GetBits(8);		DumpBits(8);

		if (s1 != a1 || s2 != a2) {
			blockMode = BAD;
		}
		localEOF = true;
	}



	// Block handling
	void inflate_blocks ()
	{
		fillLoop:
		while (outCount < outBuf.length &&
			   !pendingEOF && pendingException == null)
			{
				int copyLen = 0;
				int copyOffset = 0;
				/* process input based on current state */
				switch (blockMode)
					{
					case TYPE:
						{
							int		t;

							t = GetBits(3);	DumpBits(3);
							if ((t&1) != 0)
								lastBlock = ((t & 1) != 0) || AtEOF();
							switch (t >> 1)
								{
								case 0:						/* stored */
									DumpBitsToByte();			/* go to byte boundary */
									blockMode = LENS;			/* get length of stored block */
									break;

								case 1:						/* fixed */
									if (fixedLengthTree == null) {		/* build fixed length trees on first use */
										int	b[] = new int[HuffTree.N_MAX];
										int		i;

										for (i = 0; i <= 143; i++)
											b[i] = 8;
										for (i = 144; i <= 255; i++)
											b[i] = 9;
										for (i = 256; i <= 279; i++)
											b[i] = 7;
										for (i = 280; i <= 287; i++)
											b[i] = 8;

										fixedLengthTree = new HuffTree(b, HuffTree.N_MAX, 257, cplens, cplext, 9);
									}

									if (fixedDistTree == null) {
										int	b[] = new int[31];
										int		i;

										for (i = 0; i < 31; i++)
											b[i] = 5;

										fixedDistTree = new HuffTree(b, 30, 0, cpdist, cpdext, 6);
									}

									blockLengthTree = fixedLengthTree;
									blockDistTree = fixedDistTree;
									blockMode = CODES;
									break;

								case 2:						/* dynamic */
									blockMode = TABLE;
									break;

								case 3:						/* illegal */
									blockMode = BAD;
									continue fillLoop;
								}
						}
						break;

					case LENS:
						copyLen = GetBits(16);	DumpBits(16);
						if (GetBits(16) != (copyLen ^ 0x0ffff)) {
							blockMode = BAD;		/* NLEN not the complement of LEN */
							continue fillLoop;
						}
						DumpBits(16);
						blockMode = (copyLen > 0) ? STORED : (lastBlock ? DONE : TYPE);
						break;

					case STORED:
						{
							int		result = GetBits(8);

							DumpBits(8);
							copyLen--;
							if (AtEOF()) {
								lastBlock = true;
								copyLen = 0;			/* really an error */
							}
							outBuf[outCount++] = window[windex] = (byte)result;
							Adler32(result);
							windex = (windex + 1) & wmask;
							blockMode = (copyLen > 0) ? STORED : (lastBlock ? DONE : TYPE);
						}
						break;

					case TABLE:
						{
							int			hLit;		/* # of Literal/Lengthcodes - 257 (257-286) */
							int			hDist;		/* # of Distance codes - 1 (1-32) */
							int			hClen;		/* # of Code Length codes - 4 (4-19)*/
							int			i, btSize;
							int			b[];		/* combined length table */
							int			bt[];		/* code table for the dynamic table */
							HuffTree	bTree;

							hLit =  257 + GetBits(5);	DumpBits(5);
							hDist = 1 + GetBits(5);		DumpBits(5);
							hClen = 4 + GetBits(4);		DumpBits(4);

							if (hLit > 288 || hDist > 30) {	/* too many length or distance symbols */
								blockMode = BAD;
								continue fillLoop;
							}

							btSize = hLit + hDist;
							if (btSize < 19)
								btSize = 19;
							b = new int[btSize];
							//		blockMode = BTREE;
							/* fall through */

							//		case BTREE:		/* Get the Huffman codes for the Huffman codes table */
							bt = new int[19];
							for (i = 0; i < 19; i++)
								bt[i] = 0;

							for (i = 0; i < hClen; i++) {
								bt[border[i]] = GetBits(3);	DumpBits(3);
							}
							bTree = new HuffTree(bt, 19, 19, null, null, 7);
							//		blockMode = DTREE;
							/* fall through */

							//		case DTREE:
							for (int ti = 0; ti < btSize;)
								{
									int		t, j, e;

									e = HuffTree.EXOP_SUBTREE + bTree.maxLook;
									t = 0;

									while ((e & HuffTree.EXOP_TYPE_MASK) == HuffTree.EXOP_SUBTREE
										   && e != HuffTree.EXOP_UNIBYTE_LITERAL) {
										j = GetBits(e & 15);
										DumpBits(bTree.Bits[t+j]);
										e = bTree.Exop[t+j];
										t = bTree.Base[t+j];
									}
									if (e != HuffTree.EXOP_UNIBYTE_LITERAL) {	/* e should always be EXOP_UNIBYTE_LITERAL */
										blockMode = BAD;
										continue fillLoop;
									}

									if (t < 16) {
										b[ti++] = t;
									}
									else {	/* t == 16..18 */
										switch(t) {
										case 16:	/* REP_3_6: copy the previous code length 3-6 times */
											i = 2;		/* additional bits needed */
											j = 3;		/* counting base */
											break;

										case 17:	/* REPZ_3_10: repeat a code length of 0 for 3-10 times */
											i = 3;
											j = 3;
											break;

										case 18:	/* REPZ_11_138: repeat a code length of 0 for 11-138 times */
											i = 7;
											j = 11;
											break;

										default:    /* can't get here */
											blockMode = BAD;
											i = j = 0;
											continue fillLoop;
										}

										j += ti + GetBits(i);	DumpBits(i);
										if (j > btSize || (t == 16 && ti == 0)) {
											blockMode = BAD;	/* invalid bit length repeat */
											continue fillLoop;
										}

										if (t == 16) {
											while (ti < j) {
												b[ti] = b[ti-1];
												++ti;
											}
										}
										else {
											while (ti < j)
												b[ti++] = 0;
										}
									}
								}

							blockLengthTree = new HuffTree(b, hLit, 257, cplens, cplext, 9);

							for (i = 0; i < hDist; i++)	/* copy distance codes to array base */
								b[i] = b[hLit + i];
							blockDistTree = new HuffTree(b, hDist, 0, cpdist, cpdext, 6);
							if (blockLengthTree.Bits == null || blockDistTree.Bits == null) {
								blockMode = BAD;
								continue fillLoop;
							}
						}
						blockMode = CODES;
						/* fall through - first time */

					case CODES:
						{
							int		t, j, e;

							e = HuffTree.EXOP_SUBTREE + blockLengthTree.maxLook;
							t = 0;

							while ((e & HuffTree.EXOP_TYPE_MASK) == HuffTree.EXOP_SUBTREE
								   && e != HuffTree.EXOP_UNIBYTE_LITERAL) {
								j = GetBits(e & 15);
								DumpBits(blockLengthTree.Bits[t+j]);
								e = blockLengthTree.Exop[t+j];
								t = blockLengthTree.Base[t+j];
							}

							switch(e & HuffTree.EXOP_TYPE_MASK) {
							case HuffTree.EXOP_UNIBYTE_LITERAL:
								outBuf[outCount++] = window[windex] = (byte)t;
								Adler32(t);
								windex = (windex + 1) & wmask;
								break;

							case HuffTree.EXOP_LEN_DIST:
								{
									copyLen = t + GetBits(e & 15);
									DumpBits(e & 15);

									/* Get Distance code */
									e = HuffTree.EXOP_SUBTREE + blockDistTree.maxLook;
									t = 0;

									while ((e & HuffTree.EXOP_TYPE_MASK) == HuffTree.EXOP_SUBTREE
										   && e != HuffTree.EXOP_UNIBYTE_LITERAL) {
										j = GetBits(e & 15);
										DumpBits(blockDistTree.Bits[t+j]);
										e = blockDistTree.Exop[t+j];
										t = blockDistTree.Base[t+j];
									}
									if ((e & HuffTree.EXOP_TYPE_MASK) != HuffTree.EXOP_LEN_DIST) {	/* e should always be EXOP_LEN_DIST */
										blockMode = BAD;
										continue fillLoop;
									}
									copyOffset = t + GetBits(e & 15);
									DumpBits(e & 15);
									blockMode = COPY;
								}
								break;

							case HuffTree.EXOP_END_OF_BLOCK:
								blockLengthTree = null;		/* help memory management */
								blockDistTree = null;
								blockMode = lastBlock ? DONE : TYPE;
								break;

							default:
								blockLengthTree = null;		/* help memory management */
								blockDistTree = null;
								blockMode = BAD;	/* invalid code */
								continue fillLoop;
							}
						}
						break;

					case COPY:
						{
							int		result = window[(windex - copyOffset) & wmask] & 0x0ff;

							outBuf[outCount++] = window[windex] = (byte)result;
							Adler32(result);
							windex = (windex + 1) & wmask;
							if (--copyLen <= 0)
								blockMode = CODES;	/* done with this one */
						}
						break;

					case BAD:
					default:
						blockMode = FINAL;
						if (pendingException == null)
							pendingException =
								new FlateFilterException("Flate: bad input stream");
						break fillLoop;

					case DONE:
						blockMode = FINAL;
						checkAdler();
						break;						/* checkAdler may have changed blockMode */

					case FINAL:
						blockLengthTree = null;		/* help memory management */
						blockDistTree = null;
						fixedLengthTree = null;		/* help memory management */
						fixedDistTree = null;
						pendingEOF = true;
						break fillLoop;
					}
			}
	}

	private void inflateInit()
	{
		localEOF = false;
		rawBits = rawBitsLeft = 0;
		blockLengthTree = null;
		fixedLengthTree = null;
		blockDistTree = null;
		fixedDistTree = null;

		int cmf = GetBits(8);   DumpBits(8);    /* Read the header */
		int flg = GetBits(8);   DumpBits(8);
		if (localEOF							/* EOF at start */
			||(((cmf << 8) + flg) % 31) != 0	    /* Check bits in error */
			||  (cmf & 0x0f) != 8				    /* Compression type must be 8 */
			||  (flg & 0x20) != 0 ) {			    /* preset dictionary not allowed */
			blockMode = BAD;
			lastBlock = true;
		}
		else {
			adler_s1 = 1;
			adler_s2 = 0;
			wbits = (cmf >> 4) + 8;				/* log2 of LZ77 window size */
			wmask = (1 << wbits) - 1;
			window = new byte[wmask+1];			/* output buffer */
			windex = 0;
			blockMode = TYPE;
			lastBlock = false;
		}
	}
}
