/*
 * File: LZWEngine.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2003-2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/

package com.adobe.internal.pdftoolkit.core.filter;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

/*
 *
 * LZWEngine
 *
 * Copyright (C) 1996-2005 Adobe Systems Incorporated
 */
class LZWEngine
{
	// Constants
	private static final int	LZWMAXCODE = 4096;	/* codes */
	private static final int	LZWMINCODELEN = 9;	/* bits */
	private static final int	LZWMAXCODELEN = 12;	/* bits */
	private static final int	HASHTABSIZE = 8191;	/* must be prime */

	/* This LZW coding is intended to be identical to the TIFF 5.0 spec.
	 * when LZWMINCODELEN == 9, 256+2 codes are defined
	 * Codes 0 - 255 represent their literal byte values
	 * Code 256 is the "Clear" code
	 * Code 257 is the "EOD" code
	 * Codes >=258 represent multi-byte sequences
	 */

	// Fields
	private boolean			reading;
	private LZWInputStream  rdr;
	private OutputStream	wtr;
	private long	    totalOut;

	private int	earlyChange;	//parameters to control the LZW table

	private int	codeSize;			//instance variables
	private int	wordSize;
	private int	clearCode;
	private int	stopCode;
	private int	nextCode;
	private int	lastCode;
	private int sendCode;
	private int sendLength;
	private int	residualBits;
	private int	codeMask;
	private int	codeMaskChange;
	private int	codeWord;

	// rather than have an array of LZWEntry[prefix, terminal],
	// we split the arrays for efficiency then pack everything into a 32-bit word
	// [prefix: 12 bits][codeLen: 12 bits][terminal: 8 bits]
	private int lzwEntry[];

	// hash table for looking up codes based on terminal
	private short hashTab[];

	// Constructors
	private LZWEngine(boolean reading)
	{
		this.reading = reading;
		earlyChange = 1;
		residualBits = 0;
		codeWord = 0;

		lzwEntry = new int[LZWMAXCODE+1];
		hashTab =  new short[HASHTABSIZE+1];    //=8192 = 2^13

		LZWInit(LZWMINCODELEN-1);
	}

	public LZWEngine(InputStream r)
	{
		this(true);
		rdr = (LZWInputStream)r;
	}

	public LZWEngine(OutputStream w)
	{
		this(false);
		wtr = w;
	}

	// Methods
	protected void setEarlyChange(int value)
	{
		earlyChange = value;
	}

	private void LZWInit (int codeSize)
	{
		int	entries;

		this.codeSize = codeSize;
		wordSize = codeSize + 1;
		codeMask = (1 << wordSize) - 1;
		codeMaskChange = codeMask + 1 - earlyChange;

		entries = (1 << codeSize);
		clearCode = entries;
		stopCode = entries+1;
		nextCode = entries+2;
		lastCode = -1;
		sendCode = -1;

		for (int code = 0; code < nextCode; code++) {      /* preset the base code entries */
			lzwEntry[code] = 0xfff00100 + code; //prefix = fff, length = 1; terminal = code
		}

		for (int i = 0; i < 32; i++)
			hashTab[i] = 0;
		System.arraycopy(hashTab, 0, hashTab, 32, 32);
		System.arraycopy(hashTab, 0, hashTab, 64, 64);
		System.arraycopy(hashTab, 0, hashTab, 128, 128);
		System.arraycopy(hashTab, 0, hashTab, 256, 256);
		System.arraycopy(hashTab, 0, hashTab, 512, 512);
		System.arraycopy(hashTab, 0, hashTab, 1024, 1024);
		System.arraycopy(hashTab, 0, hashTab, 2048, 2048);
		System.arraycopy(hashTab, 0, hashTab, 4096, 4096);
	}

	private int LZWSearchTable (int newData)
	{
		int       code, key;

		key = newData << 5;				 /* first probe value */

		while (true) {

			/* Since HASHSIZE is prime, any delta EXCEPT 0 generates
			 *  the ring of integers mod HASHSIZE.
			 */
			key += lastCode + 1;
			while (key >= HASHTABSIZE)
				key -= HASHTABSIZE;

			code = hashTab[key];

			if (code == 0) {				/* empty cell */
				hashTab[key] = (short)nextCode;				/* new code will go here (unless we clear the table) */
				return (-1);
			}

			int k = lzwEntry[code];
			if ((/*terminal*/k & 0xff) == newData &&
				(/*prefix*/  k >>> 20) == lastCode ) {
				return (code);
			}
		}
	}

	private void LZWAddTable (int newData)
	{
		lzwEntry[nextCode] = (lastCode << 20) +			     //prefix
			(lzwEntry[lastCode] & 0x000fff00) + 0x100 +    //codelen
			newData;				       //terminal
		if (reading)
			++nextCode;

		if (nextCode == codeMaskChange) {       /* have we run out of codewords */
			if (++wordSize > LZWMAXCODELEN)
				wordSize = LZWMAXCODELEN;
			codeMask = (1 << wordSize) - 1;
			codeMaskChange = codeMask + 1 - earlyChange;
		}

		if (!reading)
			++nextCode;
	}

	// The way we interface to the engine
	protected int get(byte buf[], int offset, int length)
	{
		int newCode, thisCode;
		int count = 0;

		outerLoop:
		while (length > 0) {
			if (sendCode < 0) {			 /* need a new code word */
				while(residualBits < wordSize) {
					int p;

					if (residualBits <= 8 && (p = rdr.read3()) >= 0) {
						codeWord = (codeWord << 24) + p;
						residualBits += 24;
					}
					else if ((p = rdr.read2()) >= 0) {
						codeWord = (codeWord << 16) + p;
						residualBits += 16;
					}
					else if ((p = rdr.read1()) >= 0) {
						codeWord = (codeWord << 8) + p;
						residualBits += 8;
					}
					else {  /* not enough bits, ran off end, just stop */
						codeWord = stopCode;
						residualBits = wordSize;
					}
				}

				residualBits -= wordSize;
				newCode = (codeWord >> residualBits) & codeMask;
				if (newCode == stopCode) {
					sendCode = stopCode;
					break outerLoop;
				}
				else if (newCode == clearCode) {
					LZWInit(codeSize);
					continue outerLoop;
				}

				/* If we see a new code we don't have in the table,
				 * it means that this code is a duplicate of the (just) previously
				 * seen code, with the FIRST character appended to the end.
				 */

				if (newCode >= nextCode) {					/* code not yet in table, will be soon */
					thisCode = lastCode;
					if (newCode > nextCode) {				/* error check */
						sendCode = stopCode;				/* this should never happen */
						break outerLoop;
					}
				}
				else
					thisCode = newCode;

				if (nextCode < LZWMAXCODE && lastCode >= 0) {   /* if there is room in the table & not the first entry */
					int j, k = lzwEntry[thisCode];

					while ((j = (k >>> 20)) != 0xfff) {		    /* find leading character of new code string */
						k = lzwEntry[j];
					}
					LZWAddTable (k & 0xff);					    /* remember previous+new for next time */
				}

				int stringLen = (lzwEntry[newCode] >> 8) & 0xfff;
				if (stringLen <= length) {	      /* Fast move of codes to output buffer */
					lastCode = thisCode = newCode;
					length -= stringLen;
					count += stringLen;
					sendCode = -1;

					int s = offset = offset + stringLen;
					int k = lzwEntry[thisCode];
					while (stringLen-- > 1) {
						buf[--s] = (byte)k;
						k = lzwEntry[k >>> 20];
					}
					buf[--s] = (byte)k;
				}
				else {				  /* We need to split up the output string */
					sendCode = lastCode = thisCode = newCode;
					sendLength = stringLen;
				}
			}
			else if (sendCode == stopCode) {
				break outerLoop;
			}

			/* We get here if we have a string to send,
			 * but we had to break it across output buffers.
			 */
			if (sendCode >= 0) {
				int toSend = sendLength;
				if (toSend > length)
					toSend = length;

				int toSkip = sendLength = sendLength - toSend;

				length -= toSend;
				count += toSend;

				int k = lzwEntry[sendCode];
				while (toSkip-- > 0)
					k = lzwEntry[k >>> 20];

				int s = offset = offset + toSend;
				while (toSend-- > 1) {
					buf[--s] = (byte)k;
					k = lzwEntry[k >>> 20];
				}
				buf[--s] = (byte)k;

				if (sendLength == 0)
					sendCode = -1;
			}
			}

		return count;
	}

	private void LZWPutEOF ()
		throws IOException
	{
		if (lastCode != -1) {
			/* PutCode (inline) */
			codeWord = (codeWord << wordSize) + lastCode;
			residualBits += wordSize;
			while (residualBits >= 8) {
				residualBits -= 8;
				wtr.write((byte)(codeWord >> residualBits));
				++totalOut;
			}
			LZWAddTable (0);
		}

		/* PutCode (inline) */
		codeWord = (codeWord << wordSize) + stopCode;
		residualBits += wordSize;
		while (residualBits >= 8) {
			residualBits -= 8;
			wtr.write((byte)(codeWord >> residualBits));
			++totalOut;
		}

		if (residualBits != 0) {
			wtr.write((byte)(codeWord << (8 - residualBits)));
			++totalOut;
		}
	}

	protected void put(int newData)
		throws IOException
	{
		int   newCode;

		if (newData == -1) {
			LZWPutEOF();
		}
		else if (lastCode == -1) {					/* On the very first character */
			lastCode = newData;						/* the character itself starts a code sequence */
		}
		else {
			newCode = LZWSearchTable (newData);		/* Have we seen this sequence before? */

			if (newCode == -1) {					/* If not... */

				/* PutCode (inline) */
				codeWord = (codeWord << wordSize) + lastCode;
				residualBits += wordSize;
				while (residualBits >= 8) {
					residualBits -= 8;
					wtr.write((byte)(codeWord >> residualBits));
					++totalOut;
				}

				if (nextCode < (LZWMAXCODE-3))		/* if there is room in the table */
					{
						LZWAddTable (newData);			/* remember previous+new for next time */
					}
				else {								/* otherwise */

					/* PutCode (inline) */
					codeWord = (codeWord << wordSize) + clearCode;
					residualBits += wordSize;
					while (residualBits >= 8) {
						residualBits -= 8;
						wtr.write((byte)(codeWord >> residualBits));
						++totalOut;
					}
					LZWInit (codeSize);
				}

				lastCode = newData;					/* new sequence starts with this code */
			}
			else									/* Continuing sequence... */
				lastCode = newCode;					/* continue building sequence if code was in the table */
		}
	}

	protected long getTotalOut()
	{
		return totalOut;
	}

	protected int getResidualBits()
	{
		return residualBits;
	}
}

