package com.adobe.internal.pdftoolkit.core.fontset;

import java.io.Serializable;
import java.util.Locale;

import com.adobe.agl.util.ULocale;
import com.adobe.fontengine.font.Font;
import com.adobe.fontengine.fontmanagement.Platform;
import com.adobe.fontengine.fontmanagement.postscript.PostscriptFontDescription;
import com.adobe.fontengine.inlineformatting.FallbackFontSet;
import com.adobe.fontengine.inlineformatting.css20.CSS20FontDescription;
import com.adobe.fontengine.inlineformatting.css20.CSS20GenericFontFamily;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFFontException;

/**
 * This interface provides access to the methods that a client can use to build a font set
 * that is used by in the formatting of text within a PDF document.
 * 
 * PDFFontSet objects implement the Serializable interface. The intent is that PDFFontSet objects 
 * would generally not be entirely regenerated from scratch. Instead, they would be deserialized. 
 * 
 */
public interface PDFFontSet extends Serializable
{
	public static final Locale ARABIC = new Locale("ar");
	public static final Locale THAI = new Locale("th");
	public static final Locale HEBREW = new Locale("he");
	
	/**
	 * Returns true if there is a "final fallback" font installed in ULocale.ROOT.
	 */
	public boolean hasRootFallback() throws PDFFontException;
	
	/**
	 * Add a given {@link com.adobe.fontengine.font.Font Font} to the 
	 * <code>PDFFontSet</code> for use in font lookup.
	 * There is an inherent priority ordering based on the order that <code>Font</code>
	 * objects are added with earlier added fonts having higher priority.
	 * The font shall be added for default locale available and windows platform. If clients want to specify
	 * locale and platform then they should call {@link #addFont(Font, Platform, ULocale)}.
	 * @param font a non-null valid Font object
	 * @throws PDFFontException
	 */
	public void addFont(Font font) throws PDFFontException;		
	
	/**
	 * Add a given {@link com.adobe.fontengine.font.Font Font} to the 
	 * <code>PDFFontSet</code> for use in font lookup.
	 * There is an inherent priority ordering based on the order that <code>Font</code>
	 * objects are added with earlier added fonts having higher priority.
	 * The font shall be added for locale and platform specified by clients. If either of the platform and
	 * locale are passed as null then defaults shall be used.
	 * @param font a non-null valid Font object
	 * @throws PDFFontException
	 */
	public void addFont(Font font, Platform platForm, ULocale locale) throws PDFFontException;

	/**
	 * Add a given {@link com.adobe.fontengine.font.Font Font} to the
	 * <code>PDFFontSet</code>. The provided PostScript and CSS font descriptions
	 * will be used during font lookup. Either array of descriptions can be null.
	 * If the <code>Font</code>'s native font descriptions
	 * are also desired, both this API and {@link #addFont(Font font) other addFont variants}
	 * can be called with the same Font instance.
	 * There is an inherent priority ordering based on the order that <code>Font</code>
	 * objects are added with earlier added fonts having higher priority.
	 */
	public void addFont(Font font, PostscriptFontDescription[] psDescriptions, CSS20FontDescription[] cssDescriptions)
		throws PDFFontException;
	
	/**
	 * Add an array of {@link com.adobe.fontengine.font.Font Font} objects to the 
	 * <code>PDFFontSet</code> for use in font lookup.
	 * There is an inherent priority ordering based on the order that </code>Font</code>
	 * objects are added with earlier added fonts having higher priority.
	 * These fonts shall be added for default locale available and windows platform. If clients want to specify
	 * locale and platform then they should call {@link #addFont(Font[], Platform, ULocale)}.
	 * @param fonts a non-null array of non-null valid Font objects
	 * @throws PDFFontException
	 */
	public void addFont(Font[] fonts) throws PDFFontException;
	
	/**
	 * Add an array of {@link com.adobe.fontengine.font.Font Font} objects to the 
	 * <code>PDFFontSet</code> for use in font lookup.
	 * There is an inherent priority ordering based on the order that </code>Font</code>
	 * objects are added with earlier added fonts having higher priority.
	 * The font shall be added for locale and platform specified by clients. If either of the platform and
	 * locale are passed as null then defaults shall be used.
	 * @param fonts a non-null array of non-null valid Font objects
	 * @throws PDFFontException
	 */
	public void addFont(Font[] fonts, Platform platForm, ULocale locale ) throws PDFFontException;

	/**
	 * Add a given {@link com.adobe.fontengine.font.Font Font} to the 
	 * <code>PDFFontSet</code> for use as a fallback font for the given locale.
	 * There is an inherent priority ordering based on the order that </code>Font</code>
	 * objects are added with earlier added fonts having higher priority.
	 * @param locale the locale for which this fallback font should be used
	 * @param font a non-null valid Font object
	 * @throws PDFFontException
	 */
	public void addFallbackFont(Locale locale, Font font)
		throws PDFFontException;

	/**
	 * Add an array of {@link com.adobe.fontengine.font.Font Font} objects to the 
	 * <code>PDFFontSet</code> for use as a fallback font for the given locale.
	 * There is an inherent priority ordering based on the order that </code>Font</code>
	 * objects are added with earlier added fonts having higher priority.
	 * @param locale the locale for which this fallback font should be used
	 * @param fonts a non-null array of non-null valid Font objects
	 * @throws PDFFontException
	 */
	public void addFallbackFont(Locale locale, Font[] fonts)
		throws PDFFontException;

	/**
	 * An array of font family names to use as replacements for the generic CSS font family names.
	 * This array of font family names replaces any previously set names for the specific font family.
	 * @param family
	 * @param replacements a non-null array of non-null valid String objects
	 * @throws PDFFontException
	 * @see com.adobe.fontengine.inlineformatting.css20.CSS20GenericFontFamily
	 */
	public void setGenericFontFamilyName(
			CSS20GenericFontFamily family, String[] replacements)
		throws PDFFontException;
	
	/**
	 * Returns the set of fallback fonts added to this font set.
	 * @return The set of fallback fonts added to this font set.
	 */
	public FallbackFontSet getFallbackFontSet() throws PDFFontException; 
	
	/**
	 * Sets if any error while adding font to fontset should be ignored
	 * @param ignoreFontLoadingErrors
	 */
	public void setIgnoreFontLoadingErrors(boolean ignoreFontLoadingErrors);
}
