/*************************************************************************
*
*	File: FontNameResolver.java
*
**************************************************************************
*
* ADOBE CONFIDENTIAL
* ___________________
*
*  Copyright 2012 Adobe Systems Incorporated
*  All Rights Reserved.
*
* NOTICE:  All information contained herein is, and remains
* the property of Adobe Systems Incorporated and its suppliers,
* if any.  The intellectual and technical concepts contained
* herein are proprietary to Adobe Systems Incorporated and its
* suppliers and are protected by trade secret or copyright law.
* Dissemination of this information or reproduction of this material
* is strictly forbidden unless prior written permission is obtained
* from Adobe Systems Incorporated.
**************************************************************************/

package com.adobe.internal.pdftoolkit.core.fontset.impl;

import java.util.HashSet;
import java.util.Set;

import com.adobe.agl.util.ULocale;
import com.adobe.fontengine.font.CoolTypeScript;
import com.adobe.fontengine.font.FontImpl;
import com.adobe.fontengine.font.FontLoadingException;
import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.UnsupportedFontException;
import com.adobe.fontengine.font.cff.CIDKeyedFont;
import com.adobe.fontengine.fontmanagement.CacheSupportInfo;
import com.adobe.fontengine.fontmanagement.Platform;
import com.adobe.fontengine.fontmanagement.fxg.FXGFontDescription;
import com.adobe.fontengine.fontmanagement.platform.PlatformFontDescription;
import com.adobe.fontengine.inlineformatting.css20.CSS20Attribute.CSSStyleValue;
import com.adobe.fontengine.inlineformatting.css20.CSS20Attribute.CSSWeightValue;
import com.adobe.fontengine.inlineformatting.css20.CSS20FontDescription;

/**
 * This class is used generate all the possible names for the given font, platform and locale.
 * @author hkansal
 *
 */
class FontNameResolver {
	
	private enum Style{
		Bold,
		Italic,
		BoldItalic
		
	}
	
	private static final String gFontNameSuffixRemTab = "-Big5";
	
	/**
	 * This method generates all the possible names of the given font for the given platform and locale.
	 * The logic of this method is copied from BuildList() function of CTFntSvr.cpp file
	 * in PDFL code.
	 * @param font
	 * @param platForm
	 * @param locale
	 * @return Set<String>
	 * @throws InvalidFontException
	 * @throws UnsupportedFontException
	 * @throws FontLoadingException
	 */
	static Set<String> doNameMangling(com.adobe.fontengine.font.Font font, Platform platForm, ULocale locale) throws InvalidFontException, UnsupportedFontException, FontLoadingException{
		
		Set<String> names = new HashSet<String>();
		CacheSupportInfo info = ((FontImpl)font).getCacheSupportInfo();
		String fontType = info == null ? null : info.getFontType();// get the font type.
		if(font.getPDFFontDescription().pdfFontIsTrueType())
		{
			PlatformFontDescription platformFontDesc[] = font.getPlatformFontDescription(platForm);
			String fullName = null;
			for(int i=0;i<platformFontDesc.length;i++){
				if(platformFontDesc[i].getLocale().equals(locale)){
					fullName = platformFontDesc[i].getPlatformName();
					break;// get the name for given locale.
				}
				
			}
			if(fullName != null)
				addFontToList(fullName, null, names);// add name in platform description to the list.
			String psName = font.getPDFFontDescription().getPostscriptName();// get post-script name.
			if(psName != null){
				addFontToList(psName, null, names);
				/* fix TT vs Type1 name difference; e.g. (SheideEG-Medium-Big5 vs SheideEG-Medium) */
				if((psName = fontNameRemoveSuffix(psName)) != null)
					addFontToList(psName, null, names);
			}
			
			CSS20FontDescription css = font.getPreferredCSS20FontDescription();
			Style style = null;
			if(css != null){// get the style of the font.We are only considering bold and italic.
				if(css.getStyle() == CSSStyleValue.ITALIC){
					if(css.getWeight() == CSSWeightValue.BOLD.getValue())
						style = Style.BoldItalic;
					else
						style = Style.Italic;
				}else if(css.getWeight() == CSSWeightValue.BOLD.getValue()){
					style = Style.Bold;
				}
			}
			CoolTypeScript writingScript = font.getCoolTypeScript();// get writing script of font.
			// There is no method in AFE to get the lfFaceName of the font for windows platform as PDFL does.
			// After discussion with AFE team, we are using FXG description for this which seems to be working fine.
			FXGFontDescription fxg[] = font.getFXGFontDescription(platForm);
			for(int i=0;fxg != null && i<fxg.length;i++){
				if(fxg[i].getLocale().equals(locale)){
					String winLogName = fxg[i] != null ? fxg[i].getFamilyName() : null;
					if(winLogName != null){
						if(writingScript == CoolTypeScript.JAPANESE || writingScript == CoolTypeScript.TRADITIONAL_CHINESE || writingScript == CoolTypeScript.SIMPLIFIED_CHINESE ||
								writingScript == CoolTypeScript.KOREAN)
							/* Need to add both with style and ones without. */
							addFontToList(winLogName, null, names);
						addFontToList(winLogName, style, names);
					}
					break;
				}
			}
			
			// Now try the family name.
			String familyName = css == null ? null : css.getFamilyName();
			/*	 NON CJK fonts & with style */
			if(familyName != null && writingScript != CoolTypeScript.JAPANESE && writingScript != CoolTypeScript.TRADITIONAL_CHINESE && writingScript != CoolTypeScript.SIMPLIFIED_CHINESE &&
					writingScript != CoolTypeScript.KOREAN)
				addFontToList(familyName, style, names);
		}
		else if("Type1Font".equalsIgnoreCase(fontType) || ((FontImpl)font).getFontData() instanceof CIDKeyedFont){
			String psName = font.getPDFFontDescription().getPostscriptName();
			if(psName != null){
				addFontToList(psName, null, names);
			}
		}
		
		return names;
	}
	
	
	
	/**
	 * The logic of this method is copied from AddFontToList() function of CTFntSvr.cpp file
	 * in PDFL code.
	 * @param fontName
	 * @param stylename
	 * @param altNames
	 */
	private static void addFontToList(String fontName, Style stylename, Set<String> altNames)
	{		
		/* remove spaces from fontname and stylename */
		String compressedName = stripSpacesAndAtSign(fontName);
//		if (stylename != null)
//			stylename = StripSpacesAndAtSign(stylename);
//
		if (stylename == null)
			altNames.add(compressedName);
//		else if (ASstrcasecmp(stylename,"regular") == 0 ||
//				 ASstrcasecmp(stylename,"roman") == 0 ||
//				 ASstrcasecmp(stylename,"normal") == 0 ||
//				 ASstrcasecmp(stylename,"plain") == 0)
//		{
//			/* Add font without style string and replace any entries
//			   that had other styles
//			*/
//			AddFontToDictionary(compressedName, technology, aFont, true);
//		}
		else if (stylename == Style.Bold || stylename == Style.Italic || stylename == Style.BoldItalic)
		{
	 
			/* Add to fontName only if it has style and Bold, BoldItalic, or Italic */
			StringBuilder newName = new StringBuilder().append(compressedName).append(',').append(stylename);

			/* Add font name with stylename */
			altNames.add(newName.toString());
		}
//		else if (ASstrcasecmp(stylename,"medium") == 0 ||
//				 ASstrcasecmp(stylename,"light") == 0)
//		{
//			/* Add without stylename for the cases above. However if we
//			   find a normal, regular or roman style, it will replace
//			   this entry.
//			*/
//			AddFontToDictionary(compressedName, technology, aFont, false);	/* <-- shouldn't it be true? */
//		}
	}
	
	/**
	 * Remove spaces and makes sure that name doesn't start from @.
	 * @param src
	 * @return String
	 */
	private static String stripSpacesAndAtSign(String src){
		if(src.startsWith("@"))
			src = src.substring(1);
		String[] splits = src.split(" ");
		StringBuilder strBuilder = new StringBuilder();
		for(int i=0;i<splits.length;i++){
			strBuilder.append(splits[i]);
		}
		return strBuilder.toString();
	}
	
	/**
	 * Removes "-Big5" suffix, if present.
	 * @param name
	 * @return String
	 */
	private static String fontNameRemoveSuffix(String name)
	{
		if(name.endsWith(gFontNameSuffixRemTab)){
			return name.substring(0, name.length() - gFontNameSuffixRemTab.length());
		}
		return null;
	}

}
