/* ****************************************************************************
 *
 *	File: ASCoordinate.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2004 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/
package com.adobe.internal.pdftoolkit.core.types;

import java.io.IOException;

import com.adobe.internal.io.stream.OutputByteStream;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFIOException;

/**
 * ASCoordinate represents a PDF coordintate point (x, y).
 */
public class ASCoordinate extends ASObject /*implements Point*/
{
	private double x;
	private double y;
	private static final double PI_BY_2 = Math.PI / 2;
	private static final double NEG_PI_BY_2 = -1 * Math.PI / 2;
	
	/**
	 * @param x
	 * @param y
	 */
	public ASCoordinate(double x, double y)
	{
		this.x = x;
		this.y = y;
	}

	/**
	 * Copy constructor.
	 * 
	 * @param coordinate
	 */
	public ASCoordinate(ASCoordinate coordinate)
	{
		this.x = coordinate.x();
		this.y = coordinate.y();
	}

	/**
	 * Translate the coordinate by dx, dy
	 *
	 * @param dx	The x offset
	 * @param dy	The y offset
	 * @return		A new coordinate offset by dx, dy
	 */
	public ASCoordinate translate(double dx, double dy)
	{
		return new ASCoordinate(x+dx, y+dy);
	}

	/**
	 * Transform the coordinate by a coordinate
	 * transformation matrix.
	 *
	 * @param ctm	- Coordinate Transformation Matrix
	 * @return	ASCoordinate
	 */
	public ASCoordinate transform(ASMatrix ctm)
	{
		ASMatrix m = ctm.preMultiply(x, y);
		return new ASCoordinate(m.getx(), m.gety());
	}

	/**
	 * Find the shortest Euclidean distance between
	 * this coordinate point and the specified
	 * coordinate point.
	 *
	 * @param coord
	 */
	public double distanceTo(ASCoordinate coord)
	{
		double a = coord.x - this.x;
		double b = coord.y - this.y;
		return Math.sqrt(a*a + b*b);
	}

	/**
	 * Find the square of shortest Euclidean distance between
	 * this coordinate point and the specified
	 * coordinate point.
	 *
	 * @param coord
	 * @return double
	 */
	public double distanceToSquare(ASCoordinate coord)
	{
		double a = coord.x - this.x;
		double b = coord.y - this.y;
		return a*a + b*b;
	}
	
	/**
	 * Find the angle as measured along a line from this
	 * coordinate to the specified coordinate.
	 *
	 * @param coord
	 */
	public double angleTo(ASCoordinate coord)
	{
		double x = coord.x - this.x;
		double y = coord.y - this.y;
		if(y == 0.0)
		{
			return x >= 0.0 ? 0.0 : Math.PI; 
		}
		if(x == 0.0)
		{
			return y > 0.0 ? PI_BY_2 : NEG_PI_BY_2;
		}
		return Math.atan2(y, x);
	}

	
	@Override
	public String toString()
	{
		return "(" + x + ", " + y + ")";
	}

	/**
	 * Writes the ASCoordinate in to the given OutputStream in the format expected by the PDF Spec.
	 *
	 * @param outputByteStream OutputStream to write to.
	 * @throws PDFIOException
	 */
	@Override
	public void write(OutputByteStream outputByteStream)
		throws PDFIOException
	{
		try {
			outputByteStream.write(this.toString().getBytes());
		} catch (IOException e) {
			throw new PDFIOException(e);
		}
	}

	public double x() {return x;}	
	public double y() {return y;}
}
