/* ****************************************************************************
 *
 *	File: ASMatrix.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2004 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/
package com.adobe.internal.pdftoolkit.core.types;

import java.io.IOException;

import com.adobe.internal.io.stream.OutputByteStream;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFIOException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFInvalidParameterException;

/**
 * ASMatrix encapsulates an immutable PDF coordinate transfomation matrix,
 * or CTM.
 *
 * @author dpond
 */
public class ASMatrix extends ASObject
{
	/**
	 * matrix used to describe affine transforms.
	 */
	protected double		a, b, c, d, x, y;
	protected ASMatrix		inverseTransform;

	/**
	 * returns the one and only identity matrix.
	 */
	static private ASMatrix gIdentityMatrix = new ASMatrix();
	
	/**
	 * Returns an identity matrix.
	 */
	static public ASMatrix createIdentityMatrix()
	{
		return gIdentityMatrix;
	}

	/**
	 * Constructs a new matrix and sets the entries
	 * to identity.
	 */
	private ASMatrix()
	{
		a = d = 1;
		b = c = x = y = 0;
		inverseTransform = this;
	}

	/**
	 * Constructs a new matrix of 2 columns, 3 rows (a, b, c, d, x, y).
	 * @param x  Translation in x
	 * @param y  Translation in y
	 */
	public ASMatrix(double a, double b, double c, double d, double x, double y)
	{
		this.a = a;
		this.b = b;
		this.c = c;
		this.d = d;
		this.x = x;
		this.y = y;
		inverseTransform = null;
	}

	/**
	 * Constructs a new matrix of 2 columns, 3 rows from an
	 * array of doubles with the ordering ( a, b, c, d, x, y ) where
	 * x gives in x and y gives translation in y.
	 */
	public ASMatrix(double[] array)
	{
		this.a = array[0];
		this.b = array[1];
		this.c = array[2];
		this.d = array[3];
		this.x = array[4];
		this.y = array[5];
		inverseTransform = null;
	}

	/**
	 * 
	 * Constructs a new matrix from another matrix (like clone)
	 * @param B  PDFAffineTransform
	 */
	public ASMatrix(ASMatrix B)
	{
		a = B.a;
		b = B.b;
		c = B.c;
		d = B.d;
		x = B.x;
		y = B.y;
		inverseTransform = B.inverseTransform;
	}

	public double geta()
	{
		return a;
	}

	public double getb()
	{
		return b;
	}

	public double getc()
	{
		return c;
	}

	public double getd()
	{
		return d;
	}

	public double getx()
	{
		return x;
	}

	public double gety()
	{
		return y;
	}

	public double[] getValues()
	{
		double[] matrix = new double[6];
		matrix[0] = a;
		matrix[1] = b;
		matrix[2] = c;
		matrix[3] = d;
		matrix[4] = x;
		matrix[5] = y;
		return matrix;
	}

	/**
	 * Test for equality
	 */
	public boolean equals(ASMatrix t)
	{
		return (t != null &&
				a == t.a && b == t.b && c == t.c && d == t.d &&
				x == t.x && y == t.y);
	}

	/**
	 * Test for Identity of the 4x4 matrix
	 * Exclusive of translation
	 */
	public boolean isIdentity()
	{
		return (a == 1 && d == 1 &&
				b == 0 && c == 0);
	}
	
	/**
	 * Test for Identity of the 6x6 matrix
	 * Inclusive of translation
	 */
	public boolean isIdentity6x6()
	{
		return (a == 1 && d == 1 &&
				b == 0 && c == 0 && x ==0 && y==0);
	}

	/**
	 * Test for invertability
	 */
	public boolean isInvertable()
	{
		double descrim = (b*c - a*d);
		return !Double.isNaN(descrim) && !Double.isInfinite(descrim) && (descrim != 0);
	}

	/**
	 * Concatentate the current transform with a rotate
	 * @param angle Amount of rotation in radians.
	 */
	public ASMatrix rotate(double angle)
	{
		double co = Math.cos(angle);
		double si = Math.sin(angle);
		return concat(new ASMatrix(co, si, -si, co, 0, 0));
	}

	/**
	 * Concatentate the current transform with a scale
	 */
	public ASMatrix scale(double sx, double sy)
	{
		return new ASMatrix(a*sx, b*sy, c*sx, d*sy, x*sx, y*sy);
	}

	public ASMatrix scale(double x)
	{
		return scale(x, x);
	}

	/**
	 * Premultiply the current transform with a scale
	 */
	public ASMatrix preScale(double sx, double sy)
	{
		return new ASMatrix(a*sx, b*sx, c*sy, d*sy, x, y);
	}

	public ASMatrix preScale(double x)
	{
		return preScale(x, x);
	}

	/**
	 * Concatenate the current transform with a translate
	 */
	public ASMatrix translate(double h, double v)
	{
		return new ASMatrix(a, b, c, d, x+h, y+v);
	}

	/**
	 * Premultiply the current transform with a translation
	 */
	public ASMatrix preMultiply(double h, double v)
	{
		return new ASMatrix(a, b, c, d, x + a * h + c * v, y + b * h + d * v);
	}

	/**
	 * Set an absolute translation
	 */
	public ASMatrix setTranslate(double h, double v)
	{
		return new ASMatrix(a, b, c, d, h, v);
	}

	/**
	 * Set an italic angle
	 */
	public ASMatrix setItalicDegrees(double angle)
	{
		if (angle == 0)
			return this;
		else
			return new ASMatrix(a, b, Math.sin(angle * Deg2Rad)*a + c, d, x, y);
	}
	private static final double Deg2Rad = Math.PI/180;

	/**
	 * Multiply a vector by our matrix.
	 * @param px Along with py: input vector
	 * @param py  Along with px: input vector
	 * @param output  output vector -- must be at least size 2 (x is index 0, y is index 1)
	 */
	public void mult(double px, double py, double output[])
	{
		output[0] = a * px  +  c * py  +  x;
		output[1] = b * px  +  d * py  +  y;
	}
	
	public double multGetX(double px, double py)
	{
		return a * px  +  c * py  +  x;
	}
	
	public double multGetY(double px, double py)
	{
		return b * px  +  d * py  +  y;
	}

	/**
	 * (left) multiply new transform with this (computes X * A, where X is the current transform)
	 * @param A new transform.
	 */
	public ASMatrix concat(ASMatrix A)
	{
		//  now the usual 3x3 matrix multiplication
		double newa = a * A.a  +  b * A.c;
		double newc = c * A.a  +  d * A.c;
		double newx = x * A.a  +  y * A.c  +  A.x;
		double newb = a * A.b  +  b * A.d;
		double newd = c * A.b  +  d * A.d;
		double newy = x * A.b  +  y * A.d  +  A.y;
		return new ASMatrix(newa, newb, newc, newd, newx, newy);
	}

	/**
	 * Find the matrix, B, that has the property:  B * X = identity.
	 * @exception PDFInvalidParameterException   If the matrix does not have an inverse.
	 *
	 */
	public ASMatrix getInverse()
		throws PDFInvalidParameterException
	{
		if (inverseTransform == null) {
			double descrim = 1 / (b*c - a*d);

			double newa = -d * descrim;
			double newb =  b * descrim;
			double newc =  c * descrim;
			double newd = -a * descrim;
			double newx = -(x * newa + y * newc);
			double newy = -(x * newb + y * newd);

			if (Double.isNaN(newa) || Double.isInfinite(newa)
				|| Double.isNaN(newb) || Double.isInfinite(newb)
				|| Double.isNaN(newc) || Double.isInfinite(newc)
				|| Double.isNaN(newd) || Double.isInfinite(newd)
				|| Double.isNaN(newx) || Double.isInfinite(newx)
				|| Double.isNaN(newy) || Double.isInfinite(newy)) {
				throw new PDFInvalidParameterException("Singular matrix passed to invert: " + a + " " + b + " " + c + " " + d + " " + x + " " + y);
			} else {
				inverseTransform = new ASMatrix(newa, newb, newc, newd, newx, newy);
				inverseTransform.inverseTransform = this;
			}
		}
		return inverseTransform;
	}

	/**
	 * Converts this object to a String representation.
	 */
	@Override
	public String toString()
	{
		return "[ " + a + " " + b + " " + c + " " + d + " " + x + " " + y + " ]";
	}

	/**
	 * Writes the ASHexString in to the given OutputStream in the format expected by the PDF Spec.
	 * @see ASString
	 * @param outputByteStream OutputStream to write to.
	 * @throws PDFIOException
	 */
	@Override
	public void write(OutputByteStream outputByteStream)
		throws PDFIOException
	{
		try {
			outputByteStream.write(this.toString().getBytes());
		} catch (IOException e) {
			throw new PDFIOException(e);
		}
	}

	public ASMatrix getUnitRotationMatrix() {
		// Get the rotation part of the matrix, i.e remove the scaling effect and keep only the rotation effect.
		double xDenominator = Math.sqrt(this.a*this.a + this.c*this.c);
		double yDenominator = Math.sqrt(this.b*this.b + this.d*this.d);

		if(xDenominator != 0 && yDenominator != 0)
			return new ASMatrix(this.a/xDenominator, this.b/yDenominator, this.c/xDenominator, this.d/yDenominator, 0, 0);

		return this;
	}
}
