/* ****************************************************************************
 *
 *	File: ASQuad.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2004 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/
package com.adobe.internal.pdftoolkit.core.types;

import java.io.IOException;

import com.adobe.internal.io.stream.OutputByteStream;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFIOException;

/**
 * ASQuad represents a convex quadrilateral. An ASQuad is typically used to
 * represent a bounding box (a rectangle) which has undergone an affine
 * transformation (e.g., scaled, rotated, skewed, etc.). In this sense,
 * the coordinate tags (p1, p2, p3, p4) do not necessarily correspond
 * to the actual relative location of the coordinates. Instead, they
 * correspond to the logical corners of the object encased in the quad.
 *
 * For instance, a word has a natural bounding box (rectangle) with a well-
 * defined upper-left, upper-right, lower-left, and lower-right corners.
 * When rotated 45 degrees, the coordinates of the quad rotate with the
 * word and the tags are considered affixed to the bounding box of the
 * word.
 */
public class ASQuad extends ASObject
{
	private ASCoordinate p1;	// Corresponds to the lower-left corner of an unrotated rectangle
	private ASCoordinate p2;	// Corresponds to the lower-right corner of an unrotated rectangle
	private ASCoordinate p3;	// Corresponds to the upper-right corner of an unrotated rectangle
	private ASCoordinate p4;	// Corresponds to the upper-left corner of an unrotated rectangle

	/**
	 * @param p1	lower-left corner relative to the object enclosed by the quad
	 * @param p2	lower-right corner relative to the object enclosed by the quad
	 * @param p3	upper-right corner relative to the object enclosed by the quad
	 * @param p4	upper-left corner relative to the object enclosed by the quad
	 */
	public ASQuad(ASCoordinate p1, ASCoordinate p2, ASCoordinate p3, ASCoordinate p4)
	{
		this.p1 = p1;
		this.p2 = p2;
		this.p3 = p3;
		this.p4 = p4;
	}

	/**
	 * Construct a quad from a rectangle
	 *
	 * @param rect
	 */
	public ASQuad(ASRectangle rect)
	{
		p1 = new ASCoordinate(rect.ll());
		p2 = new ASCoordinate(rect.lr());
		p3 = new ASCoordinate(rect.ur());
		p4 = new ASCoordinate(rect.ul());
	}

	/**
	 * Copy constructor
	 *
	 * @param quad
	 */
	public ASQuad(ASQuad quad)
	{
		this.p1 = new ASCoordinate(quad.p1);
		this.p2 = new ASCoordinate(quad.p2);
		this.p3 = new ASCoordinate(quad.p3);
		this.p4 = new ASCoordinate(quad.p4);
	}

	public ASCoordinate p1() {return p1;}
	public ASCoordinate p2() {return p2;}
	public ASCoordinate p3() {return p3;}
	public ASCoordinate p4() {return p4;}

	public ASQuad transform(ASMatrix ctm)
	{
		ASCoordinate new_p1 = p1.transform(ctm);
		ASCoordinate new_p2 = p2.transform(ctm);
		ASCoordinate new_p3 = p3.transform(ctm);
		ASCoordinate new_p4 = p4.transform(ctm);
		return new ASQuad(new_p1, new_p2, new_p3, new_p4);
	}

	
	@Override
	public String toString()
	{
		return "[ " + p1.toString() + " " + p2.toString() + " " + p3.toString() + " " + p4.toString() + " ]";
	}

	/**
	 * 
	 * Writes the ASHexString in to the given OutputStream in the format expected by the PDF Spec.
	 * @see ASString
	 * @param outputByteStream OutputByteStream to write to.
	 * @throws PDFIOException
	 */
	@Override
	public void write(OutputByteStream outputByteStream)
		throws PDFIOException
	{
		try {
			outputByteStream.write(toString().getBytes());
		} catch (IOException e) {
			throw new PDFIOException(e);
		}
	}

	public double getMinX() {
		double[] xValues = new double[] {p1.x(), p2.x(), p3.x(), p4.x()};
		double minX = p1.x();
		for (int i = 1; i < xValues.length; i++) {
			if(xValues[i] < minX)
				minX = xValues[i];
		}
		return minX;
	}

	public double getMaxX() {
		double[] xValues = new double[] {p1.x(), p2.x(), p3.x(), p4.x()};
		double maxX = p1.x();
		for (int i = 1; i < xValues.length; i++) {
			if(xValues[i] > maxX)
				maxX = xValues[i];
		}
		return maxX;
	}

	public double getMinY() {
		double[] yValues = new double[] {p1.y(), p2.y(), p3.y(), p4.y()};
		double minY = p1.y();
		for (int i = 1; i < yValues.length; i++) {
			if(yValues[i] < minY)
				minY = yValues[i];
		}
		return minY;
	}

	public double getMaxY() {
		double[] yValues = new double[] {p1.y(), p2.y(), p3.y(), p4.y()};
		double maxY = p1.y();
		for (int i = 1; i < yValues.length; i++) {
			if(yValues[i] > maxY)
				maxY = yValues[i];
		}
		return maxY;
	}
}
