/* ****************************************************************************
 *
 *	File: ASRectangle.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2004 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/
package com.adobe.internal.pdftoolkit.core.types;

import java.awt.geom.Rectangle2D;
import java.io.IOException;

import com.adobe.internal.io.stream.OutputByteStream;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFIOException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFInvalidDocumentException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFInvalidParameterException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFSecurityException;

/**
 * ASRectangle represents an non-COS PDF rectangle.
 */
public class ASRectangle extends ASObject
{
	private double left;
	private double bottom;
	private double right;
	private double top;

	/**
	 * @param ll	lower-left corner of the rectangle
	 * @param ur	upper-right corner of the rectangle
	 */
	public ASRectangle(ASCoordinate ll, ASCoordinate ur)
	{
		this.left = ll.x();
		this.bottom = ll.y();
		this.right = ur.x();
		this.top = ur.y();
	}

	/**
	 * @param left		x coordinate of the left side of the rectangle
	 * @param bottom	y coordinate of the bottom side of the rectangle
	 * @param right		x coordinate of the right side of the rectangle
	 * @param top		y coordinate of the top side of the rectangle
	 */
	public ASRectangle(double left, double bottom, double right, double top)
	{
		this.left = left;
		this.bottom = bottom;
		this.right = right;
		this.top = top;
	}

	/**
	 * @param coords	an array of the coordinates in the form [left, bottom, right, top]
	*/
	public ASRectangle(double[] coords)
		throws PDFInvalidParameterException
	{
		if (coords.length != 4)
			throw new PDFInvalidParameterException("attempt to construct ASRectangle with invalid parameters");
		this.left = coords[0];
		this.bottom = coords[1];
		this.right = coords[2];
		this.top = coords[3];
	}

	/**
	 * Copy constructor.
	 *
	 * @param rect
	 */
	public ASRectangle(ASRectangle rect)
	{
		this.left = rect.left;
		this.bottom = rect.bottom;
		this.right = rect.right;
		this.top = rect.top;
	}

	
	@Override
	public String toString()
	{
		return "[ " + left + " " + bottom + " " + right + " " + top + " ]";
	}

	/**
	 * Writes the ASRectangle in to the given OutputStream in the format expected by the PDF Spec.
	 *
	 * @param outputByteStream OutputByteStream to write to.
	 * @throws PDFIOException exception from OutputStream
	 */
	@Override
	public void write(OutputByteStream outputByteStream)
		throws PDFIOException
	{
		try {
			outputByteStream.write(toString().getBytes());
		} catch (IOException e) {
			throw new PDFIOException(e);
		}
	}

	/**
	 * Get the coordinate values of the rectangle's corners
	 *
	 * @return	an array of the format [left, bottom, right, top]
	 */
	public double[] getValues() {
		double vals[] = new double[4];
		vals[0] = left;
		vals[1] = bottom;
		vals[2] = right;
		vals[3] = top;
		return vals;
	}

	public ASCoordinate ll() { return new ASCoordinate(left, bottom); }
	public ASCoordinate lr() { return new ASCoordinate(right, bottom); }
	public ASCoordinate ur() { return new ASCoordinate(right, top); }
	public ASCoordinate ul() { return new ASCoordinate(left, top); }

	public double left()	{ return left; }
	public double bottom()	{ return bottom; }
	public double right()	{ return right; }
	public double top()		{ return top; }

	public double width()	{ return Math.abs(right - left); }
	public double height()	{ return Math.abs(top - bottom); }

	public ASRectangle transform(ASMatrix ctm)
	{
		ASCoordinate ll = ll().transform(ctm);
		ASCoordinate ur = ur().transform(ctm);
		return new ASRectangle(ll, ur);
	}
	
	public boolean hasNonZeroDimensions()
	{
		if ((left == 0) && (bottom == 0) && (right == 0) && (top == 0))
			return false;
		else if (height() == 0 || width() == 0)
			return false;
		return true;
	}
	
	public boolean hasNonZeroDimensions(double epsilon)
	{
		if ((Math.abs(left) < epsilon) && (Math.abs(bottom) < epsilon) && (Math.abs(right) < epsilon) && (Math.abs(top) < epsilon))
			return false;
		else if (height() < epsilon || width() < epsilon)
			return false;
		return true;
	}	
	
	/**
	 * Returns true if the supplied PDFRectangle is contained in the current PDFRectangle.
	 * @param rect
	 * @return boolean
	 * @throws PDFSecurityException 
	 * @throws PDFIOException 
	 * @throws PDFInvalidDocumentException 
	 */
	public boolean contains(ASRectangle rect)
	throws PDFInvalidDocumentException, PDFIOException, PDFSecurityException
	{
		boolean isRectContained = false;
		
		Rectangle2D.Double currentRectangle = new Rectangle2D.Double(this.ll().x(), this.ll().y(), this.ur().x() - this.ll().x(), this.ur().y() - this.ll().y());    	
		
		isRectContained = currentRectangle.contains(rect.ll().x(), rect.ll().y(), rect.ur().x() - rect.ll().x(), rect.ur().y() - rect.ll().y());
		
		return isRectContained;
	}
	
	/**
	 * Creates a new ASRectangle with normalized coordinates.  The new ASRectangle will have
	 * the following properties:
	 * <ul><li> left <= right
	 * <li> bottomw <= top
	 * </ul>
	 * @return a new ASRectangle with normalized coordinates
	 * @throws PDFInvalidDocumentException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public ASRectangle normalized() 
	throws PDFInvalidDocumentException, PDFIOException, PDFSecurityException
	{
		double newVals[] = new double[4];
		newVals[0] = Math.min(left(), right());
		newVals[1] = Math.min(bottom(), top());
		newVals[2] = Math.max(left(), right());
		newVals[3] = Math.max(bottom(), top());
		return new ASRectangle(newVals[0], newVals[1], newVals[2], newVals[3]);
	}
	

}
