/* ****************************************************************************
 *
 *	File: ASString.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2003-2006 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/

package com.adobe.internal.pdftoolkit.core.types;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.charset.CharacterCodingException;
import java.nio.charset.CharsetEncoder;

import com.adobe.internal.io.stream.OutputByteStream;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFIOException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFInvalidParameterException;
import com.adobe.internal.pdftoolkit.core.util.PDFDocEncoding;
import com.adobe.internal.pdftoolkit.core.util.UTF8Support;
import com.adobe.internal.pdftoolkit.core.util.Utility;

/**
 * Represents a pdf string object. That is, a series of bytes that do NOT have an associated encoding.
 *
 */
public class ASString extends ASObject implements Comparable
{
	private void init(byte[] bytes)
	{
		this.bytes = bytes;
		this.hash = generateHash(bytes, 0, bytes.length);
	}

	public ASString(byte[] bytes)
	{
		init(bytes);
	}

	/**
	 * Creates an ASString that contains UTF-8. Note that the fact that
	 * this is UTF-8 is promptly forgotten by the instance.
	 */
	public ASString(String javaString)
	{
		init(UTF8Support.toUTF8(javaString));
	}

	/**
	 * Creates an ASString that contains the bytes received
	 * by encoding the javaString with the provided cse. Note that the
	 * encoding is promptly forgotten by the instance.
	 * @throws PDFInvalidParameterException
	 */
	public ASString(String javaString, CharsetEncoder cse)
		throws PDFInvalidParameterException
	{
		ByteBuffer nameBuf = null;
		CharBuffer charBuf = CharBuffer.wrap(javaString.toCharArray());
		try {
			nameBuf = cse.encode(charBuf);
		} catch (CharacterCodingException e) {
			throw new PDFInvalidParameterException("cannot encode string", e);
		}
		byte[] trimmedBytes = new byte[nameBuf.limit()];
		nameBuf.get(trimmedBytes, 0, nameBuf.limit());
		init(trimmedBytes);
	}

	/**
	 * Construct an <code>ASString</code> from an array of integers.
	 * @param numbers numbers to use in constructing the ASString
	 * @param bytesToUse number of bytes to use from each number starting with the LSB. Must be from 1 to 4.
	 * @param bytesToPad number of bytes to pad in front of each number beyond the number of bytes taken from each number
	 * @throws PDFInvalidParameterException if the <i>bytesToUse</i> is not between 1 and 4
	 */
	public ASString(int[] numbers, int bytesToUse, int bytesToPad) 
	throws PDFInvalidParameterException
	{
		this(convertIntegersToBytes(numbers, 0, numbers.length-1, bytesToUse, bytesToPad));
	}
	
	public ASString(int[] numbers, int startIndex, int endIndex, int bytesToUse, int bytesToPad)
	throws PDFInvalidParameterException
	{
		this(convertIntegersToBytes(numbers, startIndex, endIndex, bytesToUse, bytesToPad));
	}
	
	protected static byte[] convertIntegersToBytes(int[] numbers, int startIndex, int endIndex, int bytesToUse, int bytesToPad) 
	throws PDFInvalidParameterException
	{
		if (bytesToUse < 1 || bytesToUse > 4)
		{
			throw new PDFInvalidParameterException("Bytes to use must be from 1 to 4.");
		}
		
		int position = 0;
		int length = endIndex - startIndex + 1;
		byte[] bytes = new byte[length * (bytesToUse + bytesToPad)];
		
		for (int i = startIndex; i <= endIndex; i++)
		{
			position = (i-startIndex) * (bytesToPad + bytesToUse) + bytesToPad;
			for (int digit = bytesToUse; digit > 0; digit--)
			{
				bytes[position++] = (byte) ((numbers[i] >>> ((digit - 1) * 8)) & 0xff);
			}
		}
		return bytes;
	}
	
	/**
	 * Get the raw bytes associated with this ASString. The client must NOT alter the returned array.
	 */
	public byte[] getBytes()
	{
		return bytes;
	}

	/**
	 * Try to get an array of chars associated with this ASString. Note that, because ASString has no encoding,
	 * this just uses a series of heuristics to get the chars. No guarantees are made regarding the returned value
	 * other than "we tried our best". It may change over time.
	 */
	public char[] getChars()
	{
		if (chars == null) {
			chars = new char[bytes.length / 2];
			for (int i = 0; i < chars.length; i++)
				chars[i] = getChar(i * 2);
		}
		return chars;
	}
	
	/**
	 * Try to get a chars associated with a byte in this ASString. Note that, because ASString has no encoding,
	 * this just uses a series of heuristics to get the chars. No guarantees are made regarding the returned value
	 * other than "we tried our best". It may change over time.
	 */
	public char getChar(int index)
	{
		return (char)(((bytes[index] & 0xFF) << 8) + (bytes[index + 1] & 0xFF));
	}
	
	/**
	 * Try to get a chars associated with a byte in this ASString. Note that, because ASString has no encoding,
	 * this just uses a series of heuristics to get the chars. No guarantees are made regarding the returned value
	 * other than "we tried our best". It may change over time.
	 */
	public String asString()
	{
		return PDFDocEncoding.toAcrobatString(bytes);
	}

	/**
	 * Attempts to get a unicode representation for this unencoded string. No guarantees
	 * are made about what is returned.
	 */
	@Override
	public String toString()
	{
		return asString();
	}

	/* (non-Javadoc)
	 * @see java.lang.Comparable#compareTo(java.lang.Object)
	 */
	public int compareTo(Object anotherString)
	{
		if (!(anotherString instanceof ASString))
			throw new RuntimeException("Expected ASString");

		byte v1[] = this.getBytes();
		byte v2[] = ((ASString) anotherString).getBytes();
		int len1 = v1.length;
		int len2 = v2.length;
		int n = Math.min(len1, len2);
		int k = 0;
		while (k < n) {
			if (v1[k] != v2[k]) {
				return (v1[k]&0xff) - (v2[k]&0xff);
			}
			k++;
		}
		// If they are the same so far then the longer one is "greater".
		return len1 - len2;
	}

	public boolean equals(ASString anotherString)
	{
		return (this.compareTo(anotherString) == 0);
	}
	
	@Override
	public boolean equals(Object anotherString)
	{
		if (this == anotherString)
			return true;
		if (anotherString instanceof ASString)
			return (this.compareTo(anotherString) == 0);
		return false;
	}

	@Override
	public int hashCode()
	{
		return this.hash;
	}

	private static final int generateHash(byte[] b, int start, int length)
	{
		int hash = 0;
		for (int i = 0; i < length; i++)
		{
			hash = 	hash * 31 + b[i + start];
		}
		return hash;
	}

	public int indexOf(byte[] key)
	{
		if (bytes == null)
			return -1;
		int[] next = Utility.ComputeKMPNextArray(key);
		return Utility.KMPFindFirst(key, next, bytes);
	}

	public int lastIndexOf(int target)
	{
		int byteInd;
		for (byteInd = bytes.length - 1; byteInd >= 0; byteInd--) {
			if (bytes[byteInd] == target) {
				return byteInd;
			}
		}
		return -1;
	}

	public ASString substring(int begin)
	{
		return substring(begin, bytes.length);
	}


	public ASString substring(int begin, int end)
	{
		if (begin < 0)
			begin = 0;
		if (end > bytes.length)
			end = bytes.length;
		if (begin >= end)
			return null;
		byte[] subStr = new byte[end - begin];
		System.arraycopy(bytes, begin, subStr, 0, end - begin);
		return new ASString(subStr);
	}

	public ASString concat(ASString addStr)
	{
		if ((addStr == null) || (addStr.getBytes().length == 0))
			return this;
		byte[] addBytes = addStr.getBytes();
		byte[] concatStr = new byte[bytes.length + addBytes.length];
		System.arraycopy(bytes, 0, concatStr, 0, bytes.length);
		System.arraycopy(addBytes, 0, concatStr, bytes.length, addBytes.length);
		return new ASString(concatStr);
	}

	/**
	 * Writes the ASString in to the given OutputStream in the format expected by the PDF Spec.
	 * @see ASHexString
	 * @param outputByteStream OutputByteStream to write to.
	 * @throws PDFIOException
	 */
	@Override
	public void write(OutputByteStream  outputByteStream)
		throws PDFIOException
	{
		try {
			outputByteStream.write('(');
			byte[] outValue = this.getBytes();
			for (int i = 0; i < outValue.length; i++) {
				byte b = outValue[i];
				if (b == '\r' && (i + 1) < outValue.length && outValue[i + 1] == '\n') {
					outputByteStream.write(b);
					i++;
				}
				else if (b == '\n') {
					outputByteStream.write('\\');
					outputByteStream.write('n');
				}
				else if (b == '\r') {
					outputByteStream.write('\\');
					outputByteStream.write('r');
				}
				else if (b == '(') {
					outputByteStream.write('\\');
					outputByteStream.write('(');
				}
				else if (b == ')') {
					outputByteStream.write('\\');
					outputByteStream.write(')');
				}
				else if (b == '\\') {
					outputByteStream.write('\\');
					outputByteStream.write('\\');
				}
				else {
					outputByteStream.write(b);
				}
			}
			outputByteStream.write(')');
		} catch (IOException e) {
			throw new PDFIOException(e);
		}
	}
    
	private byte[] bytes;
	private char[] chars;
	private int hash;
}
