/* ****************************************************************************
 *
 *	File: ASByteArray.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2004-2006 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/
package com.adobe.internal.pdftoolkit.core.util;

import java.io.UnsupportedEncodingException;

import com.adobe.internal.pdftoolkit.core.exceptions.PDFParseException;
import com.adobe.internal.pdftoolkit.core.types.ASName;

/**
 * This class provides many utility methods over bytes.
 */
public final class ByteOps {
	
	private ByteOps(){}
	
	/**
	 * Whitespace character lookup table. Whitespace characters are defined
	 * in section 3.1.1 of the PDF Reference Manual version 1.4. Note that
	 * the ordering of the entries takes into account that the byte is
	 * a signed type.
	 */
	private static final boolean[] WS = {
		false,	false,	false,	false,	false,	false,	false,	false,	// 128
		false,	false,	false,	false,	false,	false,	false,	false,	// 136
		false,	false,	false,	false,	false,	false,	false,	false,	// 144
		false,	false,	false,	false,	false,	false,	false,	false,	// 152
		false,	false,	false,	false,	false,	false,	false,	false,	// 160
		false,	false,	false,	false,	false,	false,	false,	false,	// 168
		false,	false,	false,	false,	false,	false,	false,	false,	// 176
		false,	false,	false,	false,	false,	false,	false,	false,	// 184
		false,	false,	false,	false,	false,	false,	false,	false,	// 192
		false,	false,	false,	false,	false,	false,	false,	false,	// 200
		false,	false,	false,	false,	false,	false,	false,	false,	// 208
		false,	false,	false,	false,	false,	false,	false,	false,	// 216
		false,	false,	false,	false,	false,	false,	false,	false,	// 224
		false,	false,	false,	false,	false,	false,	false,	false,	// 232
		false,	false,	false,	false,	false,	false,	false,	false,	// 240
		false,	false,	false,	false,	false,	false,	false,	false,	// 248
		true,	false,	false,	false,	false,	false,	false,	false,	//   0
		false,	true,	true,	false,	true,	true,	false,	false,	//   8
		false,	false,	false,	false,	false,	false,	false,	false,	//  16
		false,	false,	false,	false,	false,	false,	false,	false,	//  24
		true,	false,	false,	false,	false,	false,	false,	false,	//  32
		false,	false,	false,	false,	false,	false,	false,	false,	//  40
		false,	false,	false,	false,	false,	false,	false,	false,	//  48
		false,	false,	false,	false,	false,	false,	false,	false,	//  56
		false,	false,	false,	false,	false,	false,	false,	false,	//  64
		false,	false,	false,	false,	false,	false,	false,	false,	//  72
		false,	false,	false,	false,	false,	false,	false,	false,	//  80
		false,	false,	false,	false,	false,	false,	false,	false,	//  88
		false,	false,	false,	false,	false,	false,	false,	false,	//  96
		false,	false,	false,	false,	false,	false,	false,	false,	// 104
		false,	false,	false,	false,	false,	false,	false,	false,	// 112
		false,	false,	false,	false,	false,	false,	false,	false	// 120
	};
	
	/**
	 * Regular character lookup table. Regular characters are defined
	 * in section 3.1.1 of the PDF Reference Manual version 1.4. Note
	 * that the ordering of the entries takes into account that the
	 * byte is a signed type.
	 */
	private static final boolean[] REG = {
		true,	true,	true,	true,	true,	true,	true,	true,	// 128
		true,	true,	true,	true,	true,	true,	true,	true,	// 136
		true,	true,	true,	true,	true,	true,	true,	true,	// 144
		true,	true,	true,	true,	true,	true,	true,	true,	// 152
		true,	true,	true,	true,	true,	true,	true,	true,	// 160
		true,	true,	true,	true,	true,	true,	true,	true,	// 168
		true,	true,	true,	true,	true,	true,	true,	true,	// 176
		true,	true,	true,	true,	true,	true,	true,	true,	// 184
		true,	true,	true,	true,	true,	true,	true,	true,	// 192
		true,	true,	true,	true,	true,	true,	true,	true,	// 200
		true,	true,	true,	true,	true,	true,	true,	true,	// 208
		true,	true,	true,	true,	true,	true,	true,	true,	// 216
		true,	true,	true,	true,	true,	true,	true,	true,	// 224
		true,	true,	true,	true,	true,	true,	true,	true,	// 232
		true,	true,	true,	true,	true,	true,	true,	true,	// 240
		true,	true,	true,	true,	true,	true,	true,	true,	// 248
		false,	true,	true,	true,	true,	true,	true,	true,	//   0
		true,	false,	false,	true,	false,	false,	true,	true,	//   8
		true,	true,	true,	true,	true,	true,	true,	true,	//  16
		true,	true,	true,	true,	true,	true,	true,	true,	//  24
		false,	true,	true,	true,	true,	false,	true,	true,	//  32
		false,	false,	true,	true,	true,	true,	true,	false,	//  40
		true,	true,	true,	true,	true,	true,	true,	true,	//  48
		true,	true,	true,	true,	false,	true,	false,	true,	//  56
		true,	true,	true,	true,	true,	true,	true,	true,	//  64
		true,	true,	true,	true,	true,	true,	true,	true,	//  72
		true,	true,	true,	true,	true,	true,	true,	true,	//  80
		true,	true,	true,	false,	true,	false,	true,	true,	//  88
		true,	true,	true,	true,	true,	true,	true,	true,	//  96
		true,	true,	true,	true,	true,	true,	true,	true,	// 104
		true,	true,	true,	true,	true,	true,	true,	true,	// 112
		true,	true,	true,	false,	true,	false,	true,	true	// 120
	};
	
	/**
	 * String literal character lookup table. String literal characters
	 * are defined in section 3.2.3 of the PDF Reference Manual version 1.4.
	 * Note that the ordering of the entries takes into account that the
	 * byte is a signed type.
	 */
	private static final boolean[] LIT = {
		true,	true,	true,	true,	true,	true,	true,	true,	// 128
		true,	true,	true,	true,	true,	true,	true,	true,	// 136
		true,	true,	true,	true,	true,	true,	true,	true,	// 144
		true,	true,	true,	true,	true,	true,	true,	true,	// 152
		true,	true,	true,	true,	true,	true,	true,	true,	// 160
		true,	true,	true,	true,	true,	true,	true,	true,	// 168
		true,	true,	true,	true,	true,	true,	true,	true,	// 176
		true,	true,	true,	true,	true,	true,	true,	true,	// 184
		true,	true,	true,	true,	true,	true,	true,	true,	// 192
		true,	true,	true,	true,	true,	true,	true,	true,	// 200
		true,	true,	true,	true,	true,	true,	true,	true,	// 208
		true,	true,	true,	true,	true,	true,	true,	true,	// 216
		true,	true,	true,	true,	true,	true,	true,	true,	// 224
		true,	true,	true,	true,	true,	true,	true,	true,	// 232
		true,	true,	true,	true,	true,	true,	true,	true,	// 240
		true,	true,	true,	true,	true,	true,	true,	true,	// 248
		true,	true,	true,	true,	true,	true,	true,	true,	//   0
		true,	true,	true,	true,	true,	true,	true,	true,	//   8
		true,	true,	true,	true,	true,	true,	true,	true,	//  16
		true,	true,	true,	true,	true,	true,	true,	true,	//  24
		true,	true,	true,	true,	true,	true,	true,	true,	//  32
		false,	false,	true,	true,	true,	true,	true,	true,	//  40
		true,	true,	true,	true,	true,	true,	true,	true,	//  48
		true,	true,	true,	true,	true,	true,	true,	true,	//  56
		true,	true,	true,	true,	true,	true,	true,	true,	//  64
		true,	true,	true,	true,	true,	true,	true,	true,	//  72
		true,	true,	true,	true,	true,	true,	true,	true,	//  80
		true,	true,	true,	true,	false,	true,	true,	true,	//  88
		true,	true,	true,	true,	true,	true,	true,	true,	//  96
		true,	true,	true,	true,	true,	true,	true,	true,	// 104
		true,	true,	true,	true,	true,	true,	true,	true,	// 112
		true,	true,	true,	true,	true,	true,	true,	true	// 120
	};
	
	/**
	 * Decimal digit character lookup table. Note that the ordering of
	 * the entries takes into account that the byte is a signed type.
	 */
	private static final boolean[] DEC_DIGIT = {
		false,	false,	false,	false,	false,	false,	false,	false,	// 128
		false,	false,	false,	false,	false,	false,	false,	false,	// 136
		false,	false,	false,	false,	false,	false,	false,	false,	// 144
		false,	false,	false,	false,	false,	false,	false,	false,	// 152
		false,	false,	false,	false,	false,	false,	false,	false,	// 160
		false,	false,	false,	false,	false,	false,	false,	false,	// 168
		false,	false,	false,	false,	false,	false,	false,	false,	// 176
		false,	false,	false,	false,	false,	false,	false,	false,	// 184
		false,	false,	false,	false,	false,	false,	false,	false,	// 192
		false,	false,	false,	false,	false,	false,	false,	false,	// 200
		false,	false,	false,	false,	false,	false,	false,	false,	// 208
		false,	false,	false,	false,	false,	false,	false,	false,	// 216
		false,	false,	false,	false,	false,	false,	false,	false,	// 224
		false,	false,	false,	false,	false,	false,	false,	false,	// 232
		false,	false,	false,	false,	false,	false,	false,	false,	// 240
		false,	false,	false,	false,	false,	false,	false,	false,	// 248
		false,	false,	false,	false,	false,	false,	false,	false,	//   0
		false,	false,	false,	false,	false,	false,	false,	false,	//   8
		false,	false,	false,	false,	false,	false,	false,	false,	//  16
		false,	false,	false,	false,	false,	false,	false,	false,	//  24
		false,	false,	false,	false,	false,	false,	false,	false,	//  32
		false,	false,	false,	false,	false,	false,	false,	false,	//  40
		true,	true,	true,	true,	true,	true,	true,	true,	//  48
		true,	true,	false,	false,	false,	false,	false,	false,	//  56
		false,	false,	false,	false,	false,	false,	false,	false,	//  64
		false,	false,	false,	false,	false,	false,	false,	false,	//  72
		false,	false,	false,	false,	false,	false,	false,	false,	//  80
		false,	false,	false,	false,	false,	false,	false,	false,	//  88
		false,	false,	false,	false,	false,	false,	false,	false,	//  96
		false,	false,	false,	false,	false,	false,	false,	false,	// 104
		false,	false,	false,	false,	false,	false,	false,	false,	// 112
		false,	false,	false,	false,	false,	false,	false,	false	// 120
	};
	
	/**
	 * Hexidecimal digit character lookup table. Note that the ordering
	 * of the entries takes into account that the byte is a signed type.
	 */
	private static final boolean[] HEX_DIGIT = {
		false,	false,	false,	false,	false,	false,	false,	false,	// 128
		false,	false,	false,	false,	false,	false,	false,	false,	// 136
		false,	false,	false,	false,	false,	false,	false,	false,	// 144
		false,	false,	false,	false,	false,	false,	false,	false,	// 152
		false,	false,	false,	false,	false,	false,	false,	false,	// 160
		false,	false,	false,	false,	false,	false,	false,	false,	// 168
		false,	false,	false,	false,	false,	false,	false,	false,	// 176
		false,	false,	false,	false,	false,	false,	false,	false,	// 184
		false,	false,	false,	false,	false,	false,	false,	false,	// 192
		false,	false,	false,	false,	false,	false,	false,	false,	// 200
		false,	false,	false,	false,	false,	false,	false,	false,	// 208
		false,	false,	false,	false,	false,	false,	false,	false,	// 216
		false,	false,	false,	false,	false,	false,	false,	false,	// 224
		false,	false,	false,	false,	false,	false,	false,	false,	// 232
		false,	false,	false,	false,	false,	false,	false,	false,	// 240
		false,	false,	false,	false,	false,	false,	false,	false,	// 248
		false,	false,	false,	false,	false,	false,	false,	false,	//   0
		false,	false,	false,	false,	false,	false,	false,	false,	//   8
		false,	false,	false,	false,	false,	false,	false,	false,	//  16
		false,	false,	false,	false,	false,	false,	false,	false,	//  24
		false,	false,	false,	false,	false,	false,	false,	false,	//  32
		false,	false,	false,	false,	false,	false,	false,	false,	//  40
		true,	true,	true,	true,	true,	true,	true,	true,	//  48
		true,	true,	false,	false,	false,	false,	false,	false,	//  56
		false,	true,	true,	true,	true,	true,	true,	false,	//  64
		false,	false,	false,	false,	false,	false,	false,	false,	//  72
		false,	false,	false,	false,	false,	false,	false,	false,	//  80
		false,	false,	false,	false,	false,	false,	false,	false,	//  88
		false,	true,	true,	true,	true,	true,	true,	false,	//  96
		false,	false,	false,	false,	false,	false,	false,	false,	// 104
		false,	false,	false,	false,	false,	false,	false,	false,	// 112
		false,	false,	false,	false,	false,	false,	false,	false	// 120
	};
	
	/**
	 * Octal digit character lookup table. Note that the ordering of
	 * the entries takes into account that the byte is a signed type.
	 */
	private static final boolean[] OCT_DIGIT = {
		false,	false,	false,	false,	false,	false,	false,	false,	// 128
		false,	false,	false,	false,	false,	false,	false,	false,	// 136
		false,	false,	false,	false,	false,	false,	false,	false,	// 144
		false,	false,	false,	false,	false,	false,	false,	false,	// 152
		false,	false,	false,	false,	false,	false,	false,	false,	// 160
		false,	false,	false,	false,	false,	false,	false,	false,	// 168
		false,	false,	false,	false,	false,	false,	false,	false,	// 176
		false,	false,	false,	false,	false,	false,	false,	false,	// 184
		false,	false,	false,	false,	false,	false,	false,	false,	// 192
		false,	false,	false,	false,	false,	false,	false,	false,	// 200
		false,	false,	false,	false,	false,	false,	false,	false,	// 208
		false,	false,	false,	false,	false,	false,	false,	false,	// 216
		false,	false,	false,	false,	false,	false,	false,	false,	// 224
		false,	false,	false,	false,	false,	false,	false,	false,	// 232
		false,	false,	false,	false,	false,	false,	false,	false,	// 240
		false,	false,	false,	false,	false,	false,	false,	false,	// 248
		false,	false,	false,	false,	false,	false,	false,	false,	//   0
		false,	false,	false,	false,	false,	false,	false,	false,	//   8
		false,	false,	false,	false,	false,	false,	false,	false,	//  16
		false,	false,	false,	false,	false,	false,	false,	false,	//  24
		false,	false,	false,	false,	false,	false,	false,	false,	//  32
		false,	false,	false,	false,	false,	false,	false,	false,	//  40
		true,	true,	true,	true,	true,	true,	true,	true,	//  48
		false,	false,	false,	false,	false,	false,	false,	false,	//  56
		false,	false,	false,	false,	false,	false,	false,	false,	//  64
		false,	false,	false,	false,	false,	false,	false,	false,	//  72
		false,	false,	false,	false,	false,	false,	false,	false,	//  80
		false,	false,	false,	false,	false,	false,	false,	false,	//  88
		false,	false,	false,	false,	false,	false,	false,	false,	//  96
		false,	false,	false,	false,	false,	false,	false,	false,	// 104
		false,	false,	false,	false,	false,	false,	false,	false,	// 112
		false,	false,	false,	false,	false,	false,	false,	false	// 120
	};	

	/**
	 * Leading octal digit character lookup table. Note that the ordering of
	 * the entries takes into account that the byte is a signed type.
	 */
	private static final boolean[] LEADING_OCT_DIGIT = {
		false,	false,	false,	false,	false,	false,	false,	false,	// 128
		false,	false,	false,	false,	false,	false,	false,	false,	// 136
		false,	false,	false,	false,	false,	false,	false,	false,	// 144
		false,	false,	false,	false,	false,	false,	false,	false,	// 152
		false,	false,	false,	false,	false,	false,	false,	false,	// 160
		false,	false,	false,	false,	false,	false,	false,	false,	// 168
		false,	false,	false,	false,	false,	false,	false,	false,	// 176
		false,	false,	false,	false,	false,	false,	false,	false,	// 184
		false,	false,	false,	false,	false,	false,	false,	false,	// 192
		false,	false,	false,	false,	false,	false,	false,	false,	// 200
		false,	false,	false,	false,	false,	false,	false,	false,	// 208
		false,	false,	false,	false,	false,	false,	false,	false,	// 216
		false,	false,	false,	false,	false,	false,	false,	false,	// 224
		false,	false,	false,	false,	false,	false,	false,	false,	// 232
		false,	false,	false,	false,	false,	false,	false,	false,	// 240
		false,	false,	false,	false,	false,	false,	false,	false,	// 248
		false,	false,	false,	false,	false,	false,	false,	false,	//   0
		false,	false,	false,	false,	false,	false,	false,	false,	//   8
		false,	false,	false,	false,	false,	false,	false,	false,	//  16
		false,	false,	false,	false,	false,	false,	false,	false,	//  24
		false,	false,	false,	false,	false,	false,	false,	false,	//  32
		false,	false,	false,	false,	false,	false,	false,	false,	//  40
		true,	true,	true,	true,	false,	false,	false,	false,	//  48
		false,	false,	false,	false,	false,	false,	false,	false,	//  56
		false,	false,	false,	false,	false,	false,	false,	false,	//  64
		false,	false,	false,	false,	false,	false,	false,	false,	//  72
		false,	false,	false,	false,	false,	false,	false,	false,	//  80
		false,	false,	false,	false,	false,	false,	false,	false,	//  88
		false,	false,	false,	false,	false,	false,	false,	false,	//  96
		false,	false,	false,	false,	false,	false,	false,	false,	// 104
		false,	false,	false,	false,	false,	false,	false,	false,	// 112
		false,	false,	false,	false,	false,	false,	false,	false	// 120
	};	

	/**
	 * Numerical value hexidecimal digit character lookup table. Note that the ordering
	 * of the entries takes into account that the byte is a signed type.
	 */
	private static final byte[] NUMERICAL_VALUE = {
		0,	0,	0,	0,	0,	0,	0,	0,	// 128
		0,	0,	0,	0,	0,	0,	0,	0,	// 136
		0,	0,	0,	0,	0,	0,	0,	0,	// 144
		0,	0,	0,	0,	0,	0,	0,	0,	// 152
		0,	0,	0,	0,	0,	0,	0,	0,	// 160
		0,	0,	0,	0,	0,	0,	0,	0,	// 168
		0,	0,	0,	0,	0,	0,	0,	0,	// 176
		0,	0,	0,	0,	0,	0,	0,	0,	// 184
		0,	0,	0,	0,	0,	0,	0,	0,	// 192
		0,	0,	0,	0,	0,	0,	0,	0,	// 200
		0,	0,	0,	0,	0,	0,	0,	0,	// 208
		0,	0,	0,	0,	0,	0,	0,	0,	// 216
		0,	0,	0,	0,	0,	0,	0,	0,	// 224
		0,	0,	0,	0,	0,	0,	0,	0,	// 232
		0,	0,	0,	0,	0,	0,	0,	0,	// 240
		0,	0,	0,	0,	0,	0,	0,	0,	// 248
		0,	0,	0,	0,	0,	0,	0,	0,	//   0
		0,	0,	0,	0,	0,	0,	0,	0,	//   8
		0,	0,	0,	0,	0,	0,	0,	0,	//  16
		0,	0,	0,	0,	0,	0,	0,	0,	//  24
		0,	0,	0,	0,	0,	0,	0,	0,	//  32
		0,	0,	0,	0,	0,	0,	0,	0,	//  40
		0,	1,	2,	3,	4,	5,	6,	7,	//  48
		8,	9,	0,	0,	0,	0,	0,	0,	//  56
		0,	10,	11,	12,	13,	14,	15,	0,	//  64
		0,	0,	0,	0,	0,	0,	0,	0,	//  72
		0,	0,	0,	0,	0,	0,	0,	0,	//  80
		0,	0,	0,	0,	0,	0,	0,	0,	//  88
		0,	10,	11,	12,	13,	14,	15,	0,	//  96
		0,	0,	0,	0,	0,	0,	0,	0,	// 104
		0,	0,	0,	0,	0,	0,	0,	0,	// 112
		0,	0,	0,	0,	0,	0,	0,	0	// 120
	};
	

	private static final byte[] hexrep = {
		(byte)'0', (byte)'1', (byte)'2', (byte)'3',
		(byte)'4', (byte)'5', (byte)'6', (byte)'7',
		(byte)'8', (byte)'9', (byte)'a', (byte)'b',
		(byte)'c', (byte)'d', (byte)'e', (byte)'f'
	};
	
	/**
	 * 
	 * Determines whether the specified byte represents a PDF whitespace
	 * character. Whitespace characters are defined in section 3.1.1 of
	 * the PDF Reference Manual version 1.4.
	 *
	 * @param b		Character to test
	 *
	 * @return true if the character is whitespace.
	 */
	public static final boolean isWhitespace(byte b)
	{
		return WS[b + 128];
	}
	
	/**
	 * 
	 * Determines whether the specified character represents a PDF whitespace
	 * character. Whitespace characters are defined in section 3.1.1 of
	 * the PDF Reference Manual version 1.4.
	 *
	 * @param c		Character to test
	 *
	 * @return true if the character is whitespace.
	 */
	public static final boolean isWhitespace(char c)
	{
		// Note that I am using here only the second half of the WS table
		// because of the (c <= 32) clause.
		return (c <= 32) && WS[c + 128];
	}
	
	public static boolean isASCII(byte[] bytes)
	{
		int strInd;
		for (strInd = 0; strInd < bytes.length; strInd++)
		{
			byte curByte = bytes[strInd];
            if ((curByte < 32) && !ByteOps.isWhitespace(curByte)) 
                return false; // TODO; enhance the test
		}
		return true;
	}

	/**
	 * Computes the XOR of the individual bytes of arrays passed here and returns it.
	 */
	public static final byte[] xorArray(byte[] array, byte with)
	{
		byte[] result = new byte[array.length];
		int counter;
		for (counter = 0; counter < array.length; counter++)
			result[counter] = (byte) (array[counter] ^ with);
		return result;
	}
	
	
	private static int startsWithPartial(byte[] src, int srcOff, byte[] suffix, int sufOff)
	{
		int count;
		int sufInd = -1;
		for (count = srcOff; count < src.length; count++) {
			sufInd = count - srcOff + sufOff;
			if (sufInd >= suffix.length)
				break;
			if (src[count] != suffix[sufInd]) {
				sufInd = -1;
				break;
			}
		}
		return sufInd;
	}
	
	/**
	 * Returns the index at which this suffix exists in the src array.
	 */
	public static int endsWithPartial(byte[] src, int srcOff, byte[] suffix, int sufOff)
	{
		int count;
		for (count = srcOff; count < src.length; count++) {
			if (startsWithPartial(src, count, suffix, sufOff) != -1)
				return count;
		}
		return -1;
	}
	
	/**
	 * Checks the equality of two byte arrays.
	 */
	public static final int equalArrays(byte[] src, int srcOff, byte[] dst, int dstOff, int len)
	{
		int count;
		if ((src.length == 0) && (dst.length == 0))
			return -1;
		if ((src.length == 0) || (dst.length == 0))
			return 0;
		for (count = 0; count < len; count++) 
		{
			if ((srcOff + count) >= src.length)
				return count;
			if ((dstOff + count) >= dst.length)
				return count;
			if (src[srcOff + count] != dst[dstOff + count])
				return count;
		}
		return -1;
	}
	
	/**
	 * Copies source to destion array.
	 */
	public static final int copy(byte[] src, int srcOff, byte[] dst, int dstOff, int len)
	{
		int srcInd =srcOff;
		int destInd = dstOff;
		int retLen = Math.min((Math.min(src.length - srcOff, dst.length - dstOff)), len);
		int count = retLen;
		while (count-- > 0)
		{
			dst[destInd++] = src[srcInd++];
		}
		return retLen;
	}
	/**
	 * Converts the integer passed here to byte array.
	 */
	public static final byte[] splitInt2Bytes(int src, int count)
	{
		byte[] result = new byte[count];
		int index;
		for (index = 0; index < count; index++)	{
			result[index] = (byte) (src & 0xFF);
			src >>= 8;
		}
		return result;
	}
	
	/**
	 * 
	 * Determines whether the specified byte represents a regular PDF
	 * character. Regular characters are defined in section 3.1.1 of
	 * the PDF Reference Manual version 1.4.
	 *
	 * @param b		Character to test
	 *
	 * @return true if the character is regular.
	 */
	public static final boolean isRegular(byte b)
	{
		return REG[b + 128];
	}
	
	/**
	 * Determines whether the specified byte represents a literal PDF
	 * character. String literal characters are defined in section 3.2.3
	 * of the PDF Reference Manual version 1.4.
	 *
	 * @param b		Character to test
	 *
	 * @return true if the character is a literal.
	 */
	public static final boolean isLiteral(byte b)
	{
		return LIT[b + 128];
	}
	
	/**
	 * Determines whether the specified byte represents a decimal digit.
	 *
	 * @param b		Character to test
	 *
	 * @return true if the character represents a decimal digit.
	 */
	public static final boolean isDigit(byte b)
	{
		return DEC_DIGIT[b + 128];
	}
	
	/**
	 * Determines whether the specified byte represents a hex digit.
	 *
	 * @param b		Character to test
	 *
	 * @return true if the character represents a hex digit.
	 */
	public static final boolean isHexDigit(byte b)
	{
		return HEX_DIGIT[b + 128];
	}
	
	/**
	 * Determines whether the specified byte represents an octal digit.
	 *
	 * @param b		Character to test
	 *
	 * @return true if the character represents an octal digit.
	 */
	public static final boolean isOctalDigit(byte b)
	{
		return OCT_DIGIT[b + 128];
	}
		
	/**
	 * Determines whether the specified byte represents a leading octal digit.  A leading
	 * octal digit is one that leads a 3 digit octal sequence that will fit within a single
	 * byte.  That means that the octal digit must be between 0 and 3 inclusive.
	 *
	 * @param b		Character to test
	 *
	 * @return true if the character represents an octal digit.
	 */
	public static final boolean isLeadingOctalDigit(byte b)
	{
		return LEADING_OCT_DIGIT[b + 128];
	}
	
	/**
	 * Returns the numerical value for a byte that is used to represent a character.  So, if a byte contains
	 * the value 0x30 (or the "character" '0') that would return a value of 0.  This will work for decimal,
	 * octal, and hexadecimal values.
	 *
	 * @param b		Character to test
	 *
	 * @return true if the character represents an octal digit.
	 */
	public static final byte getNumericalValue(byte b)
	{
		return NUMERICAL_VALUE[b + 128];
	}
	
	/**
	 * Reads two characters that are assumed to be digits and combines them
	 * into a decimal value.
	 *
	 * @param s		Byte array containing two characters to convert to a
	 * 					a numeric value
	 * @param pos		Position in the array where the characters start
	 *
	 * @return Value of the two characters combined and treated as decimal
	 * 			digits (e.g. "12" is returned as the value 12).
	 * @throws PDFParseException	If two characters could not be read from the
	 * 							byte array or the characters did not correspond
	 * 							to decimal digits.
	 */
	public static int readField(byte[] s, int pos)
	throws PDFParseException
	{
		// Verify that the array contains at least two characters starting
		// at the specified position, and that the two characters correspond
		// to decimal digits.
		//
		if (!(pos + 1 < s.length
			  && ByteOps.isDigit(s[pos])
			  && ByteOps.isDigit(s[pos + 1]))) {
			throw new PDFParseException(Integer.toString(pos));
		}
		
		// Convert the two characters to integers and combine them.
		//
		return (s[pos] - '0') * 10 + (s[pos + 1] - '0');
	}
	
	/**
	 * Returns hex representation of src. 
	 */
	public static byte[] getHex(byte[] src)
	{
		int j = 0;
		byte[] rslt = new byte[src.length * 2];
		for (int i = 0; i < src.length; i++) {
			byte b = src[i];
			rslt[j++] = hexrep[(b >> 4) & 0xf];
			rslt[j++] = hexrep[b & 0xf];
		}
		return rslt;
	}
	
	/**
	 * 
	 * Hex encodes the data in src to a new byte array that is
	 * of length size. If the hex encoded data is less than size,
	 * then the buffer is padded with zeroes.
	 * 
	 * The size parameter must be equal to or greater than 2 * src.length
	 * or an ArrayIndexOutOfBoundsException will occur.
	 * 
	 * The size parameter should also be even since hex encoded data
	 * always has an even length.
	 */
	public static byte[] getHex(byte[] src, int size)
	{
		int j = 0;
		byte[] rslt = new byte[size];
		for (int i = 0; i < src.length; i++) {
			byte b = src[i];
			rslt[j++] = hexrep[(b >> 4) & 0xf];
			rslt[j++] = hexrep[b & 0xf];
		}
		return rslt;
	}
	
	/**
	 * 
	 * Converts the contents of a byte[] to a hex string.
	 */
	public static String getHexString(byte in[])
	{
		byte[] hex = getHex(in);
		try{
			return new String(hex, "ASCII");
		}catch (UnsupportedEncodingException e)
		{
			throw new RuntimeException("ASCII encoding not supported.",e);
		}
	}

	/**
	 * Converts the contents of a byte[] to a hex name (lowercase).
	 * @param in
	 * @return ASName
	 * @throws UnsupportedEncodingException
	 */
	public static ASName getHexName(byte in[])
	throws UnsupportedEncodingException
	{
		return getHexName(in, false);
	}
	
	/**
	 * Converts the contents of a byte[] to a hex name.
	 * @param in the bytes used to make the hex name
	 * @param uppercase true if the hex should utilize upper-case alpha characters;
	 * false if the alpha characters should be lower-case
	 * @return ASName
	 */
	public static ASName getHexName(byte in[], boolean uppercase)
	{
		String hex = getHexString(in);
		return ASName.create(uppercase ? hex.toUpperCase() : hex);
	}
	
	/**
	 * Converts the contents of a byte[] to a hex string.
	 * 
	 * The size parameter must be equal to or greater than 2 * src.length
	 * or an ArrayIndexOutOfBoundsException will occur.
	 * 
	 * The size parameter should also be even since hex encoded data
	 * always has an even length.
	 */
	public static String getHexString(byte in[], int size)
	throws UnsupportedEncodingException
	{
		byte[] hex = getHex(in, size);
		return new String(hex, "ASCII");
	}

	/**
	 * Converts the contents of a byte[] to a hex string.
	 */
	public static String getHexString2(byte in[]) 
	{
	    byte ch = 0x00;
	    int i = 0; 
	    if (in == null || in.length <= 0)
	        return null;
	        
	    String pseudo[] = {"0", "1", "2","3", "4", "5", "6", "7", "8", "9", "A", "B", "C", "D", "E","F"};
	    StringBuilder out = new StringBuilder(in.length * 2);
	    
	    while (i < in.length)
	    {
	        ch = (byte) (in[i] & 0xF0); // Strip off 	high nibble
	        ch = (byte) (ch >>> 4);
	        //shift the bits down
	        ch = (byte) (ch & 0x0F);    
	        // must do this is high order bit is on!
	        out.append(pseudo[ch]); // convert the	nibble to a String Character
	        ch = (byte) (in[i] & 0x0F); // Strip off	low nibble 
	        out.append(pseudo[ch]); // convert the	nibble to a String Character
	        i++;
	    }
	    return new String(out);
	}

	/**
	 * Converts the specified hexidecimal character to a byte value. For
	 * example, the hex character 'A' is converted to the value 10.
	 *
	 * @param b
	 *            Hex character to convert
	 * @return Byte value corresponding to the specified hex character.
	 * @throws PDFParseException
	 *
	 */
	// TODO - clean up to use the same style as above
	public static byte getHex(byte b)
	throws PDFParseException
	{
		int rslt = 0;
		if (b >= '0' && b <= '9') {
			rslt = (b - '0');
		} else if (b >= 'A' && b <= 'F') {
			rslt = (b - 'A') + 10;
		} else if (b >= 'a' && b <= 'f') {
			rslt = (b - 'a') + 10;
		} else {
			throw new PDFParseException("Expected Hex Digit - instead got " + Byte.toString(b));
		}
		return (byte) rslt;
	}
	
	
	/**
	 * In presumption that the input byte array contains Unicode characters
	 * converts them to Java String.
	 * If the input byte array is null, empty or contains non-Unicode characters
	 * returns null.
	 * @param unicodeBytes
	 * @return String representation of the Unicode characters in a byte array
	 */
	public static String getUnicode(byte[] unicodeBytes)
	{
		String encoding = UTFSupport.UTF8;
		if ((unicodeBytes == null) || (unicodeBytes.length <= 2))
			return null;
		if (((unicodeBytes[0] == -2) && (unicodeBytes[1] == -1)) || ((unicodeBytes[0] == -1) && (unicodeBytes[1] == -2)))
			encoding = UTFSupport.UTF16;
		try {
			return new String(unicodeBytes, encoding);
		} catch(UnsupportedEncodingException e){
			throw new RuntimeException( encoding + " encoding not supported.",e);
		}catch (Exception exp) {
			return null;
		}
	}
	public static final byte[] toByte(short word0)
    {
        byte abyte0[] = new byte[2];
        for(byte byte0 = 0; byte0 <= 1; byte0++)
            abyte0[byte0] = (byte)(word0 >>> (1 - byte0) * 8);

        return abyte0;
    }
}
