/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */
package com.adobe.xfa;

import com.adobe.xfa.ut.ExFull;
import com.adobe.xfa.ut.ResId;

/**
 * This class describes a relationship of a parent to one of its children.
 * 
 * <h4><a name="immutability">Immutability</a></h4>
 * ChildReln objects are immutable.
 *
 * @see NodeSchema
 *
 * @exclude from published api -- Mike Tardif, May 2006.
 */
public class ChildReln {

	public static final int oneOfChild = 4;

	public static final int oneOrMore = 3;

	/**
	 * enumeration unlimited
	 */
	public static final int UNLIMITED = -1;

	public static final int zeroOrMore = 2;

	/**
	 * enumeration occursEnum <br>
	 * oneofChild = Only one child allowed -- exclusive OR
	 */
	public static final int zeroOrOne = 1;

	private static final ChildReln goOneOfChild = new ChildReln(oneOfChild, 1);

	private static final ChildReln goOneOrMore = new ChildReln(oneOrMore, 1);

	private static final ChildReln goZeroOrMore = new ChildReln(zeroOrMore, 1);

	private static final ChildReln goZeroOrOne = new ChildReln(zeroOrOne, 1);

	public static ChildReln getOneOfChild() {
		return goOneOfChild;
	}

	public static ChildReln getOneOrMore() {
		return goOneOrMore;
	}

	public static ChildReln getZeroOrMore() {
		return goZeroOrMore;
	}

	public static ChildReln getZeroOrOne() {
		return goZeroOrOne;
	}

	private final int meOccurrence;

	private final int mnMaxOccurrence;

	/**
	 * Assignments should only be done once in xfanamespace.cpp so this method
	 * is private to prevent its use. <br>
	 * Value UNLIMITED means no limit on number of occurrences.
	 * 
	 * Constructor. Populates the child relationship member variables
	 * 
	 * @param count -
	 *            The number of times this child may occur.
	 * @param nMax -
	 *            The maximum number of times this child may occur. for
	 *            zeroOrOne or oneOf elements: default = "1" for zeroOrMore or
	 *            oneOrMore elements: default = UNLIMITED
	 */
	public ChildReln(int count, int nMax /* = 1 */) {
		meOccurrence = count;
		// basis of assumption for speed-up in xfanodeimpl.cpp
		assert (nMax != 0);

		if ((meOccurrence == oneOrMore || meOccurrence == zeroOrMore)
				&& nMax == 1) {
			//
			// case where the default nMax=1 isn't appropriate
			//
			nMax = UNLIMITED;
		}
		
		mnMaxOccurrence = nMax;
	}

	/**
	 * Accessor. Returns the maximum number of times this element may occur. If
	 * the number of occurences is unlimited, this returns -1.
	 * 
	 * @return a long integer indicating maximum occurrences.
	 */
	public int getMax() {
		return mnMaxOccurrence;
	}

	/**
	 * Accessor. Returns the minimum number of times this element may occur.
	 * 
	 * @return a long integer indicating minimum occurrences. This is zero or
	 *         one
	 */
	public int getMin() {
		switch (meOccurrence) {
		case zeroOrOne:
		case zeroOrMore:
		case oneOfChild:
			return 0;
		}
		return 1;
	}

	/**
	 * Accessor. Returns the number of valid occurrences of this child
	 * 
	 * @return The number of times this child may occur.
	 */
	public int getOccurrence() {
		return meOccurrence;
	}

	/**
	 * Initializes the child relationship member variables <br>
	 * Value UNLIMITED means no limit on number of occurrences.
	 * 
	 * @param count -
	 *            The number of times this child may occur.
	 * @param nMax -
	 *            The maximum number of times this child may occur. for
	 *            zeroOrOne or oneOf elements: default = "1" for zeroOrMore or
	 *            oneOrMore elements: default = UNLIMITED
	 */
	public void init(int count, int nMax/* =1L */) {
		//
		// JavaPort: unsupported to ensure class immutability.
		//
		throw new ExFull(ResId.UNSUPPORTED_OPERATION, "ChildReln#init");
	//	meOccurrence = count;
	//	mnMaxOccurrence = nMax;
	//
	//	if ((meOccurrence == oneOrMore || meOccurrence == zeroOrMore)
	//			&& nMax == 1) {
	//		//
	//		// case where the default nMax=1 isn't appropriate
	//		//
	//		mnMaxOccurrence = UNLIMITED;
	//	}
	}

	// returns true if not a derived instance
	public boolean isBaseClass() {
		return true;
	}
}
