/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */

package com.adobe.xfa;

import java.io.ByteArrayOutputStream;
import java.util.ArrayList;
import java.util.List;

import com.adobe.xfa.SOMParser.SomResultInfo;
import com.adobe.xfa.ut.StringUtils;

/**
 * This object will manages the restoration xfa properties.
 *
 * @exclude from published api.
 */
public final class Delta extends Obj {
	// Adobe patent application tracking # B252, entitled METHOD AND SYSTEM TO PERSIST STATE, inventors: Roberto Perelman, Chris Solc, Anatole Matveief, Jeff Young, John Brinkman
	// Adobe patent application tracking # B322, entitled METHOD AND SYSTEM TO MAINTAIN THE INTEGRITY OF A CERTIFIED DOCUMENT WHILE PERSISTING STATE IN A DYNAMIC FORM, inventors: Roberto Perelman, Chris Solc, Anatole Matveief, Jeff Young, John Brinkman

	final boolean mbRecursiveRestore;

	boolean mbRestored;

	final Element mDeltaParent;

	final Object mDeltaValue;

	final Element mParent;

	final Object mValue;

	String msTarget;

	public Delta(Element parent, Element deltaParent, Object currentValue, Object savedValue, String sTarget /* "" */) {
		mValue = currentValue;
		mDeltaValue = savedValue;
		mParent = parent;
		mDeltaParent = deltaParent;
		//mbRestored = false;
		msTarget = sTarget;
		mbRecursiveRestore = false;
	}

	/**
	 * @exclude from public api.
	 */
	public Delta(Element parent, Element deltaParent, String sSOM) {
		//mbRestored = false;
		mbRecursiveRestore = true;
		
		// grab the current value and the saved value

		SOMParser parser = new SOMParser(null);
		List<SomResultInfo> results = new ArrayList<SomResultInfo>();

		Element resolvedParent = null;
		Object value = null;
		
		if (parser.resolve(parent, sSOM, results)) {
			if (results.size() == 1) {
				SOMParser.SomResultInfo result = results.get(0);
				Arg arg = result.value;
				Obj obj = result.object;
				if (obj instanceof Element) {
					resolvedParent = (Element) obj;
					msTarget = result.propertyName;

					if (result.occurrence != 0) {
						String sNum = Integer.toString(result.occurrence);
						msTarget += '[';
						msTarget += sNum;
						msTarget += ']';
					}
				}

				if (arg.getArgType() == Arg.OBJECT) {
					Obj objValue = arg.getObject();
					if (objValue instanceof Element)
						value = objValue;
				} else if (arg.getArgType() != Arg.EXCEPTION) {
					StringAttr attrValue = new StringAttr(null, arg.getString());
					value = attrValue;
				}
			}
		}
		
		mParent = resolvedParent;
		mValue = value;

		// clear the list
		results.clear();
		
		Object deltaValue = null;

		if (deltaParent != null &&
			parent.isSameClass(deltaParent) && 
			parser.resolve(deltaParent, sSOM, results)) {
			
			if (results.size() == 1) {
				SOMParser.SomResultInfo result = (SomResultInfo) results.get(0);
				Obj obj = result.object;
				if (obj instanceof Element)
					deltaParent = (Element) obj;

				Arg arg = result.value;
				if (arg.getArgType() == Arg.OBJECT) {
					Obj objValue = arg.getObject();
					if (objValue instanceof Element)
						deltaValue = objValue;
				} else if (arg.getArgType() != Arg.EXCEPTION) {
					StringAttr attrValue = new StringAttr(null, arg.getString());
					deltaValue = attrValue;
				}
			}
		}
		
		mDeltaParent = deltaParent;
		mDeltaValue = deltaValue;
	}

	/**
	 * @see Obj#getClassAtom()
	 */
	public String getClassAtom() {
		return STRS.DELTA;
	}

	/**
	 * @see Obj#getClassName()
	 */
	public String getClassName() {
		return STRS.DELTA;
	}

	String getCurrentValue() {
		return propToString(mValue);
	}

	Element getParent() {
		return mParent;
	}

	String getSavedValue() {
		if (mDeltaValue == null)
			return propToString(mValue);

		return propToString(mDeltaValue);
	}

	/**
	 * @exclude from published api.
	 */
	public ScriptTable getScriptTable() {
		return DeltaScript.getScriptTable();
	}

	String getTarget() {
		Object oDeltaValue = mValue;
		if (StringUtils.isEmpty(msTarget) && oDeltaValue instanceof Element) {
			Element oNode = (Element) mValue;
			msTarget = oNode.getSOMExpression(mParent, false);
		}
		return msTarget;
	}

	String propToString(Object property) {
		if (property == null)
			return "";

		if (property instanceof Attribute) {
			Attribute attr = (Attribute) property;
			return attr.toString();
		} else if (property instanceof Element) {
			ByteArrayOutputStream outFile = new ByteArrayOutputStream();
			
			Element node = (Element) property;
			node.saveXML(outFile, null);
			return outFile.toString();
		}
		 //CL#699561		
		else if (property instanceof TextNode) // bugs 2603320/2603245
		{
			TextNode oNode = (TextNode)property;
			AppModel appModel = oNode.getModel().getAppModel();
			if ( appModel!=null && 
				!appModel.getLegacySetting(AppModel.XFA_LEGACY_V32_SCRIPTING) )
				return oNode.getText();
			
			//By default return oNode.getText(). This behaviour different as compared to C++
			//since in Java TextNode is not derived from Element.
			return oNode.getText();
		}
		return "";
	}

	/**
	 * @exclude from public api.
	 */
	public void restore() {
		if (mbRestored)
			return;

		if (mDeltaValue == null)
			return;

		Object deltaValue = mDeltaValue;

		if (deltaValue instanceof Attribute && mParent != null) {
			String aAttrName = msTarget.intern();

			int eTag = XFA.getAttributeTag(aAttrName);
			if (eTag != -1) {
				Attribute attr = (Attribute) mDeltaValue;
				mParent.setAttribute(attr, eTag);

				// clear the attribute on the delta node
				if (mDeltaParent != null && mDeltaParent != mParent) {
					mDeltaParent.setAttribute(null, eTag);
				}
			}
		} 
		else if (deltaValue instanceof Element && mDeltaValue != mValue) {
			Element deltaNode = (Element) mDeltaValue;
			Element node = (Element) mValue;
			if (node == null) {
				if (deltaNode != null) {
					// remove the delta from the parent
					deltaNode.remove();
					mParent.appendChild(deltaNode);
				}
			} 
			else if (mbRecursiveRestore) {
				XFAList list = new XFAList();
				node.getDeltas(deltaNode, list);

				int nLen = list.length();
				for (int i = 0; i < nLen; i++) {
					Delta delta = (Delta) list.item(i);
					delta.restore();
				}
			} else {
				node.restoreDelta(deltaNode);
			}
		}
		else if (deltaValue instanceof TextNode && mDeltaValue != mValue) {
			TextNode deltaNode = (TextNode) mDeltaValue;
			TextNode node = (TextNode) mValue;
			
			if (node == null) {
				if (deltaNode != null) {
					deltaNode.remove();
				}
			}
			else {
				node.restoreDelta(deltaNode);
			}
		}

		mbRestored = true;
	}
}