/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */
package com.adobe.xfa;

import com.adobe.xfa.ut.ExFull;
import com.adobe.xfa.ut.MsgFormatPos;
import com.adobe.xfa.ut.ResId;
import com.adobe.xfa.ut.StringUtils;


/**
 * An XFAAttribute object which holds an integer value. This class may be used
 * to ruct an XFAProperty object which in turn may be used in calls to
 * <code>XFAPropety.setProperty()</code> and
 * <code>XFAProperty.getProperty()</code><br>
 * Example: ("Int i = node.getProperty(...)")
 * 
 * @exclude from published api -- Mike Tardif, May 2006.
 */

public final class Int extends Attribute {
	private final int miValue;

	/**
	 * Fully qualified constructor.
	 * @param NS the namespace for this attribute
	 * @param localName the local name for this attribute
	 * @param qName the qualified name for this attribute 
	 * @param the string value of the attribute
	 * @param internSymbols indicates whether the symbols in other parameters need to be interned.
	 */
	private Int(String NS, String localName, String qName, String value, boolean internSymbols) {
		super(NS, localName, qName, value, internSymbols);
		
		int iValue = 0;
		if (!StringUtils.isEmpty(value)) {
			try {
				iValue = Integer.parseInt(value);
			} 
			catch (NumberFormatException n) {
				MsgFormatPos oContext = new MsgFormatPos(
						ResId.InvalidPropertyValueException);
				oContext.format(value);
				oContext.format("integer");
				ExFull oErr = new ExFull(oContext);
				throw oErr;
			}
		}
		
		miValue = iValue;
	}

	/**
	 * Construct an instance from an int.
	 * 
	 * @param qName the qualified name for this attribute 
	 * @param value the int value of the attribute
	 */
	public Int(String qName, int value) {
		this(qName, value, true);
	}
	
	/**
	 * Construct an instance from an int, and specify whether symbols need to be interned.
	 * @param qName the qualified name for this attribute 
	 * @param value the int value of the attribute
	 * @param internSymbols indicates whether the symbols in other parameters need to be interned.
	 * 
	 * @exclude from published api.
	 */
	 public Int(String qName, int value, boolean internSymbols) {
		super(null, null, qName, Integer.toString(value), false);
		miValue = value;
	}
	
	/**
	 * Construct an instance from the string value
	 * 
	 * @param qName the qualified name for this attribute 
	 * @param value the string value of the attribute
	 */
	public Int(String qName, String value) {
		this(null, null, qName, value, true);
	}

	/**
	 * Get the attribute value.
	 * 
	 * @return the attribute value as integer
	 */
	public int getValue() {
		return miValue;
	}

	/**
	 * @see Attribute#newAttribute(String)
	 */
	public Attribute newAttribute(String value) {
		return newAttribute(getNS(), getLocalName(), getQName(), value, false);
	}

	/* (non-Javadoc)
	 * @see com.adobe.xfa.Attribute#newAttribute(java.lang.String, java.lang.String, java.lang.String, java.lang.String)
	 */
	public Attribute newAttribute(String NS, String localName, String qName, String value) {
		return newAttribute(NS, localName, qName, value, true);
	}
	
	/* (non-Javadoc)
	 * @see com.adobe.xfa.Attribute#newAttribute(java.lang.String, java.lang.String, java.lang.String, java.lang.String, boolean)
	 */
	public Attribute newAttribute(String NS, String localName, String qName, String value, boolean internSymbols) {
		return new Int(NS, localName, qName, value, internSymbols);
	}

	/**
	 * Cast this object to an integer
	 * 
	 * @return the internal integer value.
	 */
	public int toInt() {
		return getValue();
	}
	
	/**
	 * @see Object#toString()
	 */
	public String toString() {
		// JavaPort: Really shouldn't need to implement this method... 
		// But to be consistent with C++ we have to.
		// As a result, an empty integer value always returns a string: "0".
		return Integer.toString(miValue);
	}
}