/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */
package com.adobe.xfa;

import com.adobe.xfa.ut.ISODate;
import com.adobe.xfa.ut.ISOTime;
import com.adobe.xfa.ut.LcDate;
import com.adobe.xfa.ut.LcTime;

/**
 * Class XFALogMessageData represents message data. Members of this class
 * consist of a unique message Id, a message severity, a date time stamp and
 * text.
 *
 * @exclude from published api -- Mike Tardif, May 2006.
 */
public final class LogMessageData {

	private int meSeverity;

	private final int mnId;

	private final ISODate mpDate;

	private final ISOTime mpTime;

	private final String msDate;

	//private String msSOM;

	private final String msText;

	private final String msTime;
	
	private final String msSOM;

	private static final String[] msSeverityName = {
		"",		/* LogMessage.MSG_IGNORE */
		"t",	/* LogMessage.MSG_TRACE */
		"i",	/* LogMessage.MSG_INFORMATION */
		"w",	/* LogMessage.MSG_WARNING */
		"vw",	/* LogMessage.MSG_VALIDATION_WARNING */
		"ve",	/* LogMessage.MSG_VALIDATION_ERROR */
		"f"		/* LogMessage.MSG_FATAL_ERROR */
	};

	/**
	 * Constructor
	 * 
	 * @param nId -
	 *            the message Id
	 * @param sText -
	 *            the message text
	 * @param eSeverity -
	 *            the message severity
	 * @param pDate -
	 *            the message date
	 * @param pTime -
	 *            the message time
	 * @param sSOM -
	 *            the SOM expression for the node causing the message
	 */
	public LogMessageData(int nId, String sText, int eSeverity,
			ISODate pDate /* = NULL */, ISOTime pTime /* = NULL */, String sSOM /* = "" */) {
		mnId = nId;
		msText = sText;
		meSeverity = eSeverity;
		msSOM = sSOM;

		String sLocalDateFormat = LcDate.DATE_FMT2;
		String sLocalTimeFormat = LcTime.TIME_FMT2;

		if (pDate == null) {
			mpDate = new ISODate();
		} else {
			mpDate = pDate;
		}

//		mpTime.setLocalDate();
		msDate = mpDate.format(sLocalDateFormat);

		if (pTime == null) {
			mpTime = new ISOTime();
		} else {
			mpTime = pTime;
		}

		//
		// Set to local time format
		//
		mpTime.setLocalTime();
		msTime = mpTime.format(sLocalTimeFormat);
	}

	/**
	 * Retrieve the date and time of the message. <br>
	 * <br>
	 * uses ISO standard 8601 date and time format YYYY-MM-DDTHH:MM:SS.FFF Z
	 * 
	 * @return date and time of the message.
	 * 
	 */
	public String getDateTime() {
		return msDate + 'T' + msTime;
	}

	/**
	 * Retrieve the message Id.
	 * 
	 * @return message Id
	 * 
	 */
	public int getId() {
		return mnId;
	}

	/**
	 * Retrieve the severity of the message.
	 * 
	 * @return severity of the message.
	 * 
	 */
	public int getSeverity() {
		return meSeverity;
	}

	/**
	 * Retrieve the severity name of the message.
	 * 
	 * @return severity of the message.
	 * 
	 */
	public String getSeverityName() {
		return msSeverityName[meSeverity];
	}

	/**
	 * Retrieve the SOM expression.
	 * 
	 * @return string representing the SOM expression for the node that caused
	 *         the message.
	 * 
	 */
	public String getSOMExpression() {
		return msSOM;
	}

	/**
	 * Retrieve the message text.
	 * 
	 * @return string representing the message text.
	 * 
	 */
	public String getText() {
		return msText;
	}

	/**
	 * Set the severity of the message.
	 * 
	 * @param eSeverity - the value of the severity
	 */
	public void setSeverity(int eSeverity) {
		meSeverity = eSeverity;
	}

}
