/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */

package com.adobe.xfa;

import com.adobe.xfa.ut.UnitSpan;


/**
 * Measurement class. Used for all attributes that are expressed as
 *              an ordinate, such as X, Y, H, W, MinX.
 *
 * @exclude from published api -- Mike Tardif, May 2006.
 */
public final class Measurement extends Attribute {

	private UnitSpan mValue;

	/**
	 * Fully qualified constructor.
	 * @param qName
	 * @param value 
	 */
	public Measurement(String qName, String value) {
		super(null, null, qName, value);
	}

	/**
	 * Create a Measurement from an attribute.
	 * 
	 * @param attr
	 *            the attribute to create the measurement from.
	 */
	public Measurement(Attribute attr) {
		super(attr.getNS(), attr.getName(), attr.getQName(), attr.toString());
	}
	
	private Measurement(String NS, String localName, String qName, String value, boolean internSymbols) {
		super(NS, localName, qName, value, internSymbols);
	}

	/**
	 * Create an Measurement from a UnitSpan.
	 * 
	 * @param unit
	 *            the UnitSpan to create the measurement from.
	 */
	public Measurement(UnitSpan unit) {
		super(null, unit.toString());
		mValue = unit;
	}

	/**
	 * Get the internal value as an enum. The returned enum can be used to get a
	 * string representation of the units.
	 * 
	 * @return The enumerated units value (EnumAttr.XFAUnitEnum)
	 */
	public int getUnits() {
		if (mValue == null) {
			mValue = new UnitSpan(getAttrValue());
		}
		int eCode = mValue.units();
		switch (eCode) {
		case UnitSpan.INCHES_1M:
		case UnitSpan.INCHES_72K:
			return EnumAttr.INCH;

		case UnitSpan.CM_250K:
			return  EnumAttr.CENTIMETER;

		case UnitSpan.MM_25K:
			return  EnumAttr.MILLIMETER;

		case UnitSpan.POINTS_1K:
			return EnumAttr.POINT;

		case UnitSpan.MILLIPOINT:
			return EnumAttr.MILLIPOINT;
			
		case UnitSpan.PICAS_12K:
			return EnumAttr.PICA;

		}
		return -1;
	}

	/**
	 * Get the internal value as a UnitSpan
	 * 
	 * @return the internal value as UnitSpan
	 */
	public UnitSpan getUnitSpan() {
		if (mValue == null) {
			mValue = new UnitSpan(getAttrValue());
		}
		return mValue;
	}

	/**
	 * Get the internal value as a double
	 * 
	 * @return the internal value as a double
	 */
	public double getValue() {
		return getValueAsUnit(getUnits());
	}

	/**
	 * Get the internal value converted to specific units defined in XFAUnitEnum
	 * 
	 * @param units
	 *            the units the value will be converted to
	 *            (EnumAttr.XFAUnitEnum)
	 * @return the internal value converted to the given units
	 */

	public double getValueAsUnit(int units) {
		return( getValueAsUnit(getUnitSpan(), units) );
	}
	
	/**
     * Get the value of the provided UnitSpan, converted to specific units defined
	 * in XFAUnitEnum.  This is a static function.
	 * @param oSpan - the UnitSpan whose value should be converted
	 * @param units - the units the value will be converted to (XFAEnum::XFAUnitEnum)
     * @return the internal value converted to the given units
     */
	
	public static double getValueAsUnit(UnitSpan oSpan, int units) {
		double dRet = 0.0d;

		if (units == EnumAttr.INCH)
			dRet = oSpan.valueAsUnit(UnitSpan.INCHES_72K) / 72000.0d;

		else if (units == EnumAttr.MILLIMETER)
			dRet = oSpan.valueAsUnit(UnitSpan.MM_25K) / 25000.0d;

		else if (units == EnumAttr.POINT)
			dRet = oSpan.valueAsUnit(UnitSpan.POINTS_1K) / 1000.0d;

		else if (units == EnumAttr.CENTIMETER)
			dRet = oSpan.valueAsUnit(UnitSpan.CM_250K) / 250000.0d;

		else if (units == EnumAttr.MILLIPOINT)
			dRet = oSpan.valueAsUnit(UnitSpan.MILLIPOINT);
		
		else if (units == EnumAttr.PICA)
			dRet = oSpan.valueAsUnit(UnitSpan.PICAS_12K);

		return dRet;
	}
	
	/* (non-Javadoc)
	 * @see com.adobe.xfa.Attribute#newAttribute(java.lang.String)
	 */
	public Attribute newAttribute(String value) {
		return newAttribute(getNS(), getLocalName(), getQName(), value, false);
	}

	/* (non-Javadoc)
	 * @see com.adobe.xfa.Attribute#newAttribute(java.lang.String, java.lang.String, java.lang.String, java.lang.String)
	 */
	public Attribute newAttribute(String NS, String localName, String qName, String value) {
		return newAttribute(NS, localName, qName, value, true);
	}
	
	/* (non-Javadoc)
	 * @see com.adobe.xfa.Attribute#newAttribute(java.lang.String, java.lang.String, java.lang.String, java.lang.String, boolean)
	 */
	public Attribute newAttribute(String NS, String localName, String qName, String value, boolean internSymbols) {
		return new Measurement(NS, localName, qName, value, internSymbols);
	}

	public void normalize() {
		if (mValue == null) {
			mValue = new UnitSpan(getAttrValue());
			setAttrValue(mValue.toString());
		}
	}
	
	public String toString() {
		if (mValue == null) {
			mValue = new UnitSpan(getAttrValue());
		}
		return mValue.toString();
	}

}
