/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */
package com.adobe.xfa;

import org.xml.sax.Attributes;

import com.adobe.xfa.ut.ExFull;
import com.adobe.xfa.ut.FindBugsSuppress;
import com.adobe.xfa.ut.ResId;

/**
 * A class to represent model factories.
 * A model factory is used by application models to create a model
 * where appropriate, when loading an XML file.
 * The factory provides interfaces
 * for defining the behaviors of the model once it is created.
 */
public abstract class ModelFactory {

	/**
	 *
	 * @exclude from published api.
	 */
	protected final int		meClassTag;

	/**
	 *
	 * @exclude from published api.
	 */
	protected final String	maRootName;	// interned

	/**
	 *
	 * @exclude from published api.
	 */
	protected final String	maShortCutName;
	
	/**
	 *
	 * @exclude from published api.
	 */
	public ModelFactory(int	eClassTag,
					String aRootName,
					String aShortCutName) {
		meClassTag		= eClassTag;
		maRootName 		= aRootName;
		maShortCutName	= aShortCutName;
	}	
	
	/**
	 * Returns the root name of the factory.
	 * @return the name of the root as an atom
	 *
	 * @exclude from published api.
	 */
	final public String rootName() {
		return maRootName;
	}

	/**
	 * Sets an option
	 *
	 * @param optionName - the name of the option.
	 * @param optionValue - the value of the option.
	 * @param bCritical - disallow further modification of this option.
	 * @exception InvalidOptionException since not allowed to 
	 * set options for this model
	 *
	 * @exclude from published api.
	 */
	public void setOption(String optionName, String optionValue, boolean bCritical) {
		throw new ExFull(ResId.InvalidOptionException, optionName);
	}
	
	
	/**
	 * @param parent
	 * @param uri
	 * @param localName This string must be interned.
	 * @exclude from published api.
	 */
	@FindBugsSuppress(code="ES")
	protected boolean isRootNode(AppModel parent, String uri, String localName) {
		return (localName == rootName());
	}
	
	/**
	 * @param parent
	 * @param prevSibling
	 * @param uri This string must be interned.
	 * @param localName This string must be interned.
	 * @param qName This string must be interned.
	 * @param a
	 * @exclude from published api.
	 */
	abstract protected Model newModel(AppModel parent, Node prevSibling, String uri, String localName, String qName, Attributes a);
	
	/**
	 *
	 * @exclude from published api.
	 */
	protected Model findModel(Node parent) {
		Node oChild = parent.getFirstXFAChild();
		while (oChild != null) {
			if (isRootName(oChild.getName())) {
				if (oChild instanceof Model)
					return (Model) oChild;
			}
			oChild = oChild.getNextXFASibling();
		}
		return null;
	}
	
	/**
	 * Determines whether the class of Model that this ModelFactory creates 
	 * allows parsing a duplicate model that is added to an existing model.
	 * In C++, this determination is made after the XML DOM is parsed and
	 * XMLStorage.loadModel attempts to call Model.add, which by default
	 * throws an exception if adding to an existing model isn't supported.
	 * @exclude from published api.
	 */
	public boolean getAllowAdd() {
		return false;
	}

	/**
	 * Create the XML DOM node and the corresponding XFA Model.
	 * In C++, this only creates the XML DOM node, and a separate
	 * step creates the corresponding Model node. In XFA4J, the
	 * we don't have the capability of calling AppModel.loadNode
	 * to instantiate an XFA Model from the DOM node, so this method
	 * is changed to create the XFA model.  
	 * @param parent the XFA DOM node that will be the XML parent of
	 * the new DOM peer.
	 * @return the XFA Model that was created.
	 * @exclude from published api.
	 */
	abstract public Model createDOM(Element parent);
	
	/**
	 * @param name the name to be tested. This string must be interned.
	 * @exclude from published api.
	 */
	@FindBugsSuppress(code="ES")
	public boolean isRootName(String name) {
		return rootName() == name;
	}
}
