/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */
package com.adobe.xfa;


/**
 * This class describes the schema of a node: What child nodes it may (or may
 * not) have, as well as what attributes may be set.
 * 
 * @exclude from published api -- Mike Tardif, May 2006.
 */

public final class NodeSchema {

	private AttributeInfo[] mAttributeInfoArray;

	private SchemaPairs mAttributes; // Map<eTag, Attribute>

	private SchemaPairs mChildren; // Map<eTag, ChildReln>

	private ChildRelnInfo[] mChildrenInfoArray;

	private SchemaPairs mForeignChildren;

	private final int mnAttributeMax;

	private final int mnAttributeMin;

	private final int mnElementMax;

	private final int mnElementMin;

	/**
	 * Default Constructor.
	 * @param nAttributeMin - min attribute tag for this schema
	 * @param nAttributeMax - max attribute tag for this schema
	 * @param nElementMin - min element tag for this schema
	 * @param nElementMax - max element tag for this schema
	 */
	public NodeSchema(int nAttributeMin, int nAttributeMax, int nElementMin,
			int nElementMax) {
		mnAttributeMin = nAttributeMin;
		mnAttributeMax = nAttributeMax;
		mnElementMin = nElementMin;
		mnElementMax = nElementMax;
	}

	/**
	 * Copy constructor
	 * @param other - the NodeSchema to copy
	 * @param nAttributeMin - min attribute tag for this schema
	 * @param nAttributeMax - max attribute tag for this schema
	 * @param nElementMin - min element tag for this schema
	 * @param nElementMax - max element tag for this schema
	 */
	public NodeSchema(NodeSchema other, int nAttributeMin, int nAttributeMax,
			int nElementMin, int nElementMax) {
		
		this(nAttributeMin, nAttributeMax, nElementMin, nElementMax);

		if (other != null) {
			int attrCount = ATTR_COUNT();
			if (other.mAttributes != null)
				mAttributes = new SchemaPairs(other.mAttributes);

			// copy over the attribute info
			if (other.mAttributeInfoArray != null) {
				mAttributeInfoArray = new AttributeInfo[attrCount];

				for (int i = 0; i < attrCount; i++) {
					if (other.mAttributeInfoArray[i] != null)
						mAttributeInfoArray[i] = other.mAttributeInfoArray[i];
				}
			}

			int elementCount = ELEMENT_COUNT();
			if (other.mChildren != null)
				mChildren = new SchemaPairs(other.mChildren);

			// copy over the children info
			if (other.mChildrenInfoArray != null) {
				mChildrenInfoArray = new ChildRelnInfo[elementCount];

				for (int i = 0; i < elementCount; i++) {
					if (other.mChildrenInfoArray[i] != null)
						mChildrenInfoArray[i] = other.mChildrenInfoArray[i];
				}
			}

			// copy over the for foreign children info
			if (other.mForeignChildren != null) {
				int nSize = other.mForeignChildren.size();
				// TODO: JavaPort Is there an equivalent Java syntax for the
				// next two lines?
				// mForeignChildren = new SchemaPairs(nSize);
				// mForeignChildren.defaultValue(null);

				for (int i = 0; i < nSize; i++) {
					ChildRelnInfo copy = null;
					if (other.mForeignChildren.value(i) != null)
						copy = (ChildRelnInfo) other.mForeignChildren.value(i);

					mForeignChildren.put(other.mForeignChildren.key(i), copy);
				}
			}
		}
	}

	/**
	 * Add an attribute relationship
	 * @param eTag
	 * @param oAttr
	 * @param nVersionIntro
	 * @param nVersionDep
	 * @param nAvailability
	 * @param descriptionResID
	 */
	void addAttr(int eTag, Attribute oAttr, int nVersionIntro, int nVersionDep,
	         int nAvailability, int descriptionResID) {
		if (mAttributes == null) {
			mAttributes = new SchemaPairs();
			// mAttributes.defaultValue(null);

			assert (mAttributeInfoArray == null);
			mAttributeInfoArray = new AttributeInfo[ATTR_COUNT()];
		}

		int nSlot;

		mAttributes.put(eTag, oAttr);

		nSlot = ATTR_OFFSET(eTag);

		assert (nSlot < ATTR_COUNT());
		assert (nSlot >= 0);

		mAttributeInfoArray[nSlot] = new AttributeInfo(oAttr, nVersionIntro,
				nVersionDep, nAvailability, descriptionResID);

		// Test our macro
		assert (ATTR_OFFSET(mnAttributeMin) == 0);
	}

	/**
	 * Add a child relationship.
	 * @param eTag
	 * @param oChildReln
	 * @param nVersionIntro
	 * @param nVersionDep
	 * @param nAvailability
	 * @param descriptionResID
	 */
	void addChild(int eTag, ChildReln oChildReln, int nVersionIntro,
			int nVersionDep, int nAvailability, int descriptionResID) {

		if (mChildren == null) {
			mChildren = new SchemaPairs();
			// mChildren.defaultValue(null);

			assert (mChildrenInfoArray == null);
			mChildrenInfoArray = new ChildRelnInfo[ELEMENT_COUNT()];

		}

		int nSlot;

		mChildren.put(eTag, oChildReln);

		nSlot = ELEMENT_OFFSET(eTag);

		assert (nSlot < ELEMENT_COUNT());
		assert (nSlot >= 0);

		mChildrenInfoArray[nSlot] = new ChildRelnInfo(oChildReln,
				nVersionIntro, nVersionDep, nAvailability, descriptionResID);
	}

	/**
	 * Add a relationshio that gets stored in a foreign schema.
	 * @param eTag
	 * @param oChildReln
	 * @param nVersionIntro
	 * @param nVersionDep
	 * @param nAvailability
	 * @param descriptionResID
	 */
	void addForeignChild(int eTag, ChildReln oChildReln, int nVersionIntro,
			int nVersionDep, int nAvailability, int descriptionResID) {
		if (mForeignChildren == null) {
			mForeignChildren = new SchemaPairs();
			// mForeignChildren.defaultValue(null);
		}

		if (mChildren == null) {
			mChildren = new SchemaPairs();
			// mChildren.defaultValue(null);
		}

		mChildren.put(eTag, oChildReln);

		// now add the version info
		ChildRelnInfo info = new ChildRelnInfo(oChildReln, nVersionIntro,
				nVersionDep, nAvailability, descriptionResID);

		mForeignChildren.put(eTag, info);
	}

	private int ATTR_COUNT() {
		return mnAttributeMax - mnAttributeMin + 1;
	}

	private int ATTR_OFFSET(int _e) {
		return _e - mnAttributeMin;
	}

	private int ELEMENT_COUNT() {
		return mnElementMax - mnElementMin + 1;
	}

	private int ELEMENT_OFFSET(int _e) {
		return _e - mnElementMin;
	}

	/**
	 * Accessor returning the attribute information
	 * 
	 * @return a structure containing all the attribute information
	 */
	public AttributeInfo getAttributeInfo(int eTag) {
		if (mAttributeInfoArray == null || eTag > mnAttributeMax
				|| eTag < mnAttributeMin)
			return null;

		return mAttributeInfoArray[eTag - mnAttributeMin];
	}

	/**
	 * Accessor returning the child relationship information
	 * 
	 * @return a structure containing all the child relationship information
	 */
	public ChildRelnInfo getChildRelnInfo(int eTag) {
		if (mChildrenInfoArray == null || eTag > mnElementMax
				|| eTag < mnElementMin) {
			if (mForeignChildren == null)
				return null;

			// check for foreign elements
			return (ChildRelnInfo)mForeignChildren.get(eTag);
		}
		return mChildrenInfoArray[eTag - mnElementMin];
	}

	/**
	 * Accessor returning the set of valid attribute names
	 * 
	 * @return array of pointers to attribute names
	 */
	public SchemaPairs getValidAttributes() {
		return mAttributes;
	};

	/**
	 * Accessor returning the set of valid children
	 * 
	 * @return a structure containing names and occurrence numbers of the valid
	 *         children.
	 */
	public SchemaPairs getValidChildren() {
		return mChildren;
	};
}
