/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */
package com.adobe.xfa;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

import com.adobe.xfa.ut.ExFull;


/**
 * Defines the parameters required to get/set a script property
 * 
 * @exclude from published api -- Mike Tardif, May 2006.
 */
public final class ScriptPropObj {
	private final String msName;	// interned, may be null
	private final int meParamType;
	private final int mnXFAVersion; // intro version
	private final int mnAvailability; // 0 means deprecated
	private final int mnVersionDep; // deprecated version
	
	private final Class<?> mClass;
	
	private final Method mGetMethod;
	private final boolean mbGetHasDependencyTracker;
	private final Method mSetMethod;
	

	public ScriptPropObj(Class<?> clazz, String sName, String sGetFunc, String sSetFunc, 
	            		int eParamType, int nXFAVersion, int nAvailability, int nVersionDep) {
		
		assert clazz != null;
		assert sGetFunc != null || sSetFunc != null;
		
		mClass = clazz;
		msName = sName;
		meParamType = eParamType;
		mnXFAVersion = nXFAVersion;
		mnAvailability = nAvailability;
		mnVersionDep = nVersionDep;
		
		Method m = null;
		boolean bGetHasDependencyTracker = false;
		if (sGetFunc != null) {
			try {
				m = mClass.getMethod(sGetFunc, Obj.class, Arg.class);
			}
			catch (NoSuchMethodException ex) {
			}
		}		
		
		// Only look for a getter that includes a DependencyTracker parameter if we didn't
		// find a getter without one since this there can only be one or the other. Since
		// a getter without a dependency tracker is the more common case, it is tested first
		// to avoid excessive exceptions being thrown at initialization.
		if (sGetFunc != null && m == null) {
			try {
				m = mClass.getMethod(sGetFunc, Obj.class, Arg.class, DependencyTracker.class);
				bGetHasDependencyTracker = true;
			}
			catch (NoSuchMethodException ex) {
				assert false;
			}
		}
		mGetMethod = m;
		mbGetHasDependencyTracker = bGetHasDependencyTracker;
		
		m = null;
		if (sSetFunc != null) {
			try {
				m = mClass.getMethod(sSetFunc, Obj.class, Arg.class);
			}
			catch (NoSuchMethodException ex) {
			}
		}
		mSetMethod = m;
	}
	
	public boolean invokeGetProp(Obj scriptThis, Arg oRetValue, DependencyTracker dependencyTracker) {
		
		boolean bError = false;
		try {
			
			if (mbGetHasDependencyTracker)
				mGetMethod.invoke(this, scriptThis, oRetValue, dependencyTracker);
			else
				mGetMethod.invoke(this, scriptThis, oRetValue);
			
			// ensure the return type is correct
			assert getParamType() == Arg.INVALID || oRetValue.isCompatibleWith(getParamType());
			
		} catch (IllegalAccessException e3) {
			bError = true;
		} catch (InvocationTargetException e2) {
			Throwable t = e2.getCause();
			if (t instanceof ExFull)
			    throw (ExFull) t;
			
			bError = true;
		}
		
		return bError;
	}
	
	public boolean invokeSetProp(Obj scriptThis, Arg oRetValue) {
		assert hasSetter();
		
		boolean bError = false;
		try {
			mSetMethod.invoke(scriptThis, scriptThis, oRetValue);
		} catch (IllegalAccessException e3) {
			bError = true;
		} catch (InvocationTargetException e2) {
			Throwable t = e2.getCause();
			if (t instanceof ExFull)
			    throw (ExFull) t;
			
			bError = true;
		}
		
		return bError;
	}
	
	public boolean invokePermsFunc(Obj scriptThis) {
		
		// For properties, the permission function is always ObjScript#setPropPermsCheck(Obj),
		// so we call it directly rather than invoking through Reflection.
			
		return ObjScript.setPropPermsCheck(scriptThis);
	}
	
	public boolean hasGetter() {
		return mGetMethod != null;
	}
	
	public boolean hasSetter() {
		return mSetMethod != null;
	}

	public int getXFAVersion() {
		return mnXFAVersion;
	}

	public int getAvailability() {
		return mnAvailability;
	}

	public int getVersionDeprecated() {
		return mnVersionDep;
	}

	public int getParamType() {
		return meParamType;
	}

	public String getName() {
		return msName;
	}
}
