/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */
// Adobe Patent or Adobe Patent Pending Invention Included Within this File
package com.adobe.xfa;

import java.io.ByteArrayOutputStream;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.List;

import com.adobe.xfa.service.canonicalize.Canonicalize;
import com.adobe.xfa.ut.FindBugsSuppress;
import com.adobe.xfa.ut.StringUtils;

/**
 *
 * @exclude from published api.
 */
public final class XMLMultiSelectNode extends ProtoableNode {
	public XMLMultiSelectNode(Element parent, Node prevSibling) {
		super(parent, prevSibling, null, XFA.XMLMULTISELECTNODE,
				XFA.XMLMULTISELECTNODE, null, XFA.XMLMULTISELECTNODETAG,
				XFA.XMLMULTISELECTNODE);
	}

	/**
	 * Add an item under the top element for this node
	 * 
	 * @param sItem
	 *            a string containing the new item.
	 * @param sItemsName
	 *            a string containing the name for this new item.
	 */
	public void addValue(String sItem, String sItemsName) {
		if (StringUtils.isEmpty(sItem))
			return;
		
		// This is an XML element that would only appear in the XML DOM,
		// so we need to create it in a way that doesn't cause validation.
		Element oNewItem = new Element(this, null);
		oNewItem.setDOMProperties(null, sItemsName, sItemsName, null);

		// watson bug 1523632,  the value changed notification should come from the content node
		// not this node.  This way the dependency tracking code will pick up changes to the multi select list
		Element poParent = getXMLParent();
		if (poParent != null)
			poParent.notifyPeers(VALUE_CHANGED, "", null);

	}

	/**
	 * Clear items under the top element for this node.
	 */
	public void clearValues() {
		Node node = getFirstXMLChild();
		while (node != null) {
			Node next = node.getNextXMLSibling();
			removeChild(node);
			node = next;
		}
	}

	/**
	 * @see ProtoableNode#createProto(Element, boolean)
	 */
	public ProtoableNode createProto(Element pParent, boolean bFull) {
		return (ProtoableNode)clone(pParent, true);
	}

	/**
	 * @see Element#getDeltas(Element, XFAList)
	 * @exclude from public api.
	 */
	public void getDeltas(Element delta, XFAList list) {
//		 Adobe patent application tracking # B252, entitled METHOD AND SYSTEM TO PERSIST STATE, inventors: Roberto Perelman, Chris Solc, Anatole Matveief, Jeff Young, John Brinkman
//		 Adobe patent application tracking # B322, entitled METHOD AND SYSTEM TO MAINTAIN THE INTEGRITY OF A CERTIFIED DOCUMENT WHILE PERSISTING STATE IN A DYNAMIC FORM, inventors: Roberto Perelman, Chris Solc, Anatole Matveief, Jeff Young, John Brinkman
		if (isSameClass(delta) && list != null)	{
			Element parent = getXFAParent();
			Element deltaParent = delta.getXFAParent();
			Delta newDelta = new Delta(parent, deltaParent, this, delta, "");
			list.append(newDelta);
		}
	}

	/**
	 * @exclude from published api.
	 */
	public ScriptTable getScriptTable() {
		return XMLMultiSelectNodeScript.getScriptTable();
	}
	
	/**
	 * Get the pcData for this node.
	 * 
	 * @param bAsFragment
	 *            if true, it returns a String containing an html fragment,
	 *            false return only the text content.
	 * @return the pcData as a string.
	 */
	public String getValue(boolean bAsFragment /* = false */, boolean bSuppressPreamble /* = false */) {
		StringBuilder sReturn = new StringBuilder();
		List<String> oItems = new ArrayList<String>();

		getValuesFromDom(oItems, this, bAsFragment, bSuppressPreamble);

		for (int i = 0; i < oItems.size(); i++) {
			if (i != 0)
				sReturn.append('\n');
			sReturn.append(oItems.get(i));
		}

		return sReturn.toString();
	}

	/**
	 * Get the pcData for this node as a list of selected items.
	 * 
	 * @param oSelectionList
	 *            a list of all items
	 */
	public void getValues(List<String> oSelectionList) {
		getValuesFromDom(oSelectionList, this, false, false);
	}

	public void getValuesFromDom(List<String> xmlTextValues, Node oDomNode,
			boolean bAsFragment, boolean bSuppressPreamble) {

		if (bAsFragment) {
			// Use UTF-8 converter to avoid losing asian characters
			ByteArrayOutputStream oFragmentStream = new ByteArrayOutputStream();
			DOMSaveOptions oOptions = new DOMSaveOptions();
			// XFAPlugin Vantive bug#595482 Convert CR and extended characters
			// to entity
			// references. Acrobat prefers these to raw utf8 byte data.
			oOptions.setDisplayFormat(DOMSaveOptions.RAW_OUTPUT);
			oOptions.setSaveTransient(true);
			oOptions.setEntityChars("\r");
			oOptions.setRangeMin('\u007F');
			oOptions.setRangeMax('\u00FF');
			oOptions.setExcludePreamble(bSuppressPreamble);
			
			oDomNode.getOwnerDocument().saveAs(oFragmentStream, oDomNode, oOptions);

			try {
				xmlTextValues.add(oFragmentStream.toString("UTF-8"));
			} catch (UnsupportedEncodingException e) {
				// Not possible - UTF-8 is always supported
			}
		} 
		else {
			// return a newline delimited list of selected item values

			if (oDomNode instanceof TextNode) {
				TextNode oDomText = (TextNode) oDomNode;
				if (!oDomText.isXMLSpace()) {
					String sValue = oDomText.getValue();
					if (!StringUtils.isEmpty(sValue)){
						String sSeparator = "\n";
						String[] tokens = sValue.split(sSeparator);
						for (String sToken: tokens)
						xmlTextValues.add(sToken);
					}
				}
			}

			Node child = oDomNode.getFirstXMLChild();
			while (child != null) {
				getValuesFromDom(xmlTextValues, child, bAsFragment, bSuppressPreamble);
				child = child.getNextXMLSibling();
			}
		}
	}

	/**
	 * Set the pcdata for this node.
	 * 
	 * @param sData
	 *            a string containing the new pcdata.
	 */
	public void setValue(String sData) {
		List<String> oSelectionList = new ArrayList<String>();

		// Store in a jfStorage to avoid string parsing
		if (!StringUtils.isEmpty(sData)) {
			String[] tokens = sData.split("\n");

			for (int i = 0; i < tokens.length; i++) {
				oSelectionList.add(tokens[i]);
			}
		}

		setValuesToDom(oSelectionList, this);
	}

	/**
	 * Set the pcdata for this node.
	 * 
	 * @param oSelectionList
	 *            a jfStorage of strings containing the new values.
	 */
	public void setValues(List<String> oSelectionList) {
		setValuesToDom(oSelectionList, this);
	}

	public void setValuesToDom(List<String> oSelectionList, Element oDomNode) {
		// <listBoxFieldName>
		// <itemsName>A</itemsName>
		// <itemsName>B</itemsName>
		// </listBoxFieldName>

		// UNDO(XFAXMLUndo, (this));

		int nNumberSelected = oSelectionList.size();

		// get the number of children under the one-of child, #xml
		Node oChild = getFirstXMLChild();

		String aElementName = "";
		if (oChild == null) {
			// painful - just use 'value' or muck about to find
			// save items name?
			aElementName = XFA.VALUE;
		} else {
			aElementName = oChild.getName();
			if (aElementName == "")
				aElementName = XFA.VALUE;
		}

		// blank out the #xml node
		// maybe not the best strategy, but for now, get rid of them
		// for (int i = 0; i < numChildren; i++)
		while (oChild != null) {
			Node oNext = oChild.getNextXMLSibling();
			removeChild(oNext);
			oChild = oNext;
		}

		// add all the selected items as elements
		for (int i = 0; i < nNumberSelected; i++) {
			
			// This is an XML element that would only appear in the XML DOM,
			// so we need to create it in a way that doesn't cause validation.
			Element oNewItem = new Element(oDomNode, null);
			oNewItem.setDOMProperties(null, aElementName, aElementName, null);
			
			getModel().createTextNode(oNewItem, null, oSelectionList.get(i));
		}
		// After a "set" operation we won't be a default property anymore, and
		// we'll no longer delegate to any protos.
		makeNonDefault(false);
		
		// watson bug 1523632,  the value changed notification should come from the content node
		// not this node.  This way the dependency tracking code will pick up changes to the multi select list
		Element poParent = getXMLParent();
		if (poParent != null)
			poParent.notifyPeers(VALUE_CHANGED, "", null);

	}

	/**
	 * Cast this node to a string value.
	 * 
	 * @return the string representing the pcdata.
	 */
	public String toString() {
		return getValue(false,false);
	}

	/**
	 * Override of Element.compareVersions.
	 * 
	 * @exclude from published api.
	 */
	@FindBugsSuppress(code="ES")
	protected boolean compareVersions(Node oRollback, Node oContainer,
									   Node.ChangeLogger oChangeLogger /* null */, Object oUserData /* null */ ) {
		boolean bMatches = super.compareVersions(oRollback, oContainer, oChangeLogger, oUserData);
		if (bMatches == false && oChangeLogger == null)
			return false;		// performance optimization
		
		if (getClassAtom() != oRollback.getClassAtom()) {
			// Rollback must also be a multiSelectNode.  (If it's not, XFANodeImpl will have already flagged 
			// it as non-matching anyway.)
			return false;
		}
			
		// 
		// XHTML needs canonicalizing before comparing
		//
		String sCanonicalSource = "";
		String sCanonicalRollback = "";
		
		Canonicalize c = new Canonicalize(this, false, true);
		byte [] pBuffer = c.canonicalize(Canonicalize.CANONICALWITHOUT, null);
		try {
			sCanonicalSource = new String(pBuffer, "UTF-8");
		}
		catch (UnsupportedEncodingException ex) {
			// not possible - UTF-8 is always supported
		}
		
		Canonicalize cRollBack = new Canonicalize(oRollback, false, true);
		pBuffer = cRollBack.canonicalize(Canonicalize.CANONICALWITHOUT, null);
		try {
			sCanonicalRollback = new String(pBuffer, "UTF-8");
		}
		catch (UnsupportedEncodingException ex) {
			// not possible - UTF-8 is always supported
		}
		
		if (!sCanonicalSource.equals( sCanonicalRollback ))
		{
			bMatches = false;
			if (oChangeLogger != null)
				logValueChangeHelper(oContainer, sCanonicalSource, oChangeLogger, oUserData);
		}
		
		return bMatches;
	}
}
