/*
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material is
 * strictly forbidden unless prior written permission is obtained from Adobe
 * Systems Incorporated.
 */

package com.adobe.xfa.agent.formprocessor;


import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.List;
import java.util.StringTokenizer;

import com.adobe.xfa.AppModel;
import com.adobe.xfa.ArrayNodeList;
import com.adobe.xfa.Chars;
import com.adobe.xfa.DOMSaveOptions;
import com.adobe.xfa.Document;
import com.adobe.xfa.Element;
import com.adobe.xfa.LogMessage;
import com.adobe.xfa.LogMessenger;
import com.adobe.xfa.Node;
import com.adobe.xfa.NodeList;
import com.adobe.xfa.STRS;
import com.adobe.xfa.TextNode;
import com.adobe.xfa.XFA;
import com.adobe.xfa.agent.Agent;
import com.adobe.xfa.agent.MessageHandlerXML;
import com.adobe.xfa.configuration.ConfigurationKey;
import com.adobe.xfa.configuration.ConfigurationModel;
import com.adobe.xfa.configuration.ConfigurationValue;
import com.adobe.xfa.connectionset.ConnectionSetModel;
import com.adobe.xfa.connectionset.ConnectionSetModelFactory;
import com.adobe.xfa.data.DataModel;
import com.adobe.xfa.data.DataModelFactory;
import com.adobe.xfa.localeset.LocaleSetModelFactory;
import com.adobe.xfa.service.href.HrefService;
import com.adobe.xfa.service.storage.XMLStorage;
import com.adobe.xfa.template.TemplateModel;
import com.adobe.xfa.template.TemplateModelFactory;
import com.adobe.xfa.ut.BooleanHolder;
import com.adobe.xfa.ut.ExFull;
import com.adobe.xfa.ut.FindBugsSuppress;
import com.adobe.xfa.ut.MsgFormatPos;
import com.adobe.xfa.ut.ResId;


/**
 * A class to facilitate
 * the loading of XML documents containing an XFA template.
 * It is intended that this class parallels the functionality
 * offered by XFA's "xmlform -pa" application.
 * <p>
 * Here's a snippet of code illustrating the use of a
 * {@link #FormProcessor()}.
 * <pre><code>
 * 
 *  import com.adobe.xfa.formprocessor.FormProcessor;
 *  ...
 *  FormProcessor pa = new FormProcessor();
 *  try  {
 *      InputStream template = ...
 *      InputStream config = ...
 *      InputStream data = ...
 *      OutputStream out = ...
 *      OutputStream log = ...
 *      int ok = pa.execute(template, config, data, null, out, log, null, null);
 *      if (ok == FAIL) {
 *          // look at resulting log stream.
 *      }
 *		else {
 *          // look at resulting out stream.
 *      }
 *  } catch (IOException o) {
 *      ...
 *  } catch (ExFull e) {
 *      System.err.println(e.toString());
 *  }
 *  
 * </code></pre>
 *  
 * @exclude from published api.
 *  
 */
public final class FormProcessor extends Agent {

	/**
	 * The default size ({@value}), in filesystem bytes of memory,
	 * for the cache used to hold loaded template fragments.
	 * <p>
	 * Server-based applications may prefer to use custom values instead --
	 * this being a relatively small value.
	 * @see	#FormProcessor(int)
	 */
	public static final int HREF_CACHE_SIZE = 50 * 1024;

	// FileType: XFA-enabled PDF
	//private static final int PDFwithXFA = 0;

	// FileType: PDF containing acroforms data in the XFA key
	//private static final int PDFwithXFAAcroform = 1;

	// FileType: Standard PDF
	//private static final int PDFwithoutXFA = 2;

	// FileType: An open-password-protected PDF
	// (can't determine if XFA-enabled or not).
	//private static final int PDFwithOpenPassword = 3;

	// FileType: A permissions-password-protected PDF
	// (can't determine if XFA-enabled or not).
	//private static final int PDFwithPermPassword = 4;

	// FileType: Plain XML (not in XDP format)
	private static final int XML = 5;

	// FileType: XDP format
	private static final int XDP = 6;

	// FileType: Some other file type.
	private static final int Unknown = 7;

	/**
	 * Instantiates an XFA form processor.
	 */
	public FormProcessor() {
		this(HREF_CACHE_SIZE);
	}

	/**
	 * Instantiates an XFA form processor with a template
	 * fragment cache of specified size.
	 *
	 * @param cacheSize	the size of the template fragment cache.
	 * <p>
	 * This is the maximal size, in filesize bytes of memory,
	 * allowed of all loaded template fragments.  Once exceeded,
	 * the DOM of loaded template fragments gets discarded on a
	 * fifo basis.
	 */
	public FormProcessor(int cacheSize) {
		//
		// Setup an app model and its factories.
		//
		super();
		AppModel oAppModel = getAppModel();
		TemplateModelFactory oTemplateFactory = new TemplateModelFactory();
		oAppModel.addFactory(oTemplateFactory);
	// JavaPort: not yet available.
	//	SourceSetModelFactory oSourceSetFactory;
	//	getAppModel()).addFactory(oSourceSetFactory);
		DataModelFactory oDataFactory = new DataModelFactory();
		oAppModel.addFactory(oDataFactory);
		ConnectionSetModelFactory oConnectionSetFactory
										= new ConnectionSetModelFactory();
		oAppModel.addFactory(oConnectionSetFactory);
		LocaleSetModelFactory oLocaleSetFactory = new LocaleSetModelFactory();
		oAppModel.addFactory(oLocaleSetFactory);
		//
		// Tell it about the href resolver service.
		//
		if (cacheSize < 0)
			cacheSize = HREF_CACHE_SIZE;
		HrefService oHrefService = new HrefService(XFA.PRESENT, cacheSize);
		oAppModel.setHrefHandler(oHrefService);
	}

	/**
	 * @exclude from published api.
	 */
	public String getConfigSchemaName() {
	    return XFA.PRESENT;
	}

	String getConfigValue(String somExpr) {
		try {
			Node oContext = getContextNode();
			Node oValue = oContext.resolveNode(somExpr);
			if (oValue instanceof ConfigurationValue) {
				ConfigurationValue oConfigValue = (ConfigurationValue) oValue;
				return oConfigValue.getValue().getAttrValue();
			}
			throw new ExFull(ResId.InvalidSOMException, somExpr);
		} catch (ExFull oEx) {
			logException("getConfigValue", oEx);
			throw oEx;
		}
	}

	private ByteArrayOutputStream moXMLMsgStream = null;

	private void XMLFormAgentInit(String sDefaultConfig) {
		//
		// Create and add a MessageHandlerXML object, that we can use later
		// to retrieve all the errors/messages that was logged during the
		// process This is appended to the CSA redirection alreay in place
		//
		moXMLMsgStream = new ByteArrayOutputStream();
		MessageHandlerXML oXMLHandler = new MessageHandlerXML(moXMLMsgStream);
		LogMessenger oLogMessenger = getAppModel().getLogMessenger();
		oLogMessenger.addHandler(oXMLHandler, "");
		if (sDefaultConfig.length() > 0) {
			try {
				ByteArrayInputStream oFixStream
						= new ByteArrayInputStream(sDefaultConfig.getBytes(Document.Encoding));
				setXFAInput(Agent.CONFIG, oFixStream);
				loadXFAModel(Agent.CONFIG, false);
			} catch (ExFull oEx) {
				logException("loadDefaultConfig", oEx);
			} catch (UnsupportedEncodingException e) {
				logException("loadDefaultConfig", new ExFull(e));
            }
		}

	}

	void clearMessages() {
	    //  
	    //  Clear what's there and put back the processing info and root tag.
	    //  Make sure not do do this if the stream is empty (never written to)
	    //  or we'll get this twice.
	    //  
	    if (moXMLMsgStream.size() > 0) {
	        moXMLMsgStream.reset();
			try {
				moXMLMsgStream.write("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n<log>".getBytes("UTF-8"));
			} catch (IOException e) {
				throw new ExFull(e);
			}
	    }
	}

	/**
	 * Executes a form as directed by the specified inputs.
	 * @param oTemplate the template input stream.
	 * @param oXCI the configuration input stream. 
	 * @param oData the data input stream, if available.
	 * @param oXDC the XFA XDC input stream if available. This is not currently supported.
	 * @param oOutput the output stream.  Currently, only XDP output is supported.
	 * @param oInfo the info output stream.
	 * @param masterPassword the encrypted PDF master password, if applicable.
	 * @param userPassword the encrypted PDF user password, if applicable.
	 * @return the completion code, e.g., {@link Agent#SUCCESS}, {@link Agent#FAIL}, ...
	 * @throws ExFull upon error.
	 */
	public int execute(InputStream oTemplate,
					InputStream oXCI, 
				    InputStream oData,
					InputStream oXDC,
					OutputStream oOutput,
					OutputStream oInfo,
					String masterPassword,
					String userPassword) throws ExFull {
		//	Init
		//	This will do log redirection and create a XFAPresentationAgent
		//	NOTE: FormServer always sends the definitive xci file
		XMLFormAgentInit("");
		//
		//	Did we fail in creating (config errors for example)
		//
		int initCode = getCompletionStatus(getMessenger());
		if (initCode == FAIL) {
			//
			// Attempt to return procInfo with errors.
			//
			if (getMessenger() != null) {
				getMessenger().flush();
				String sLogMessages = moXMLMsgStream.toString();
				if (sLogMessages.length() > 0)
					getProcessingInfoErrors(oInfo, sLogMessages);
			}
			return initCode;
		}
		//
		//	Everything is set up, keep going
		//
		byte[] oResult;
		//
		// Load config
		//
		loadPacket(oXCI, CONFIG, false);
		if (getConfigModel() == null && getMessenger() != null) {
			getMessenger().flush();
			//
			// Attempt to return procInfo with errors.
			//
			String sLogMessages = moXMLMsgStream.toString();
			if (sLogMessages.length() > 0)
				getProcessingInfoErrors(oInfo, sLogMessages);
			return FAIL;
		}
		//
		// We need only update config when generating PDFs!
		// add passwords to the xci file if they are given
		//
		boolean bEncrypt = false;
		if (masterPassword != null && masterPassword.length() > 0) {
			// Set master password
			getConfigModel().assignNode("present.pdf.encryption.masterPassword", masterPassword, Node.CREATE_REPLACE);
			bEncrypt = true;
		}
		if (userPassword != null && userPassword.length() > 0) {
			// Set user password
			getConfigModel().assignNode("present.pdf.encryption.userPassword",
										userPassword, Node.CREATE_REPLACE);
			bEncrypt = true;
		}
		//
		// if we have passwords, then we also need to encrypt
		//
		if (bEncrypt == true)
			getConfigModel().assignNode("present.pdf.encryption.encrypt", "1", Node.CREATE_REPLACE);
		//
		// Fire $config.ready event
		//
		getConfigModel().ready(false);
		Node oContextNode = getContextNode();
		//
		// Set context for config model.
		//
		getConfigModel().setContext(oContextNode);
		//
		// Set up our temp dir where our resolver will write temporary files.
		//
		Node oTempDir = null;
		if (oContextNode != null)
			oTempDir = oContextNode.resolveNode("temp.uri");
		if (oTempDir instanceof ConfigurationValue
										&& oTempDir.getXFAChildCount() == 1) {
			ConfigurationValue oConfigurationValue
										= (ConfigurationValue) oTempDir;
			String sTempDirStr = oConfigurationValue.getValue().toString();
			if (! (new File(sTempDirStr)).isDirectory()) {
				MsgFormatPos oBadTempDirPos
					= new MsgFormatPos(ResId.XFAAgentBadTempDir, sTempDirStr);
				LogMessage oBadTempDirMessage
					= new LogMessage(oBadTempDirPos, LogMessage.MSG_WARNING);
				getMessenger().sendMessage(oBadTempDirMessage);
			}
		}
		//
		// Set XDP Options, and check if we should save as XDP/merged XDP and if 
		// the rendered output should be embedded
		//
		BooleanHolder oSaveAsXDP = new BooleanHolder();
		BooleanHolder oIsMergedXDP = new BooleanHolder();
		BooleanHolder oEmbedRenderedOutput = new BooleanHolder();
		checkXDPOptions(oSaveAsXDP, oIsMergedXDP, oEmbedRenderedOutput);
		//
		// If the input template is a PDF file,
		// just do the data replacement in the file.
		//
		if (oTemplate.markSupported()) {
			oTemplate.mark(4);
			try {
				if (oTemplate.read() == '%'
						&& oTemplate.read() == 'P'
							&& oTemplate.read() == 'D'
								&& oTemplate.read() == 'F') {
				// JavaPort: the input template is never PDF file.
				//	boolean bUpdated = updatePDF(oTemplate, oData, oOutput, oInfo);
					throw new ExFull(ResId.UNSUPPORTED_OPERATION, "FormProcessor#execute - updatePDF");
				//	if (! bUpdated) {
				//		// Capture errors raised by updatePDF().
				//		if (getMessenger() != null) {
				//			getMessenger().flush();
				//			String sLogMessages = moXMLMsgStream.toString();
				//			if (sLogMessages.length() > 0)
				//				getProcessingInfoErrors(oInfo, sLogMessages);
				//		}
				//	}
				//	return getCompletionStatus(getMessenger());
				}
				oTemplate.reset();
			} catch (IOException e) {
				throw new ExFull(e);
			}
		}
		try {
			//
			// Load XDC + fire event
			//
			if (oXDC != null)
				throw new ExFull(ResId.UNSUPPORTED_OPERATION, "FormProcessor#execute - load XDC packet");
		// JavaPort: stuff commented out awaiting support for XDC Model.
		//	if (loadPacket(oXDC, XDC, false) && getXDCModel() != null) {
		//		getXDCModel().ready();
		//
		//		// XFATemplate can include RenderCache Processing Instructions
		//		// (PIs).  These PIs are meant to be used ONLY by drivers that
		//		// support positioned text using formatted fragments.  If the
		//		// device does not support handling text as formatted fragments,
		//		// then we will turn off the use of renderCache, by updating
		//		// the config option.  The Template load looks at this config
		//		// option when loading the document and will remove these PIs
		//		// if present.cache.renderCache.enable=0
		//		String sOption
		//				= getXDCModel().getOption("boilerplateTextRetrieval");
		//		if (sOption.length() > 0
		//					&& ! sOption.equals(STRS.FORMATTEDFRAGMENTS)) {
		//			getConfigModel().assignNode("present.cache.renderCache.enable", 
		//								"0", Node.CREATE_REPLACE);
		//		}
		//	}
			//
			// Load template + fire event
			//
			if (loadPacket(oTemplate, TEMPLATE, false)
											&& getTemplateModel() != null) {
				getTemplateModel().ready(false);
			}
			//
			// Load data + fire event
			//
			if (loadPacket(oData, DATA, oIsMergedXDP.value)
											&& getDataModel() != null) {
				getDataModel().ready(false);
			}
		} catch (ExFull oEx) {
			logException("load", oEx);
			if (getMessenger() != null) {
				getMessenger().flush();
				String sLogMessages = moXMLMsgStream.toString();
				if (sLogMessages.length() > 0)
					getProcessingInfoErrors(oInfo, sLogMessages);
			}
			return getCompletionStatus(getMessenger());
		}
		//
		// Save as XDP, no rendering needed
		//
		if (oSaveAsXDP.value && ! oEmbedRenderedOutput.value) {
			List<String> oPacketList = new ArrayList<String>();
			if (oIsMergedXDP.value) {
			// JavaPort: merge not available.
			//	jfTraceHandler::Timer oMergeOnlyTimer(jfTraceHandler::XFAPA_RENDER_TIMING);
			//	mergedXDP();
			//	//
			//	//	Only ouput packages asked for in the config XDP packet list
			//	//	"*" means all
			//	//	Should "*" with excludes be allowed
			//	//
			//	ArrayList oXDPList = new ArrayList();
			//	StringTokenizer sXDPPackets
			//			= new StringTokenizer(getConfigPacketList());
			//	boolean	bStar = false;
			//	while (sXDPPackets.hasMoreTokens()) {
			//		String sToken = sXDPPackets.nextToken();
			//		if (sToken.equals("*")) {
			//			bStar = true;
			//		}
			//		oXDPList.add(sToken);
			//	}
			//	ArrayList oLoadedPackages = getPackets();
			//	boolean bAddPacket = false;
			//	for (int i = 0; i < oLoadedPackages.size(); i++) {
			//		String sPacket = (String) oLoadedPackages.get(i);
			//		//	
			//		//	If all, check for exclusions (could have "* -template"
			//		// for example)
			//		//
			//		if (bStar) {
			//			bAddPacket = true;
			//			for (int j = 0; j < oXDPList.size(); j++) {	
			//				String sXDP = (String) oXDPList.get(j);
			//				char cFirstChar = sXDP.charAt(0);
			//				if (cFirstChar == '-') {
			//					//	Swallow the '-' before comparing
			//					String sTrimmed = sXDP.substring(1);
			//					if (sPacket.equals(sTrimmed)) {
			//						bAddPacket = false;
			//						break;
			//					}
			//				}
			//			}
			//		} 
			//		else {
			//			for (int j = 0; j < oXDPList.size(); j++) {	
			//				bAddPacket = false;
			//				if (sPacket.equals(oXDPList.get(j))) {
			//					bAddPacket = true;
			//					break;
			//				}
			//			}
			//		}
			//		if (bAddPacket) {
			//			oPacketList.add(sPacket);
			//		}
			//	}
				throw new ExFull(ResId.UNSUPPORTED_OPERATION, "FormProcessor#execute - merge");
			}
			else {
				oPacketList.add("*");
			}
			oResult = exportXDP(oPacketList);
		}
		//
		// Else rendering needed.
		//
		else {
			//	Render to uri if we have one.
			String sOutputTo  = getConfigValue("output.to");
			String sOutputUri = getConfigValue("output.uri");
			if (sOutputTo.equalsIgnoreCase("uri")
											&& sOutputUri.length() > 0) {
			// JavaPort: can't render yet.
			//	oResult = doRender(oSaveAsXDP.value,
			//								sOutputUri, eCompletionCode);
				throw new ExFull(ResId.UNSUPPORTED_OPERATION, "FormProcessor#execute - render1");
				// Return the output uri bit so we don't have an empty
				// return and the app can easily see the output destination 
				// StringBuilder sRet = new StringBuilder("<output><to>");
				// sRet.append(sOutputTo);
				// sRet.append("</to><uri>");
				// sRet.append(sOutputUri);
				// sRet.append("</uri></output>");
				// oResult = sRet.toString().getBytes(Document.Encoding);
			} 
			else {
			// JavaPort: can't render yet.
			//	oResult = doRender(oSaveAsXDP.value, "", eCompletionCode);
				throw new ExFull(ResId.UNSUPPORTED_OPERATION, "FormProcessor#execute - render2");
			}
		}
		//
		// get the processingInfo
		//
		String sLogMessages = null;
		if (getMessenger() != null) {
			getMessenger().flush();
			sLogMessages = moXMLMsgStream.toString();
		}
		//
		// copy the output.  
		// Return the processing info messages even if no output,
		// since the reason for the failure can likely be found
		// in the messages.
		//
		if (oResult.length > 0) {
			try {
				if (oOutput != null)
					oOutput.write(oResult);
			} catch (IOException e) {
				logException("io", new ExFull(e));
			}
			getProcessingInfo(oInfo, sLogMessages, oIsMergedXDP.value);
		} 
		else {
			getProcessingInfoErrors(oInfo, sLogMessages);
		}
		int status = getCompletionStatus(getMessenger());
		reset();
		return status;
	}

	/*
	 *  Return the messages accumulated to this point
	 */
	String getMessages() {
		StringBuilder sLogMessages = new StringBuilder();
		//
		// Get the messages.
		//
		if (getMessenger() != null) {
			getMessenger().flush();
			sLogMessages.append(moXMLMsgStream.toString());
		}
		//
		//  Place root tag <log> if not already created.
		//
		if (sLogMessages.length() == 0)
			sLogMessages.append("<log>");
		sLogMessages.append("</log>");
		return sLogMessages.toString();
	}

	@FindBugsSuppress(code="DE")
	boolean loadPacket(InputStream oInputStream, 
						   int eType, boolean bIsMergedXDP /* =false */) {
		InputStream oStream = null;
		boolean bMustCloseStream = false;	// ensure any file stream this method opens get closed
		
		try {
			//
			// If the dataBuffer is empty, check for uri in config
			//
			if (oInputStream == null) {
				StringBuilder sURI = new StringBuilder();
				switch (eType) {
				case CONFIG:
					return false;
				case XDC:
					sURI.append(getXDCDest(false));
					if (sURI.length() == 0)
						return false;
					break;
				case DATA:
					sURI.append(STRS.DATASETS);
					break;
				case TEMPLATE:
					sURI.append(XFA.TEMPLATE);
					break;
				case SOURCESET:
					sURI.append(XFA.SOURCESET);
					break;
				case CONNECTIONSET:
					sURI.append(XFA.CONNECTIONSET);
					break;
				}
				sURI.append('.');
				sURI.append(XFA.URI);
				Node oContextNode = getContextNode();
				Node oResolvedNode = null;
				if (oContextNode != null)
					oResolvedNode = oContextNode.resolveNode(sURI.toString());
				if (oResolvedNode instanceof ConfigurationValue) {
					// Use the source specified in the config model.
					ConfigurationValue oConfigurationValue
											= (ConfigurationValue) oResolvedNode;
					String sConfigSourceFile
								= oConfigurationValue.getValue().toString();
					if (sConfigSourceFile.length() > 0)
						try {
							oStream = new FileInputStream(sConfigSourceFile);
							bMustCloseStream = true;
						} catch (IOException e) {
							throw new ExFull(e);
						}
				}
			}
			else {
				oStream = oInputStream;
			}
			if (oStream != null) {
				if (eType == DATA) {
					int eFileType = identify(oStream);
					if (eFileType == XDP) {
						// Load everything except template bits
						// Config can not be overriden here
						// Basically, anything that is not defined to be part of
						// Template set should be loaded (datasets execEvent xfdf
						// formstate xmp pdf ..)
						// See the template load section to see what is considered
						// 'template'
						StringBuilder sPackets = new StringBuilder("-");
						sPackets.append(XFA.CONFIG);
						sPackets.append(" -");
						sPackets.append(XFA.TEMPLATE);
						sPackets.append(" -");
						sPackets.append(XFA.SOURCESET);
						sPackets.append(" -");
						sPackets.append(XFA.CONNECTIONSET);
						sPackets.append(" -");
						sPackets.append(XFA.XSL);
						sPackets.append(" -");
						sPackets.append(XFA.LOCALESET);
						loadXDP(oStream, sPackets.toString(), false);
					}
					else if (eFileType == XML) {
						loadXDP(oStream, STRS.DATASETS, false);
					}
				}
				else {
					String sPackets = "";
					switch (eType) {
					case CONFIG:
						sPackets = XFA.CONFIG;
						break;
					case CONNECTIONSET:
						sPackets = XFA.CONNECTIONSET;
						break;
					case SOURCESET:
						sPackets = XFA.SOURCESET;
						break;
					// Only specific packets from the template are loaded.
					// This list is updated from time to time, when new packets are
					// added (like localeSet)
					// Note: packets added here should be excluded from
					// the data packets.
					case TEMPLATE:
						// Javaport: it's safer to only omit config instead,
						// thus avoiding filtering unknown packets.   
						// sPackets = getTemplatePackageList();
						sPackets = "-" + XFA.CONFIG + " *";
						break; 
					case XDC:
						return loadXDC(oStream);
					}
					loadXDP(oStream, sPackets, false);
				}
				return true;
			}
			return false;
		}
		finally {
			if (bMustCloseStream && oStream != null) {
				try {
					oStream.close();
				}
				catch (IOException ignored) {
				}
			}
		}
	}

	String getConfigPacketList() {
		Node oResolvedNode = null;
		Node oContextNode = getContextNode();
		if (oContextNode != null)
			oResolvedNode = oContextNode.resolveNode("xdp.packets");
		if (oResolvedNode instanceof ConfigurationValue) {
			ConfigurationValue oConfigurationValue
								= (ConfigurationValue) oResolvedNode;
			return oConfigurationValue.getValue().toString();
		}
		return "";
	}

	List<String> getPackets() {
		List<String> oPacketList = new ArrayList<String>();
		for (Node oChild = getAppModel().getFirstXFAChild(); oChild != null; oChild = oChild.getNextXFASibling()) {
			String sPacketName = oChild.getName();  // Packet name.
			oPacketList.add(sPacketName);
		}
		return oPacketList;
	}

	byte[] exportXDP(List<String> packetList) {
		ByteArrayOutputStream oMemStream = new ByteArrayOutputStream();
		try {
			// Convert the packets to a string.
			StringBuilder sPackets = new StringBuilder();	
			for (int i = 0; i < packetList.size(); i++) {
				if (i > 0)
					sPackets.append(' ');
				sPackets.append(packetList.get(i));
			}
			saveXDP(oMemStream, sPackets.toString());
		} catch (ExFull oEx) {
			logException("exportXDP", oEx);
			throw oEx;
		}
		return oMemStream.toByteArray();
	}

	String logException(String sFunctionName, ExFull oEx) {
		oEx.resolve();
		StringBuilder sMsg = new StringBuilder("XMLForm, ");
		sMsg.append(sFunctionName);
		sMsg.append(" : ");
		int nId = 0;
		for (int i = 0, n = oEx.count(); i < n; i++) {
			if (i > 0)
				sMsg.append('\n');
			nId = oEx.item(i).resId();
			sMsg.append(Integer.toString(nId));
			sMsg.append(", ");
			sMsg.append(oEx.item(i).text());
		}
		LogMessage oExceptionMessage = new LogMessage(nId, sMsg.toString(),
												LogMessage.MSG_FATAL_ERROR);
		LogMessenger oMessenger = getMessenger();
		if (oMessenger != null)
			oMessenger.sendMessage(oExceptionMessage);
		return sMsg.toString();
	}


	private void getProcessingInfo(OutputStream oInfo,
										String sLogMessages,
											boolean bIsMergedXDP) {
		//
		// Create a DOM document
		//
		AppModel app = new AppModel(null);
		Document oDoc = app.getDocument();
		app.newDOM();
		//
		// Create and append our <status> node
		//
		Element oStatus = oDoc.createElementNS("", "status", null);
		oDoc.appendChild(oStatus, false);
		//
	// JavaPort: page number and count are not available.
	//	addProcInfoPageNumber(oStatus);
	//	addProcInfoPageCount(oStatus);
		addProcInfoOutputUri(oStatus);
		addProcInfoMessages(oStatus, sLogMessages);
	// JavaPort: the following seem frivolous.
	//	addProcInfoData(oStatus, bIsMergedXDP);
		//
		// Write it all out to the output stream
		//
		ByteArrayOutputStream oOutput = new ByteArrayOutputStream();
		oDoc.saveAs(oOutput, oStatus, new DOMSaveOptions());
		if (oOutput.size() > 0) {
			try {
				if (oInfo != null)
					oInfo.write(oOutput.toByteArray());
			} catch (IOException e) {
				throw new ExFull(e);
			}
		}
	}

	private void getProcessingInfoErrors(OutputStream oInfo,
											String sLogMessages) {
		//
		// Create a DOM document
		//
		AppModel app = new AppModel(null);
		Document oDoc = app.getDocument();
		app.newDOM();
		//
		// Create and append our <status> node
		//
		Element oStatus = oDoc.createElementNS ("", "status", null);
		oDoc.appendChild(oStatus);
		addProcInfoMessages(oStatus, sLogMessages);
		//
		// Write it all out to the output stream
		//
		ByteArrayOutputStream oOutput = new ByteArrayOutputStream();
		oDoc.saveAs(oOutput, oStatus, new DOMSaveOptions());
		if (oOutput.size() > 0) {
			try {
				if (oInfo != null)
					oInfo.write(oOutput.toByteArray());
			} catch (IOException e) {
				throw new ExFull(e);
			}
		}
	}

	private void addProcInfoOutputUri(Element oParent) {
		Document oDoc = oParent.getOwnerDocument();
		Element oOut = oDoc.createElementNS("", "output", null);
		//
		//  Create <to> under output
		//
		Element oOutTo = oDoc.createElementNS("", "to", null);
		TextNode oOutToText = oDoc.getAppModel().createTextNode(null, null,
												getConfigValue("output.to"));
		oOutTo.appendChild(oOutToText, false);
		oOut.appendChild(oOutTo, false);
		//
		//  Create <uri> under <output>
		//
		Element oOutUri = oDoc.createElementNS("", "uri", null);
		TextNode oOutUriText = oDoc.getAppModel().createTextNode(null, null,
												getConfigValue("output.uri"));
		oOutUri.appendChild(oOutUriText, false);
		oOut.appendChild(oOutUri, false);
		//
		//  Add the <output> node to the parent.
		//
		oParent.appendChild(oOut, false);
	}

	private void addProcInfoMessages(Element oParent, String sLogMessages) {
		if (sLogMessages.length() == 0)
			return;
		//
		// Create and append our <messages> node
		//
		Document oDoc = oParent.getOwnerDocument();
		Element oMessages = oDoc.createElementNS("", "messages", null);
		//
		// Create and append our <validationErrors> node
		//
		Element oValidationErrors
					= oDoc.createElementNS("", "validationErrors", null);
		//
		// Create and append our <validationWarnings> node
		//
		Element oValidationWarnings
					= oDoc.createElementNS("", "validationWarnings", null);
		//
		// Load our messages, but first we need to add a closing tag.
		//
		Document oXMLMsgDoc = new AppModel(null).getDocument();
		ByteArrayInputStream oFixMemStream;
        try {
            oFixMemStream = new ByteArrayInputStream(sLogMessages.getBytes(Document.Encoding));
    		oXMLMsgDoc.load(oFixMemStream, null, false);
        } catch (UnsupportedEncodingException e) {
        }
		//
		// Get the message nodes and append them to the <messages> node
		// It it is a validation error/message, also append it to
		// to the <validationError> node
		//
		Node oRoot = oXMLMsgDoc.getDocumentElement();
		Node oChild = null;
		if (oRoot != null)
			oChild = oRoot.getFirstXFAChild();
		while (oChild != null) {
			if (oChild instanceof Element) {
				Element oMsgNode
							= (Element) oDoc.importNode(oChild, true);
				String sAttValue = "";
				int i = oMsgNode.findAttr("", "sev");
				if (i != 1)	
					sAttValue = oMsgNode.getAttrVal(i);
				if (sAttValue.equals("ve"))
					oValidationErrors.appendChild(oMsgNode, false);
				else if (sAttValue.equals("vw"))
					oValidationWarnings.appendChild(oMsgNode, false);
				else
					oMessages.appendChild(oMsgNode, false);
			}
			oChild = oChild.getNextXFASibling();
		}
		oParent.appendChild(oMessages, false);
		oParent.appendChild(oValidationWarnings, false);
		oParent.appendChild(oValidationErrors, false);
	}

	int identify(InputStream is) {
		//
		// mark current position in case we need to rewind.
		//
		if (is.markSupported())
			is.mark(Integer.MAX_VALUE);
		int eFType = Unknown;
		XMLStorage xml = new XMLStorage();
		AppModel oAppModel = new AppModel(null);
		try {
			//
			// try loadling XDP/XML.
			//
			xml.loadModel(oAppModel, is, "", null);
			//
			// try rewinding the input source.
			//
			if (is.markSupported())
				is.reset();
		} catch (IOException e) {
			throw new ExFull(e);
		} catch (ExFull e) {
			if (e.getResId(0) == ResId.XFAAgentFailure /* EXPAT_ERROR */)
				return eFType;
			//
			// re-throw whatever the exception was
			//
			throw e;
		}
		if (oAppModel.getFirstXFAChild() != null)
			eFType = XDP; 
		else
			eFType = XML;
		return eFType;
	}

	/**
	 * @exclude from published api.
	 */
	public boolean loadXDP(InputStream is,
						String sPackets,
							boolean bReload /* =false */) {
		return loadXFAModel(is, sPackets, bReload);
	}

	boolean loadXDC(InputStream is) {
		if (is != null)
			throw new ExFull(ResId.UNSUPPORTED_OPERATION, "FormProcessor#loadXDC");
		return false;
	}

	String getTemplatePackageList() {
		StringBuilder sPackets = new StringBuilder();
		sPackets.append(XFA.TEMPLATE);
		sPackets.append(' ');
		sPackets.append(XFA.SOURCESET);
		sPackets.append(' ');
		sPackets.append(XFA.CONNECTIONSET);
		sPackets.append(' ');
		sPackets.append(XFA.XSL);
		sPackets.append(' ');
		sPackets.append(XFA.LOCALESET);
		sPackets.append(' ');
		sPackets.append(STRS.DATASETS);
		return sPackets.toString();
	}

	void saveXDP(OutputStream oStream, String sPacketList) {
		assert (oStream != null);
		NodeList oNodesToSave = new ArrayNodeList();
		boolean bIncludeData = true;
		AppModel oAppModel = getAppModel();
		//	
		// get the models
		//
		ConfigurationModel oConfigModel
					= ConfigurationModel.getConfigurationModel(oAppModel, false);
		TemplateModel oTemplateModel
					= TemplateModel.getTemplateModel(oAppModel, false);	
	// JavaPort: n/a
	//	SourceSetModel oSSModel = SourceSetModel.getSourceSetModel(oAppModel);	
		ConnectionSetModel oCSModel 
				= ConnectionSetModel.getConnectionSetModel(oAppModel, false);	
		//
		// create acrobat section
		//
		//Element oAcrobat =
					oConfigModel.createElement(XFA.ACROBAT);
		//
		// create common tag
		//
		Element oCommon =
					oConfigModel.createElement(XFA.COMMON);
		//
		// set the data options
		//
		Node oDataOptions
			= getContextNode().resolveNode("common.data");
		if (oDataOptions != null) {
			//
			// Must be done before we modify config so we can set the xsl
			// uri for all agents. If there is an input XSL file, embed it
			// as well
			//
			Node oConfigValue
					= oDataOptions.resolveNode("xsl.uri");
			if (oConfigValue instanceof  ConfigurationValue) {
				String sXSLFile = ((ConfigurationValue) oConfigValue).getValue().toString();
				File oXSLFile = new File(sXSLFile);
				if (oXSLFile.exists()) {
				// JavaPort: no xlst processing.
				//	try {
				//		FileInputStream oXSLStream
				//							= new FileInputStream(oXSLFile); 
				//	} catch (IOException e) {
				//		throw new ExFull(e);
				//	}
				//	Document oDoc = pAppModel->domDocument();
				//	Element oRoot (oDoc.loadIntoDocument(oXSLStream));
				//	Node oChild = oRoot.getFirstChild());
				//	while (oChild != null) {
				//		if (oChild instanceof Element) {	
				//			// add xsl packet to the xfa
				//			Packet oDataInputXSL
				//							= new Packet(oAppModel, oChild);
				//			//
				//			// set the id of the xsl
				//			//
				//			oDataInputXSL.setAttribute("XFADataInputXSL", "id");
				//			oNodesToSave.append(pDataInputXSL);
				//			break;
				//		}
				//		oChild = oChild.getNextSibling();
				//	}
					throw new ExFull(ResId.UNSUPPORTED_OPERATION, "FormProcessor#saveXDP - data input XSL");
				}
			}
			//
			// save the output xsl stream
			//
			oConfigValue = oDataOptions.resolveNode("outputXSL.uri");
			if (oConfigValue instanceof  ConfigurationValue) {
				String sXSLFile = ((ConfigurationValue) oConfigValue).getValue().toString();
				File oXSLFile = new File(sXSLFile);
				if (oXSLFile.exists()) {
				// JavaPort: no XSLT processing.
				//	try {
				//		FileInputStream oXSLStream
				//							= new FileInputStream(oXSLFile);
				//	} catch (IOException e) {
				//		throw new ExFull(e);
				//	}
				//	Document oDoc = pAppModel->domDocument();
				//	Element oRoot (oDoc.loadIntoDocument(oXSLStream));
				//	Node oChild = oRoot.getFirstChild();
				//	while (oChild != null) {
				//		if (oChild instanceof Element) {	
				//			// add xsl packet to the xfa
				//			Packet oDataOutputXSL
				//							= new Packet(pAppModel, oChild);
				//			//
				//			// set the id of the xsl
				//			//
				//			oDataOutputXSL.setAttribute("XFADataOutputXSL",
				//														"id");
				//			oNodesToSave.append(pDataOutputXSL);
				//			break;
				//		}
				//		oChild = oChild.getNextSibling();
				//	}
					throw new ExFull(ResId.UNSUPPORTED_OPERATION, "FormProcessor#saveXDP - data input XSL");
				}
			}
			//
			// clone common section and append it to the acrobat section.
			//
			Node oClone = oDataOptions.clone(null);
			oCommon.getNodes().append(oClone);
			oNodesToSave.append(oConfigModel);
		}
		//
		// set the template options
		//
		Node oTemplateOptions = getContextNode().resolveNode("common.template");
		if (oTemplateOptions instanceof ConfigurationKey) {
			//
			// clone common section and append it to the acrobat section.
			//
			Node oClone = oTemplateOptions.clone(null);
			oCommon.getNodes().append(oClone);
		}
		//
		// embed the template
		//
		oNodesToSave.append(oTemplateModel);
		//
		// Add the sourceSet model (if there is one) to the PDF.
		//
		// JavaPort: no support for SourceSetModel
//		if (oSSModel != null)
//			oNodesToSave.append(oSSModel);
		//
		// Add the connection Set model (if there is one) to the PDF.
		//
		if (oCSModel != null)
			oNodesToSave.append(oCSModel);
		if (bIncludeData) {
			DataModel oDataModel
							= DataModel.getDataModel(oAppModel, false, false);
			if (oDataModel != null) {
				//
				// Add the data file (if there was one) to the PDF.
				//
				oNodesToSave.append(oDataModel);
				//
				// Just to delineate this workaround,
				// which should be resolved more elegantly after alpha...
				// In XDP format, we need the entire data structure
				// (datasets/data/root) to be present and connected
				// to the same XML DOM document.  If we add
				// a peer to the root element, this structure will be
				// created automatically // for us by xfadata.  If we
				// then remove the node this work is not undone.
				//
			// JavaPort: ugh?
			//	oDataModel.getAliasNode().getImpl().connectPeerToParent();
			//	// force recalc of record depth
				oDataModel.getDataWindow().resetRecordDepth();
			}
		}
		//
		// Mark the encryption passwords as transient - they are never
		// to be saved out in the xfa stream!
		// Javaport: passords are marked as hidden.
		//
		Element oUser = null;
		Element oMaster = null;
		Node oPDFEncryption
			= oConfigModel.resolveNode("$config.present.pdf.encryption");
		if (oPDFEncryption instanceof Element) {
			if (((Element) oPDFEncryption).isPropertySpecified(XFA.USERPASSWORDTAG, true, 0)) {
				oUser = ((Element) oPDFEncryption).getElement(XFA.USERPASSWORDTAG, false, 0, false, false);
				oUser.isHidden(true);
			}
			if (((Element) oPDFEncryption).isPropertySpecified(XFA.MASTERPASSWORDTAG, true, 0)) {
				oMaster = ((Element) oPDFEncryption).getElement(XFA.MASTERPASSWORDTAG, false, 0, false, false);
				oMaster.isHidden(true);
			}
		}		
		//
		// Allow fine control over which packets are saved.  Note that
		// we're resetting oNodesToSave if xdp.packets is specified, which
		// makes some of the above code a waste.  However it keeps this
		// feature much cleaner if we keep it here, and the waste is minimal.
		//
		String sPackets = sPacketList;
		if (sPackets.length() == 0) {
			Node oPackets = getContextNode().resolveNode("xdp.packets");
			if (oPackets != null) {
				ConfigurationValue oPacketsValue
											= (ConfigurationValue) oPackets;
				sPackets = oPacketsValue.getValue().toString();
			}
		}
		if (sPackets.length() != 0) {
			boolean bAllPackets = (sPackets.equals("*"));
			oNodesToSave = new ArrayNodeList();	// reset list
			for (Node oChild = oAppModel.getFirstXFAChild(); oChild != null; oChild = oChild.getNextXFASibling()) {
				//
				// See if the packet's name is in the list of packets to save.
				// Models are considered to be packets here, for consistency
				// with XDPTK.
				//
				if (bAllPackets && ! (oChild instanceof Chars)) {
					oNodesToSave.append(oChild);
					continue;
				}
				StringTokenizer oToker = new StringTokenizer(sPackets);
				while (oToker.hasMoreTokens()) {
					String sToken = oToker.nextToken();
					if (sToken.equals(oChild.getName())) {
						oNodesToSave.append(oChild);
						break;
					}
				}
			}
		}
		//
		// Save the nodes
		//
		XMLStorage oStorage = new XMLStorage();
		oStorage.saveAggregate("", oStream, oNodesToSave, "");
		//			
		// Re-enable the passwords (?)
		//			
		if (oUser != null)
			oUser.isHidden(false);
		if (oMaster != null)
			oMaster.isHidden(false);
	}
}
