/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */
package com.adobe.xfa.data;

import java.util.ArrayList;
import java.util.BitSet;
import java.util.List;
import java.util.SortedMap;
import java.util.StringTokenizer;
import java.util.TreeMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.adobe.xfa.Chars;
import com.adobe.xfa.Element;
import com.adobe.xfa.EnumAttr;
import com.adobe.xfa.EnumType;
import com.adobe.xfa.Node;
import com.adobe.xfa.XFA;
import com.adobe.xfa.ut.ExFull;
import com.adobe.xfa.ut.ResId;
import com.adobe.xfa.ut.StringUtils;

/**
 * DataTransformation contains and implements the config <transform> commands.
 *
 * @exclude from published api -- Mike Tardif, May 2006.
 */

public final class DataTransformations {
	
	// represents a single transformation

	private static class Transformation {
		
		private static final Pattern pattern = Pattern.compile("\\s*([^\\s]+)\\s*=\\s*(['\"])(.*?)\\2\\s*");
		private static final Pattern badName = Pattern.compile("\\s*([^\\s]+)\\s*");

		// These maps use case-insensitive lookup for compatibility with some test inputs in transform.in.
		// It doesn't seem to be specified anywhere if case sensitivity is required, so be tolerant.
		private final static SortedMap<String, Integer> operationTypes = new TreeMap<String, Integer>(String.CASE_INSENSITIVE_ORDER);
		private final static SortedMap<String, Integer> operationValues = new TreeMap<String, Integer>(String.CASE_INSENSITIVE_ORDER);

		static {
			initOperationTypes();
			initOperationValues();		
		}

		private final String	msNodeNameList;
		
		private final int		mePresence;
		private final int		meWhiteSpace;
		private final int		meIfEmpty;
		private final String	msMapName;
		private final String	msGroupParent;
		private final String	msNameAttr;
		private final boolean	mbHasMapName;
		private final boolean	mbHasGroupParent;
		private final boolean	mbHasNameAttr;
		private final String	msPictureFormat;
		
		private int				mnListIndex;
		
	
		public Transformation(String sTransformation) {
			
			int		ePresence = EnumAttr.UNDEFINED;
			int		eWhiteSpace = EnumAttr.UNDEFINED;
			int		eIfEmpty = EnumAttr.UNDEFINED;
			String	sMapName = "";
			String	sGroupParent = "";
			String	sNameAttr = "";
			boolean	bHasMapName = false;
			boolean	bHasGroupParent = false;
			boolean	bHasNameAttr = false;
			String	sPictureFormat = null;
			
			sTransformation = sTransformation.trim();
			Matcher matcher = pattern.matcher(sTransformation);
			
			boolean refMatched = false;
			
			String sNodeNameList = "";
			
			while (matcher.lookingAt()) {
				
				String sName = matcher.group(1);
				String sValue = matcher.group(3);
				
				if (!refMatched) {
					assert sName.equals(XFA.REF);
					sNodeNameList = sValue;
					refMatched = true;
				}
				else {
					Integer slot = operationTypes.get(sName);
					if (slot != null) {
						int oOperationType = slot.intValue();
						slot = operationValues.get(sName + "_" + sValue);
						
						if (slot != null) {
							int eOperationValue = slot.intValue();
							switch (oOperationType)
							{
								case EnumType.PRESENCE	: ePresence	= eOperationValue; break;
								case EnumType.WHITESPACE: eWhiteSpace	= eOperationValue; break;
								case EnumType.IFEMPTY	: eIfEmpty		= eOperationValue; break;
							}
						}
					}
					else if (sName.equals("rename")) {
						
						if ( !StringUtils.isEmpty(sValue) ) {
							sMapName = sValue;
							bHasMapName = true;
						}
						// else this should invalidate this transformation
					}
					else if (sName.equals("groupParent")) {
						
						if ( !StringUtils.isEmpty(sValue) ) {
							sGroupParent = sValue;
							bHasGroupParent = true;
						}
						// else this should invalidate this transformation
					}
					else if (sName.equals(XFA.NAMEATTR)) {
						
						if ( !StringUtils.isEmpty(sValue) ) {
							sNameAttr = sValue;
							bHasNameAttr = true;
						}
						// else this should invalidate this transformation
					}
					else if (sName.equals("picture")) {
						
						if (!StringUtils.isEmpty(sValue)) {
							assert(StringUtils.isEmpty(sPictureFormat));
							sPictureFormat = sValue;
						}
						// else this should invalidate this transformation
					}	
				}
				
				matcher.region(matcher.end(), matcher.regionEnd());
			}			
			
			if (!refMatched || !matcher.hitEnd()) {
				matcher.usePattern(badName);
				matcher.lookingAt();
				String sName = matcher.group(1);
				
				throw new ExFull(ResId.MalformedOptionException, sName);
			}
			
			msNodeNameList = sNodeNameList;
			
			mePresence = ePresence;
			meWhiteSpace = eWhiteSpace;
			meIfEmpty = eIfEmpty;
			msMapName = sMapName;
			msGroupParent = sGroupParent;
			msNameAttr = sNameAttr;
			mbHasMapName = bHasMapName;
			mbHasGroupParent = bHasGroupParent;
			mbHasNameAttr = bHasNameAttr;
			msPictureFormat = sPictureFormat;
		}
		
		public String getNodeNameList() {
			return msNodeNameList;
		}
		
		public int getOperation(int eOpType) {

			switch (eOpType) {

			case EnumType.PRESENCE:		return mePresence; 
			case EnumType.WHITESPACE:	return meWhiteSpace;
			case EnumType.IFEMPTY:		return meIfEmpty;
			}

			return EnumAttr.UNDEFINED;
		}
		
		public int getIndex() {
			return mnListIndex;
		}
		
		public void	setIndex(int nIndex) {
			mnListIndex = nIndex;
		}

		public String getMapName() {
			return msMapName;
		}
		
		public boolean	hasMapName() {
			return mbHasMapName;
		}

		private boolean matchName(String sNodeNameList, String sNodeName) {
			StringTokenizer tokenizer = new StringTokenizer(sNodeNameList, " ");
			while ( tokenizer.hasMoreTokens() ) {
				if (tokenizer.nextToken().equals(sNodeName))
					return true;
			}
			
			return false;
		}
		
		public String getGroupName(List<Transformation> transformations, int nCurrent) {
			String sParentList = msGroupParent;

			if (!StringUtils.isEmpty(msGroupParent)) {
				for (int i = nCurrent + 1; i < transformations.size(); i++) {
					
					Transformation transformation = transformations.get(i);
					
					if (transformation.hasGroupParent()) {
						String sNodeNameList = transformation.getNodeNameList();
						if (!StringUtils.isEmpty(sNodeNameList)) {
							
							if (matchName(sNodeNameList, msGroupParent)) {
								
								sParentList = transformation.getGroupName(transformations, i) + "." + msGroupParent;
								break;
							}
						}
					}
				}
			}
			
			return sParentList;
		}
		
		public boolean hasGroupParent() {
			return mbHasGroupParent;
		}

		public String getNameAttr() {
			return msNameAttr;
		}
		
		public boolean hasNameAttr() {
			return mbHasNameAttr;
		}

		public String getPictureFormat() {
			return msPictureFormat;
		}
		
		public boolean hasPictureFormat() {
			return !StringUtils.isEmpty(msPictureFormat);
		}
	
		private static void initOperationTypes() {
			operationTypes.put("presence",		EnumType.PRESENCE	);
			operationTypes.put("whitespace",	EnumType.WHITESPACE	);
			operationTypes.put("ifEmpty",		EnumType.IFEMPTY	);
		}
		
		private static void initOperationValues() {
			operationValues.put("presence_preserve", 			EnumAttr.NODEPRESENCE_PRESERVE			);
			operationValues.put("presence_ignore", 				EnumAttr.NODEPRESENCE_IGNORE			);
			operationValues.put("presence_dissolve", 			EnumAttr.NODEPRESENCE_DISSOLVE			);
			operationValues.put("presence_dissolveStructure", 	EnumAttr.NODEPRESENCE_DISSOLVESTRUCTURE	);
			
			operationValues.put("whitespace_preserve", 			EnumAttr.WHITESPACE_PRESERVE			);
			operationValues.put("whitespace_rtrim", 			EnumAttr.WHITESPACE_RTRIM				);
			operationValues.put("whitespace_ltrim",				EnumAttr.WHITESPACE_LTRIM				);
			operationValues.put("whitespace_trim",				EnumAttr.WHITESPACE_TRIM				);
			operationValues.put("whitespace_normalize",			EnumAttr.WHITESPACE_NORMALIZE			);

			operationValues.put("ifEmpty_dataGroup",			EnumAttr.IFEMPTY_DATAGROUP				);
			operationValues.put("ifEmpty_dataValue", 			EnumAttr.IFEMPTY_DATAVALUE				);		
			operationValues.put("ifEmpty_ignore", 				EnumAttr.IFEMPTY_IGNORE					);
			operationValues.put("ifEmpty_remove", 				EnumAttr.IFEMPTY_REMOVE					);
		}
	}
	
	
	private final static Pattern whitespaceDelimitedToken = Pattern.compile("\\s*([^\\s]+)\\s*");
	
	// list of transformations
	private final List<Transformation> 	mTransformations = new ArrayList<Transformation>();

	// list of transformations for a specific node (for performance)
	private final List<Transformation>	mNodeTransformations = new ArrayList<Transformation>();
	// the node name we are currently transforming (for performance)
	private String						msCurNodeName;

	// the group node we are currently transforming to
	private String		 msCurGroupName = "";
	private final BitSet msCurGroupAvailable = new BitSet();
	private Element 	 mCurGroupNode;
	private DataNode	 mLastParent;
	
	
	/**
	 * Default constructor. Creates an empty wrapper with no implementation.
	 */
	DataTransformations() {
	}

	/**
	 * Adds this transformation to the list.
	 */
	void add(String sTransformations) {
		
		// JavaPort: The C++ implementation has some obscure code that seems to handle
		// the case where sTransformations starts with "ref,", but that never seems to
		// happen and the code doesn't look correct, so it is omitted.
		
		mTransformations.add(new Transformation(sTransformations));
	}
	
	private List<DataNode> getDataValues(DataNode dataValue) {
		List<DataNode> dataValues = new ArrayList<DataNode>();
		return getDataValues(dataValue, dataValues);
	}

	private List<DataNode> getDataValues(DataNode dataValue, List<DataNode> dataValues) {
		if (dataValue.getFirstXFAChild() == null) {
			dataValues.add(dataValue);
		}
		else {
			
			for (Node child = dataValue.getFirstXFAChild(); child != null; child = child.getNextXFASibling()) {
				getDataValues((DataNode)child, dataValues);
			}
		}
		
		return dataValues;
	}

	/**
	 * Returns the operation value based on the type of operation performed or 0
	 * if the operation is not defined.
	 * 
	 * When two transformations exist (e.g for a specific node and the global
	 * node: name=""), the operation representing the combined operations is
	 * returned.
	 * 
	 * For example if the presence operations are DISSOLVE and IGNORE, then
	 * DISSOLVE is returned. If the match operation is MANY and NONE, then NONE
	 * is returned since this is the last match operation.
	 */
	int getOperation(String aNodeName, int eOpType) {
		
		int ePrevOperation;
		int eOperation = EnumAttr.UNDEFINED;

		getTransformations(aNodeName);
		for (int i = 0; i < mNodeTransformations.size(); i++) {
			
			ePrevOperation = eOperation;
			eOperation = mNodeTransformations.get(i).getOperation(eOpType);

			switch (eOpType) {
			
			case EnumType.PRESENCE:
				eOperation = getOverridingOperation(ePrevOperation, eOperation);
				break;

			case EnumType.WHITESPACE:
				if ((ePrevOperation == EnumAttr.WHITESPACE_LTRIM && eOperation == EnumAttr.WHITESPACE_RTRIM) ||
					(ePrevOperation == EnumAttr.WHITESPACE_RTRIM && eOperation == EnumAttr.WHITESPACE_LTRIM)) {
					
					eOperation = EnumAttr.WHITESPACE_TRIM;
				}
				else {
					
					eOperation = getOverridingOperation(ePrevOperation, eOperation);
				}
				break;

			case EnumType.IFEMPTY:
				{
					int eOverridingOperation = getOverridingOperation(ePrevOperation, eOperation);
					if (eOverridingOperation == EnumAttr.IFEMPTY_IGNORE ||
						eOverridingOperation == EnumAttr.IFEMPTY_REMOVE) {
						
						eOperation = eOverridingOperation;
					}
				}
				break;
			}
		}
		
		return eOperation;
	}

	String getMapName(String aNodeName) {
		
		getTransformations(aNodeName);		
		int nTrans = mNodeTransformations.size();
		String sNewName = "";
		boolean bFound = false;

		if (nTrans > 0) {
			
			// if we have a global transformation and a node specific
			// transformation, the node specific transformation should take
			// precedence. Caller should check for empty map name, meaning
			// no mapping was found.
			String sGlobalName = "";
			String sNodeNameList;

			for (int i = 0; i < nTrans; i++) {
				
				Transformation transformation = mNodeTransformations.get(i);

				if (transformation.hasMapName()) {
					
					// get the ref of this transform
					sNodeNameList = transformation.getNodeNameList();

					// see if the ref name matches this node's name
					if (matchName(sNodeNameList, aNodeName)) {
						sNewName = transformation.getMapName();
						bFound = true;
						break;			// first one wins
					}
					else if (StringUtils.isEmpty(sNodeNameList)) {
						// empty name denotes global transform
						// last one wins
						sGlobalName = transformation.getMapName();
					}
				}
			}

			if (!bFound) {
				// if we didn't find a direct mapping, try the
				// global mapping.
				sNewName = sGlobalName;
			}
		}
		
		return sNewName.intern();	// create atom if necessary
	}
	
	private int getOperationPrecedence(int eOperation) {
		
		switch (eOperation) {
		
		case EnumAttr.NODEPRESENCE_PRESERVE:			return 0;
		case EnumAttr.NODEPRESENCE_IGNORE:				return 1;
		case EnumAttr.NODEPRESENCE_DISSOLVE:			return 2;
		case EnumAttr.NODEPRESENCE_DISSOLVESTRUCTURE:	return 3;

		case EnumAttr.WHITESPACE_PRESERVE:				return 0;
		case EnumAttr.WHITESPACE_RTRIM:					return 1;	
		case EnumAttr.WHITESPACE_LTRIM:					return 1;	
		case EnumAttr.WHITESPACE_TRIM:					return 2;
		case EnumAttr.WHITESPACE_NORMALIZE:				return 3;

		case EnumAttr.IFEMPTY_DATAGROUP:				return 0;
		case EnumAttr.IFEMPTY_DATAVALUE:				return 0;
		case EnumAttr.IFEMPTY_IGNORE:					return 1;
		case EnumAttr.IFEMPTY_REMOVE:					return 2;
		}
		
		return -1;
	}
	
	private int getOverridingOperation(int ePrevOperation, int eOperation) {
		int nPrevPrecedence = getOperationPrecedence(ePrevOperation);
		int nPrecedence     = getOperationPrecedence(eOperation);
		return nPrevPrecedence > nPrecedence ? ePrevOperation : eOperation;
	}
	
	private void getTransformations(String sNodeName) {
		
		if (!sNodeName.equals(msCurNodeName)) {
			
			msCurNodeName = null;
			mNodeTransformations.clear();
			
			for (int i = 0; i < mTransformations.size(); i++) {
				Transformation transformation = mTransformations.get(i);
				
				String sNodeNameList = transformation.getNodeNameList();
				if (StringUtils.isEmpty(sNodeNameList)) {
					mNodeTransformations.add(transformation);
				} 
				else {

					if (matchName(sNodeNameList, sNodeName)) {
						// set the index from the main list so we know where to 
						// start searching from when building Parent Lists.
						transformation.setIndex(i);
						mNodeTransformations.add(transformation);
					}
				}
			}

			if (mNodeTransformations.size() > 0) {
				msCurNodeName = sNodeName;
			}
		}
	}

	private DataNode insertGroupParent(
			DataNode	node, 
			DataNode	groupNode, 
			String 		sCurList,
			String 		sParentList, 
			int 		nLevel) {
		
		int nFoundAt = sParentList.indexOf('.');
		if (nFoundAt >= 0) {
			// we are processing an ancestor level
			String sCurRest = "";
			String sCurRoot = "";
			String sTheRest = sParentList.substring(nFoundAt + 1);
			String sTheRoot = sParentList.substring(0, nFoundAt);

			nFoundAt = sCurList.indexOf('.');
			if (nFoundAt >= 0) {
				sCurRest = sCurList.substring(nFoundAt + 1);
				sCurRoot = sCurList.substring(0, nFoundAt);
			}

			if (sTheRoot.equals(sCurRoot)) {
				// Just use the existing parent branch that was added before.
				
				Node lastXFAChild = null;
				for (Node child = groupNode.getFirstXFAChild(); child != null; child = child.getNextXFASibling())
					lastXFAChild = child;
				
				groupNode.appendChild(node);
				groupNode = (DataNode)lastXFAChild;
				groupNode = insertGroupParent(node, groupNode, sCurRest, sTheRest, nLevel + 1);
			} 
			else {
				// OK so now we will build up the new parent branch.
				groupNode = insertGroupParent(node, groupNode, "", sTheRest, nLevel + 1);
				// Now create the group above the current groupParents.
				DataNode parentNode = (DataNode)node.getModel().createNode(XFA.DATAGROUPTAG, null, sTheRoot.intern(), null, true);
				groupNode.getXFAParent().insertChild(parentNode, groupNode, true);
				parentNode.appendChild(groupNode, true);
				groupNode = parentNode;
				// if processing at the root then reset it.
				if ( nLevel == 0 ) {
					mLastParent = groupNode;
				}
			}
		} 
		else {
			// we are at the immediate parent node so just add it in.
			groupNode = (DataNode)node.getModel().createNode(XFA.DATAGROUPTAG, null, sParentList.intern(), null, true);
			node.getXFAParent().insertChild(groupNode, node, true);
			groupNode.appendChild(node, true);
			mCurGroupNode = groupNode;
			// if processing at the root then reset it.
			if ( nLevel == 0 ) {
				mLastParent = groupNode;
			}
		}

		return groupNode;
	}

	private DataNode applyGroupParent(DataNode node, String sNodeName) {
		getTransformations(sNodeName);
		int nTrans = mNodeTransformations.size();
		boolean bFound = false;
		boolean bAvailable = false;
		int nCurrent = 0;

		if (nTrans > 0) {
			// if we have a global transformation and a node specific
			// transformation, the node specific transformation should take
			// precedence. Caller should check for empty map name, meaning
			// no mapping was found.
			String sNodeNameList;

			// This Field name appears in more than one subform, 
			// so we must check to see if the current group is one of them.
			if (nTrans > 1) {
				for (int i = 0; i < nTrans; i++) {
					
					Transformation transformation = mNodeTransformations.get(i);

					if (transformation.hasGroupParent()) {
						
						String sGroupName = transformation.getGroupName(mTransformations, transformation.getIndex());
						if (sGroupName.equals(msCurGroupName)) {
							nCurrent = i;
						}
					}
				}
			}

			int i = nCurrent;
			while (i < nTrans) {
				
				Transformation transformation = mNodeTransformations.get(i);

				if (transformation.hasGroupParent()) {
					
					// get the ref of this transform
					sNodeNameList = transformation.getNodeNameList();
					String sGroupName = transformation.getGroupName(mTransformations, transformation.getIndex());

					// see if the ref name matches this node's name
					int index = 0;
					int count = 0;
					StringTokenizer tokenizer = new StringTokenizer(sNodeNameList, " ");
					while (tokenizer.hasMoreTokens()) {
						String sOneName = tokenizer.nextToken();
						if (sOneName.equals(sNodeName)) {
							bFound = true;
							index = count;
							if ( !(sGroupName.equals(msCurGroupName)) || msCurGroupAvailable.get(index) ) {
								bAvailable = true;
								break;
							}
						}
						count++;
					}
					
					// We found a subform that contains this field.
					if (bFound) {
						// If available in the current subform then use it, otherwise reset and start a new group.
						if ( bAvailable && sGroupName.equals(msCurGroupName) ) {
							mCurGroupNode.appendChild(node);
						} 
						else {

							insertGroupParent(node, mLastParent, msCurGroupName, sGroupName, 0);

							// Now create and set the "Used" list to check against.
							int j = 0;
							StringTokenizer tokenizer2 = new StringTokenizer(sNodeNameList, " ");
							while (tokenizer2.hasMoreTokens()) {
								tokenizer2.nextToken();
								j++;
							}
							msCurGroupAvailable.set(0, j);
							msCurGroupName = sGroupName;
						}
						msCurGroupAvailable.clear(index);
						break;			// first one wins
					}
				}
				
				// Field was not in current group so start looking through ALL transforms.
				// THW Future Option - This check could be set to continue from the current subform and then loop back to the start.
				if ( nCurrent != 0 ) {
					i = 0;
					nCurrent = 0;
				}
				else
					i++;
			}

			/* THW - Stub for now.  We may want to make this some sort of option.  
			*   For now we want to just ignore completely unknown fields in the data.
			*		if (!bFound) {
			*			msCurGroupName = "";
			*			msCurGroupAvailable.clear();
			*		}
			*/

		}
		return node;
	}

	String getNameAttr(String aNodeName) {
		String sNameAttr = "";
		boolean bFound = false;

		getTransformations(aNodeName);
		int nTrans = mNodeTransformations.size();

		if (nTrans > 0) {
			
			// if we have a global transformation and a node specific
			// transformation, the node specific transformation should take
			// precedence. Caller should check for empty map name, meaning
			// no mapping was found.
			String sGlobalName = "";
			String sNodeNameList;

			for (int i = 0; i < nTrans; i++) {
				
				Transformation transformation = mNodeTransformations.get(i);

				if (transformation.hasNameAttr()) {
					
					// get the ref of this transform
					sNodeNameList = transformation.getNodeNameList();

					// see if the ref name matches this node's name
					if (matchName(sNodeNameList, aNodeName)) {
						
						bFound = true;
						sNameAttr = transformation.getNameAttr();
						break;			// first one wins
					}
					else if (StringUtils.isEmpty(sNodeNameList))
					{
						// empty name denotes gloabal transform
						// last one wins
						sGlobalName = transformation.getNameAttr();
					}
				}
			}

			if (!bFound)
			{
				// if we didn't find a direct mapping, try the
				// global mapping.
				sNameAttr = sGlobalName;
			}
		}
		
		return sNameAttr.intern();	// create atom if necessary
	}

	String getPictureFormat(String aNodeName) {
		String sPictureFormat = "";
		boolean bFound = false;

		getTransformations(aNodeName);
		int nTrans = mNodeTransformations.size();
		if (nTrans > 0) {
			
			// if we have a global transformation and a node specific
			// transformation, the node specific transformation should take
			// precedence. Caller should check for empty name, meaning
			// no mapping was found.
			String sGlobalPictureFormat = "";
			String sNodeNameList;

			for (int i = 0; i < nTrans; i++) {
				
				Transformation transformation = mNodeTransformations.get(i);

				if (transformation.hasPictureFormat()) {
					
					// get the ref of this transform
					sNodeNameList = transformation.getNodeNameList();

					/* see if the ref name matches this node's name */
					if (matchName(sNodeNameList, aNodeName)) {
						
						bFound = true;
						sPictureFormat = transformation.getPictureFormat();
						break;			// first one wins
					}
					else if (StringUtils.isEmpty(sNodeNameList)) {
						
						// empty name denotes gloabal transform
						// last one wins
						sGlobalPictureFormat = transformation.getPictureFormat();
					}
				}
			}

			if (!bFound) {
				// if we didn't find a direct picture format, try the
				// global picture format.
				sPictureFormat = sGlobalPictureFormat;
			}
		}
		
		return sPictureFormat;
	}
	
	private void ignore(DataNode node) { // ifempty=ignore
	
		// remove any records from the data window
		if (node.getClassTag() == XFA.DATAGROUPTAG) {
			
			DataModel dataModel = (DataModel)node.getModel();
			DataWindow dataWindow = dataModel.getDataWindow();
			if (dataWindow != null)
				dataWindow.removeRecordGroup(node);
		}

		for (Node child = node.getFirstXFAChild(); child != null; child = child.getNextXFASibling())
			ignore((DataNode)child);

		// set the dom peer to null so that when we call remove the dom node isn't removed
		node.getXmlPeer().setXfaPeer(null);
		node.setXmlPeer(null);
		node.remove();
	}


	private void remove(DataNode node) {
		
		// remove any records from the data window
		if (node.getClassTag() == XFA.DATAGROUPTAG) {
			DataModel dataModel = (DataModel)node.getModel();
			DataWindow dataWindow = dataModel.getDataWindow();
			if (dataWindow != null)
				dataWindow.removeRecordGroup(node);
		}

		for (Node child = node.getFirstXFAChild(); child != null; child = child.getNextXFASibling())
			remove((DataNode)child);
		
		node.remove();
	}

	private void ltrim(List<DataNode> dataValues) {
		
		for (int i = 0; i < dataValues.size(); i++) {
			DataNode dataValue = dataValues.get(i);
			Node peer = dataValue.getXmlPeer();
			
			if (peer instanceof Chars && ((Chars)peer).isXMLSpace()) {
				dataValue.setValue("", true);
			}
			else {
				String sValue = dataValue.getValue();
				sValue = sValue == null ? "" : StringUtils.trimStart(sValue);
				dataValue.setValue(sValue, true);
				return;
			}
		}
	}

	private void rtrim(List<DataNode> dataValues) {
		
		for (int i = dataValues.size(); i > 0; i--) {
			DataNode dataValue = dataValues.get(i - 1);			
			Node peer = dataValue.getXmlPeer();
			
			if (peer instanceof Chars && ((Chars)peer).isXMLSpace()) {
				dataValue.setValue("", true);
			}
			else {
				String sValue = dataValue.getValue();
				sValue = sValue == null ? "" : StringUtils.trim(sValue);
				dataValue.setValue(sValue, true);
				return;
			}
		}
	}

	private void normalize(List<DataNode> dataValues) {
		boolean bPrevNodeEndsWithWS = true;
		
		for (int i = 0; i < dataValues.size(); i++) {
			
			DataNode dataValue = dataValues.get(i);
			String sNodeValue = dataValue.getValue();
			
			if (StringUtils.isEmpty(sNodeValue))
				continue;
			
			boolean bNodeStartsWithWS = Character.isWhitespace(sNodeValue.charAt(0));
			boolean bNodeEndsWithWS   = Character.isWhitespace(sNodeValue.charAt(sNodeValue.length() - 1));

			Matcher matcher = whitespaceDelimitedToken.matcher(sNodeValue);
			
			if (matcher.lookingAt()) {
				StringBuilder sResult = new StringBuilder();
				
				if (bNodeStartsWithWS && !bPrevNodeEndsWithWS)
					sResult.append(' ');
				
				sResult.append(matcher.group(1));
				matcher.region(matcher.end(), matcher.regionEnd());
				
				while (matcher.lookingAt()) {
					sResult.append(' ');
					sResult.append(matcher.group(1));
					matcher.region(matcher.end(), matcher.regionEnd());
				}
				
				if (bNodeEndsWithWS) 
					sResult.append(' ');
				
				dataValue.setValue(sResult.toString(), true);
				bPrevNodeEndsWithWS = bNodeEndsWithWS;
			}
			else {
				dataValue.setValue("", true);
			}
		}
	}

	// scans a space delimited list of names for the given one
	private boolean matchName(String sNodeNameList, String sNodeName) {

		StringTokenizer tokenizer = new StringTokenizer(sNodeNameList, " ");
		while (tokenizer.hasMoreTokens()) {
			if (tokenizer.nextToken().equals(sNodeName)) {
				return true;
			}
		}
		
		return false;
	}

	/**
	 * Performs transformations on an XFA DOM node. All operations are performed
	 * except for presence which is ignored. Returns poNode or NULL if it has
	 * been removed.
	 */
	DataNode transform(DataNode node) {
		String aNodeName = node.getName();
		if (aNodeName != "") {
			
			performOperation(node, getOperation(aNodeName, EnumType.WHITESPACE));
			node = performOperation(node, getOperation(aNodeName, EnumType.IFEMPTY));
			//THW TRANSFORM CALL GOES HERE ************************************************************************************
			node = applyGroupParent(node, aNodeName);
		}
		return node;
	}
	
	private DataNode toDataGroup(DataNode node) {
		Element parent	= node.getXFAParent();

		Node xmlNode = node.getXmlPeer();

		// Set the dom peer to null so that when we call remove the dom node isn't removed.
		// Do this before creating the new node as XFANodeImpl will assert that the
		// jfDomNode's user-object is NULL.
		xmlNode.setXfaPeer(null);
		node.setXmlPeer(null);
		
		// Diversion from C++ here: C++ can pass the XML peer to the constructor, but Java
		// doesn't have such a constructor.  However this constructor will create a dataGroup,
		// and will not attempt to create its own XML peer if localName is null.
		DataNode dataGroup = new DataNode(parent, null, null, null, null, null);
		dataGroup.setXmlPeer(xmlNode);
		xmlNode.setXfaPeer(dataGroup);

		node.remove();
		return dataGroup;
	}

	private DataNode toDataValue(DataNode node) {
		Element parent	= node.getXFAParent();
		DataModel dataModel = (DataModel)node.getModel();

		// remove any records from the data window
		if (node.getClassTag() == XFA.DATAGROUPTAG) {
			DataWindow dataWindow = dataModel.getDataWindow();
			if (dataWindow != null)
				dataWindow.removeRecordGroup(node);
		}

		// Set the dom peer to null so that when we call remove the dom node isn't removed.
		// Do this before creating the new node as XFANodeImpl will assert that the
		// jfDomNode's user-object is NULL.
		Node oDomNode = node.getXmlPeer();
		node.setXmlPeer(null);
		oDomNode.setXfaPeer(null);
		
		DataNode dataValue = (DataNode)dataModel.getSchema().getInstance(XFA.DATAVALUETAG, dataModel, parent, null, false); 
		dataValue.setXmlPeer(oDomNode);
		oDomNode.setXfaPeer(dataValue);
		
		node.remove();
		return dataValue;
	}

	/**
	 * Performs transformation operation on an XFA DOM node. All operations are
	 * supported except for presence which is ignored. Returns poNode or NULL if
	 * it has been removed.
	 */
	private DataNode performOperation(DataNode node, int eOperation) {

			if (eOperation == EnumAttr.UNDEFINED) {
				return node;
			}

			//Ensure we don't convert root xfadatagroup (i.e. 'data') 
			//into a datavalue or ignore it.
			boolean bIsRoot 	 = node.getClassTag() == XFA.DATAGROUPTAG && node.getXFAParent() instanceof DataModel;
			boolean bIsDataValue = node.getClassTag() == XFA.DATAVALUETAG;
			boolean bIsEmpty	 = isEmpty(node);

			switch (eOperation) {
			
				case EnumAttr.IFEMPTY_DATAGROUP:
					if (!bIsRoot && bIsDataValue && bIsEmpty) {
						return toDataGroup(node);
					}
					break;

				case EnumAttr.IFEMPTY_DATAVALUE:
					if (!bIsRoot && !bIsDataValue && bIsEmpty) {
						return toDataValue(node);
					}
					break;

				case EnumAttr.IFEMPTY_IGNORE:
					if (!bIsRoot && bIsEmpty) {
						ignore(node);
						return node;
					}
					break;

				case EnumAttr.IFEMPTY_REMOVE:
					if (!bIsRoot && bIsEmpty) {
						remove(node);
						return node;
					}
					break;
			}

			if (!bIsDataValue) {
				return node;
			}
			
			switch (eOperation) {

				case EnumAttr.WHITESPACE_TRIM:
					{
						List<DataNode> dataValues = getDataValues(node);
						rtrim(dataValues);
						ltrim(dataValues);
					}
					break;

				case EnumAttr.WHITESPACE_RTRIM:		
					{
						List<DataNode> dataValues = getDataValues(node);
						rtrim(dataValues);
					}
					break;

				case EnumAttr.WHITESPACE_LTRIM:		
					{
						List<DataNode> dataValues = getDataValues(node);
						ltrim(dataValues);
					}
					break;

				case EnumAttr.WHITESPACE_NORMALIZE:
					{
						List<DataNode> dataValues = getDataValues(node);
						rtrim(dataValues);
						ltrim(dataValues);
						normalize(dataValues);
					}
				break;

				case EnumAttr.WHITESPACE_PRESERVE:
					// do nothing
					break;

				default:
					// should not happen
					// We actually do get here in the C++ and Java though!
					break;

			}
			return node;
	}

	/**
	 * Returns true if the node is empty
	 */
	private boolean isEmpty(DataNode node) {
		if (node.getClassTag() == XFA.DATAVALUETAG) {
			 return StringUtils.isEmpty(node.getValue());
		}
		else {
			// check if the datagroup is empty
			return node.getFirstXFAChild() == null;
		}
	}

	/**
	 * Returns the number of transformations.
	 */
	int size() {
		return mTransformations.size();
	}
	
	void
	reset() 
	{
		// Clear out all transformations
		mTransformations.clear();

		// Clear all the context sensitive cached data.
		for (int i=0; i <mNodeTransformations.size(); i++)
			mNodeTransformations.remove(i);

		msCurNodeName="";
		msCurGroupName="";
		msCurGroupAvailable.clear();
		mCurGroupNode=null;
		mLastParent=null;
	}
}