/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2009 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */
package com.adobe.xfa.dom;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.xml.XMLConstants;
import javax.xml.namespace.NamespaceContext;

import org.w3c.dom.Node;

/**
 * Implements the interface javax.xml.namespace.NamespaceContext using the
 * namespace prefix mapping in scope from an existing Node.
 * <p>
 * Normally, the namespace prefixes mappings in an XPath expression are 
 * unrelated to any namespace mappings in the XML document, but this implementation
 * uses the existing in-scope namespace mappings from some node in the document to
 * define a set of mappings.
 * <p>
 * Additional mappings can be added explicitly.
 * @exclude from published api.
 */
public class NamespaceContextImpl implements NamespaceContext {
	
	private final ElementImpl mElementImpl;
	private Map<String, String> mNamespaceMappings;
	
	/**
	 * Creates a new NamespaceContext using mappings from an existing node
	 * in some document.
	 * @param node a node to use the in-scope namespace mappings from;
	 * 		  if <code>null</code> then no namespace mappings are defined.
	 */
	public NamespaceContextImpl(Node node) {
		
		ElementImpl element = null;
		if (node instanceof ElementImpl)
			element = (ElementImpl)node;
		else {
			node = node.getParentNode();
			if (node instanceof ElementImpl)
				element = (ElementImpl)node;
		}
		
		mElementImpl = element;
	}

	public String getNamespaceURI (String prefix) {
		
		if (prefix == null)
			throw new IllegalArgumentException("prefix");
		
		if (prefix.equals(XMLConstants.XML_NS_PREFIX))
			return XMLConstants.XML_NS_URI;
		
		if (prefix.equals(XMLConstants.XMLNS_ATTRIBUTE))
			return XMLConstants.XMLNS_ATTRIBUTE_NS_URI;
		
		if (mElementImpl != null) {
			Iterator<AttrImpl> i = mElementImpl.getNamespacesInScopeIterator();
			while (i.hasNext()) {
				AttrImpl attr = i.next();
				if (prefix.equals(extractPrefix(attr)))
					return attr.getValue();
			}
		}
		
		if (mNamespaceMappings != null) {
			String ns = mNamespaceMappings.get(prefix);
			if (ns != null)
				return ns;
		}
		
		return XMLConstants.NULL_NS_URI;
	}

	public String getPrefix (String namespaceURI) {
		
		if (namespaceURI == null)
			throw new IllegalArgumentException("namespaceURI");
		
		if (namespaceURI.equals(XMLConstants.XML_NS_URI))
			return XMLConstants.XML_NS_PREFIX;
		
		if (namespaceURI.equals(XMLConstants.XMLNS_ATTRIBUTE_NS_URI))
			return XMLConstants.XMLNS_ATTRIBUTE;
		
		if (mElementImpl != null) {
			Iterator<AttrImpl> i = mElementImpl.getNamespacesInScopeIterator();
			while (i.hasNext()) {
				AttrImpl attr = i.next();
				if (attr.getValue().equals(namespaceURI))
					return extractPrefix(attr);
			}
		}
		
		if (mNamespaceMappings != null) {
			for (Map.Entry<String, String> mapping : mNamespaceMappings.entrySet())
				if (namespaceURI.equals(mapping.getValue()))
					return mapping.getKey();
		}
		
		if (namespaceURI.equals(XMLConstants.NULL_NS_URI))
			return XMLConstants.DEFAULT_NS_PREFIX;
		
		return null;
	}

	public Iterator<String> getPrefixes (String namespaceURI) {
		
		if (namespaceURI == null)
			throw new IllegalArgumentException("namespaceURI");
		
		List<String> prefixes = new ArrayList<String>();
		
		if (namespaceURI.equals(XMLConstants.XML_NS_URI)) 
			prefixes.add(XMLConstants.XML_NS_PREFIX);
		else if (namespaceURI.equals(XMLConstants.XMLNS_ATTRIBUTE_NS_URI))
			prefixes.add(XMLConstants.XMLNS_ATTRIBUTE);
		else {
			if (mElementImpl != null) {
				Iterator<AttrImpl> i = mElementImpl.getNamespacesInScopeIterator();
				while (i.hasNext()) {
					AttrImpl attr = i.next();
					if (attr.getValue().equals(namespaceURI)) {
						prefixes.add(extractPrefix(attr));
					}
				}
			}
			
			if (mNamespaceMappings != null) {
				for (Map.Entry<String, String> mapping : mNamespaceMappings.entrySet()) {
					if (namespaceURI.equals(mapping.getValue())) {
						prefixes.add(mapping.getKey());
					}
				}
			}
		}
		
		return prefixes.iterator();
	}

	/**
	 * Add a namespace mapping this namespace context.
	 * <p>
	 * Additional namespace mappings can be added to this NamespaceContext
	 * -- mappings which aren't expressed anywhere in the document's
	 * elements.
	 * </p>
	 * <p>
	 * Mappings provided through this method will be used only as a last
	 * resort.	If an element's mapping redefines either the prefix or
	 * namespace URI given in this call, the element's definition will take
	 * precedence for it and its descendants.
	 * </p>
	 * @param prefix Namespace prefix, empty string for default mapping
	 * namespace.
	 * @param namespaceURI Namespace URI associated with the prefix.
	 */
	public void addNamespaceMapping (String prefix, String namespaceURI) {
		if (mNamespaceMappings == null)
			mNamespaceMappings = new HashMap<String, String>();
		
		mNamespaceMappings.put(prefix, namespaceURI);
	}
	
	private static String extractPrefix(AttrImpl attr) {
		String prefix = attr.getName();
		int index = prefix.indexOf(':');
		return index != -1 ? prefix.substring(index + 1) : "";
	}
}