package com.adobe.xfa.font;

import com.adobe.fontengine.font.Font;
import com.adobe.xfa.ut.UnitSpan;

/**
 * @exclude from published api.
 */

public class FontInstance implements Comparable<FontInstance> {
	private final FontItem mFontItem;
	private final double mXScale;
	private final double mYScale;
	private final UnitSpan mUnitSize;
	private final double mHorizontalScale;
	private final double mVerticalScale;

	FontInstance (FontItem item, UnitSpan unitSize) {
		this (item, unitSize, 1, 1);
	}

	FontInstance (FontItem item, UnitSpan unitSize, double horizontalScale, double verticalScale) {
		mFontItem = item;
		mUnitSize = unitSize;
		mHorizontalScale = horizontalScale;
		mVerticalScale = verticalScale;
		double scale = unitSize.valueAsUnit (UnitSpan.POINTS_1K) / 1000.0;
		mXScale = scale * mHorizontalScale;
		mYScale = scale * mVerticalScale;
	}

	public FontItem getFontItem () {
		return mFontItem;
	}

	public Font getAFEFont () {
		return mFontItem.getAFEFont();
	}

	public String getTypeface () {
		return (mFontItem == null) ? null : mFontItem.getTypeface();
	}

	public int getWeight () {
		return (mFontItem == null) ? FontInfo.WEIGHT_NORMAL : mFontItem.getWeight();
	}

	public boolean getItalic () {
		return (mFontItem == null) ? false : mFontItem.getItalic();
	}

	public UnitSpan getSize () {
		return mUnitSize;
	}

	public double getHorizontalScale () {
		return mHorizontalScale;
	}

	public double getVerticalScale () {
		return mVerticalScale;
	}

	public UnitSpan getAscent () {
		return toUnitSpanVertical (mFontItem.getAscent());
	}

	public UnitSpan getLegacyAscent () {
		return toUnitSpanVertical (mFontItem.getLegacyAscent());
	}

	public UnitSpan getDescent () {
		return toUnitSpanVertical (mFontItem.getDescent());
	}

	public UnitSpan getLineGap () {
		return toUnitSpanVertical (mFontItem.getLineGap());
	}

	public UnitSpan getLegacyLineGap () {
		return toUnitSpanVertical (mFontItem.getLegacyLineGap());
	}

	public UnitSpan getspacing () {
		return toUnitSpanVertical (mFontItem.getSpacing());
	}

	public float getCharWidth (int c, boolean useHorizontalGlyphs) {
		return (float) toDoubleHorizontal (mFontItem.getCharWidth (c, useHorizontalGlyphs));
	}

	public double getDoubleCharWidth (int c, boolean useHorizontalGlyphs) {
		assert (false);
		return 0;
	}

	public float getGlyphWidth (int glyphID, boolean useHorizontalGlyphs) {
		return (float) toDoubleHorizontal (mFontItem.getGlyphWidth (glyphID, useHorizontalGlyphs));
	}

	public int getGlyphID (int c) {
		return mFontItem.getGlyphID (c);
	}

	public boolean getGlyphID (int c, int g, boolean h) {
		assert (false);
		return false;
	}

	public void mapGlyph (int c, int glyphID) {
		mFontItem.mapGlyph (c, glyphID);
	}

	public boolean equals (Object object) {
		
		if (this == object)
			return true;
		
		// This overrides Object.equals(boolean) directly, so...
		if (object == null)
			return false;
		
		if (object.getClass() != getClass())
			return false;
		
		FontInstance other = (FontInstance)object;
		
		return UnitSpan.match (mUnitSize, other.mUnitSize)
			&& FontItem.match (mFontItem, other.mFontItem);
	}
	
	public int hashCode() {
		
		int result = mUnitSize == null ? 0 : mUnitSize.hashCode();
		result = (result * 31) ^ (mFontItem == null ? 0 : mFontItem.hashCode());
		return result;
	}

	public static boolean match (FontInstance o1, FontInstance o2) {
		if (o1 == o2) {
			return true;
		}
		if ((o1 == null) || (o2 == null)) {
			return false;
		}
		return o1.equals (o2);
	}

	private final double toDoubleHorizontal (double value) {
		return value * mXScale;
	}

	private final double toDoubleVertical (double value) {
		return value * mYScale;
	}

	private final UnitSpan toUnitSpanVertical (double value) {
		return new UnitSpan (toDoubleVertical (value), UnitSpan.POINTS_1K);
	}

	public int compareTo (FontInstance other) {
		if (other == null)
			throw new NullPointerException();
		
		int result = mFontItem.compareTo (other.mFontItem);
		return (result != 0) ? result :  mUnitSize.compareTo (other.getSize());
	}
}
