/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */
package com.adobe.xfa.form;

import com.adobe.xfa.ut.Peer;
import com.adobe.xfa.Node;
import com.adobe.xfa.XFA;
import com.adobe.xfa.Element;
import com.adobe.xfa.Obj;
import com.adobe.xfa.data.DataNode;


/**
 * FormDataListener manages the mapping (i.e., binding) between an Element in the FormModel and
 * its corresponding DataNode in the DataModel. An Element and DataNode that are
 * bound will each be peered to a FormDataListener.
 */
public class FormDataListener extends Obj {

	private Element		mFormNode;
	private DataNode	mDataNode;
	
	/** @exclude from published api. */
	public FormDataListener(Element 	formNode,
							DataNode	dataNode) {
		
		mFormNode = formNode;
		mDataNode = dataNode;

		//add peers
		mFormNode.addPeer(this);
		mDataNode.addPeer(this);
		
		// Set the mapped state on the form side; XFAFormModel.consumeDataNode() handles
		// the map flag on the data side.
		mFormNode.setMapped(true);
	}

	//	~FormDataListener()
	/** @exclude from published api. */
	void dispose() {
		// Set the mapped state
	
		// should call this.peerRemoved
		if (mFormNode != null) {
			mFormNode.setMapped(false);
			mFormNode.removePeer(this);
		}
		
		if (mDataNode != null) {
//#if 0
//			// Technically we should only unmap if we're the last peer.  Otherwise, creating 
//			// and then destroying a leader/trailer subform during layout will cause any data 
//			// nodes referenced from within the leader/trailer via MATCH_GLOBAL bindings to be 
//			// setMapped(TRUE) and then setMapped(FALSE), leaving them unmapped even though
//			// other nodes may still point to them.
//			//
//			// This code would fix that, but might also produce legacy issues.  Given that we
//			// don't have any reports of this bug, we're going to leave well enough alone for
//			// now.  [jey -- 10 June 2009]
//			//
//			if (mDataNode.isMapped() && mDataNode.getPeer(1) == null)
//				mDataNode.setMapped(false);
//#else
			mDataNode.setMapped(false);
//#endif
			mDataNode.removePeer(this);	
		}
	}

	/**
	 * Gets the DataNode associated with this mapping.
	 * @return the DataNode associated with this mapping.
	 */
	public DataNode getDataNode() {
		return mDataNode;
	}

	/**
	 * Gets the form Element associated with this mapping.
	 * @return the form Element associated with this mapping.
	 */
	public Element getFormNode() {
		return mFormNode;
	}

	// Used with incremental merge
	/** @exclude from published api. */
	public void setDataNode(DataNode newDataNode) {
		mDataNode.removePeer(this);	// this will call peerRemoved
		assert(mDataNode == null);
	
		mDataNode = newDataNode;
	
		// add peer
		mDataNode.addPeer(this);
		
		if (mFormNode instanceof FormField)
			((FormField)mFormNode).setFromData(null, true);
		else if (mFormNode instanceof FormExclGroup)
			((FormExclGroup)mFormNode).setFromData(null);
	}
	
	/**
	 * Determines whether a DataNode is currently mapped (i.e., bound) to a form node.
	 * @param dataNode
	 * @return <code>true</code> if <code>dataNode</code> is currently mapped to a form node.
	 */
	public static boolean isMapped(DataNode dataNode) {
		int nPeer = 0;
		Peer peer = dataNode.getPeer(nPeer);
		while (peer != null) {
			if (peer instanceof FormDataListener) {
				return true;
			}
			nPeer++;
			peer = dataNode.getPeer(nPeer);
		}
		return false;
	}

	/** @exclude from published api. */
	public boolean isFieldValueChange(Object peerNode, 
							   int eventType, 
							   String arg1, 
							   Object arg2) {
		
		if (eventType == Peer.VALUE_CHANGED ||
			eventType == Peer.DESCENDENT_VALUE_CHANGED) {
			if (peerNode == mDataNode) {
				return true;
			}
			else if (peerNode == mFormNode) {
				// only want to respond if the field value changed - not some other 
				// text property of the field such as caption or items (via complex binding)
				// i.e. update if we're within <field><value>...</value><field>
				if (mFormNode instanceof FormField) {
					for (Node oNode = (Node)arg2; oNode != null; oNode = oNode.getXFAParent()) {
						if (oNode.isSameClass(XFA.VALUETAG)) {
							Node oParent = oNode.getXFAParent();
							if (oParent != null && oParent == mFormNode) {
								return true;
							}
						}
						else if (oNode == mFormNode)
							break;
					}			
				}
				else if (mFormNode instanceof FormExclGroup) {
					for (Node oNode = (Node)arg2; oNode != null; oNode = oNode.getXFAParent()) {
						if (oNode.isSameClass(XFA.VALUETAG)) {
							Node oParent = oNode.getXFAParent();
							if (oParent instanceof FormField) {
								if (oParent.getXFAParent() == mFormNode) {
									return true;
								}
							}
						}
						else if (oNode == mFormNode)
							break;
					}
				}
			}
		}
		
		return false;
	}

	// overridden from Peer
	/** @exclude from published api. */
	public void updateFromPeer(Object peerNode, 
						int eventType, 
						String arg1, 
						Object arg2) {
		if (isFieldValueChange(peerNode, eventType, arg1, arg2)) {
			// block any other update calls
			deafen();
		
			// update the field
			if (peerNode == mDataNode) {
				if (mFormNode instanceof FormField)
					((FormField)mFormNode).setFromData(null, true);
				else if (mFormNode instanceof FormExclGroup)
					((FormExclGroup)mFormNode).setFromData(null);
			}
			// need to update the data node since our content changed
			else if (peerNode == mFormNode) {
				if (mFormNode instanceof FormField) {
					if (mFormNode instanceof FormChoiceListField)
						((FormChoiceListField)mFormNode).setData(null);
					else
						((FormField)mFormNode).setData(null);
				}
				else if (mFormNode instanceof FormExclGroup) {
					((FormExclGroup)mFormNode).setData(null);
				}
			}
		
			// allow update calls
			unDeafen();
		}
		else if (eventType == Peer.PERMS_LOCK_SET || eventType == Peer.PERMS_LOCK_CLEARED) {
			// block any other update calls
			deafen();
			
			if (peerNode == mDataNode) {	// data changed
				// update any form nodes that need updating
				if (mFormNode instanceof FormField ||
					mFormNode instanceof FormExclGroup ||
					mFormNode instanceof FormSubform) {
					if (eventType == Peer.PERMS_LOCK_SET)
						mFormNode.setPermsLock(true);
					else
						mFormNode.setPermsLock(false);
				}
			}
			else if (peerNode == mFormNode) {	// form changed
				// update any data nodes that need updating
				if (mFormNode instanceof FormField ||
					mFormNode instanceof FormExclGroup ||
					mFormNode instanceof FormSubform) {
					if (eventType == Peer.PERMS_LOCK_SET)
						mDataNode.setPermsLock(true);
					else
						mDataNode.setPermsLock(false);
				}
			}
		
			// allow update calls
			unDeafen();
		}
	}

	/** @exclude from published api. */
	public void peerRemoved(Peer peer) {
		if (peer == mDataNode) {
				// update the mapped state of the data node
				if (!isMapped(mDataNode)) {
					mDataNode.setMapped(false);
					Element parent = mDataNode.getXFAParent();
					if (parent != null && mDataNode.isDefault(false))
						mDataNode.remove();
				}
		
			mDataNode = null;
		}
		else if (peer == mFormNode) {
			mFormNode.setMapped(false);				
			mFormNode = null;
		}
	}
	
	/** @exclude from published api. */
	public String getClassName() {
		return "formDataListener";
	}
	
	/** @exclude from published api. */
	public String getClassAtom() {
		return "formDataListener";
	}
}
