/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */

package com.adobe.xfa.form;

import com.adobe.xfa.AppModel;
import com.adobe.xfa.Attribute;
import com.adobe.xfa.Delta;
import com.adobe.xfa.Element;
import com.adobe.xfa.Node;
import com.adobe.xfa.NodeList;
import com.adobe.xfa.ProtoableNode;
import com.adobe.xfa.StringAttr;
import com.adobe.xfa.STRS;
import com.adobe.xfa.TextNode;
import com.adobe.xfa.XFA;
import com.adobe.xfa.XFAList;
import com.adobe.xfa.content.Content;
import com.adobe.xfa.content.ExDataValue;
import com.adobe.xfa.data.DataModel;
import com.adobe.xfa.data.DataNode;
import com.adobe.xfa.template.containers.Container;
import com.adobe.xfa.template.containers.Field;
import com.adobe.xfa.template.Items;
import com.adobe.xfa.ut.ExFull;
import com.adobe.xfa.ut.MsgFormatPos;
import com.adobe.xfa.ut.Peer;
import com.adobe.xfa.ut.ResId;
import com.adobe.xfa.ut.StringUtils;
import com.adobe.xfa.LogMessage;

import java.util.ArrayList;
import java.util.List;


/**
 * @exclude from public api.
 */
public class FormField extends Field {

	private FormDataListener mDataListener;

    private Object mCPDField;

	private FormItemsDataListener mItemsDataListener;

	private List<FormListener> mListenerTable; // list of listeners

	private boolean mbValidateRegistered;
	
	private boolean mbCalculateRegistered;

	public FormField(Element parent, Node prevSibling) {
		super(parent, prevSibling);
	}

	public void remove() {
		if (getModel() instanceof FormModel) {
			((FormModel) getModel()).removeReferences(this);
		}

		super.remove();
	}

	void setDataNode(
			DataNode dataNode, 
			boolean bUpdateData /* = false */, 
			boolean bPeerNode /* = true */,
			String sConnectPicture /* = "" */,
			boolean bDefault /*=true*/ /*CL#708930 */) {
		
		// remove old listener
		if (mDataListener != null && bPeerNode) {
			mDataListener.dispose();
			mDataListener = null;
		}

		// Type and null check
		if (dataNode == null || dataNode.getClassTag() != XFA.DATAVALUETAG)
			return;

		// Establish a peer relationship with the data node
		if (bPeerNode)
			setFormDataListener(new FormDataListener(this, dataNode));

		// Apply data picture format that the field may have specified
		String sDataPictureFormat = "";
		if (bPeerNode) {
			Element bind = getElement(XFA.BINDTAG, true, 0, false, false);
			if (bind != null) {
				Element picture = bind.getElement(XFA.PICTURETAG, true, 0, false, false);
				if (picture != null) {
					TextNode textNode = (TextNode) picture.locateChildByClass(XFA.TEXTNODETAG, 0);
					if (textNode != null) {
						sDataPictureFormat = textNode.getValue();
					}
				}
			}
		} else {
			// connect picture was predetermined and passed in
			sDataPictureFormat = sConnectPicture;
		}

		if (!StringUtils.isEmpty(sDataPictureFormat)) {
			boolean bIsTextNode = false;
			Element value = getElement(XFA.VALUETAG, true, 0, false, false);
			if (value != null) {
				Node content = value.getOneOfChild(true, true);
				if (content != null && content.isSameClass(XFA.TEXTTAG))
					bIsTextNode = true;
			}

			String sLocale;
			ProtoableNode proto = getProto();
			if (proto != null)
				sLocale = proto.getInstalledLocale();
			else
				sLocale = getInstalledLocale();

			if (bPeerNode)
				getDataNode().setPictureFormat(sDataPictureFormat, sLocale, bIsTextNode);
			else
				dataNode.setPictureFormat(sDataPictureFormat, sLocale, bIsTextNode);
		}

		if (bPeerNode && getFormDataListener() != null)
			getFormDataListener().deafen();
		
		if (bUpdateData)
			setData(dataNode);
		else
			setFromData(dataNode, bDefault);
		
		if (bPeerNode && getFormDataListener() != null)
			getFormDataListener().unDeafen();
	}

	void setData(DataNode dataValue) {
		// null check
		if (dataValue == null) {
			dataValue = getDataNode();

			if (dataValue == null)
				return;
		}

		// if this field is null then set the data node to null;
		if (getIsNull(false)) {
			dataValue.setIsNull(true, true);

			return;
		}

		// Create an XFADataValue node

		// Getting a field value involves traversing either of these
		// hierarchies:
		//
		// <field>
		//   <value>
		//     <text>could also be an integer/decimal or other content</text>
		//   </value>
		// </field>
		//
		// <field>
		//   <value>
		//     <exData contentType="text/html">
		//       <body xmlns="http://www.w3.org/1999/xhtml"> ... </body>
		//     </exData>
		//   </value>
		// </field>

		Element value = getElement(XFA.VALUETAG, false, 0, false, false);
		Node valueContent = value.getOneOfChild();

		// Fix for Bug #39261. We were assuming that all children
		// of value had a textnode. This is not true for arcs, lines
		// and rectangles. Only nodes derived from Content exhibit
		// this behaviour.
		if (valueContent instanceof Content) {
			Content content = (Content)valueContent;
			Node valueContentChild = null;

			// set the content type of the dataValue for images
			if (valueContent.isSameClass(XFA.IMAGETAG)) {
				Attribute contentType = content.getAttribute(XFA.CONTENTTYPETAG);
				dataValue.setContentType(contentType.toString());

				// set the href attribute on the dv element
				if (valueContent.isPropertySpecified(XFA.HREFTAG, true, 0)) {
					Attribute href = content.getAttribute(XFA.HREFTAG);
					
					dataValue.setAttribute("", XFA.HREF, XFA.HREF, href.getAttrValue(), false);
				}
			}

			// A content node is one of <text>, <exData>, <decimal>,
			// <integer>, <date>, <time>, <dataTime>, <boolean>,
			// <float> or <image>
			//
			// A content value is one of <richTextNode>,
			// <textNode>
			boolean bIsValueContentExData = false; // added for bug 2358640
			if (dataValue.getContentType().equals(STRS.TEXTHTML)){
				bIsValueContentExData = true;
			}			
			if (valueContent.isSameClass(XFA.EXDATATAG)) {
				valueContentChild = content.getOneOfChild(true, false);
			} 
			else {
				valueContentChild = content.getText(true, false, false);
			}

			// Create the data node only if our value is represented by pcdata
			// or by the special richTextNode.
			// Don't allocate a dataNode if we're intended to inherit a value
			// via a link or reference.
			if (valueContentChild != null) {
				// If it's a text node then update the data value node with
				// the value of the text node. Otherwise, we have a
				// more complex structure to take care of.
				// ADDED FOR BUG 2358640 (Feb04,2010):
				//   If it is pure text node, we can just update the value, 
				//   otherwise - go the complex way.
				if (valueContentChild instanceof TextNode && !bIsValueContentExData) {
					dataValue.setValue(((TextNode)valueContentChild).getValue(), true);
				} 
				else if (valueContentChild.isSameClass(XFA.RICHTEXTNODETAG) || bIsValueContentExData) {
					dataValue.clearNull();

					// If it's an RichTextNode, then we need to clone the
					// html fragment and copy it as a child of the XFADataValue

					// Watson bug 0945102
					// if the data node is an attribute simply update the node
					// with the text value.
					if (dataValue.isAttribute()) {
						ExDataValue exData = (ExDataValue)valueContent;
						String sValue = exData.getStrValue();
						dataValue.setValue(sValue, true);
					} 
					else {
						// Import the htmlFragment from the template model to the dataModel
						Node importedNode = dataValue.getOwnerDocument().importNode(valueContentChild, true);

						// This node will have a single #text node child
						// added during the createNode method above.
						Node oldNode = dataValue.getFirstXMLChild();
						if (oldNode != null) {
							dataValue.replaceChild(importedNode, oldNode);
						} else {
							dataValue.appendChild(importedNode, false);
						}
					}

					// Fix for Watson bug#1242681 referring to Global binding changes in RTF.
					dataValue.deafen();
					dataValue.notifyPeers(VALUE_CHANGED, "", null);
					dataValue.unDeafen();
				}
			}
		}
	}// end setDataNode
	
	/**
	 * @exclude from public api.
	 */	
	public DataNode getDataNode() {
		if (null != getFormDataListener())
			return getFormDataListener().getDataNode();

		return null;
	}

	protected void setFormDataListener(FormDataListener dataListener) {
		if (mDataListener != null)
			mDataListener.dispose();
		
		mDataListener = dataListener;
	}

	protected FormDataListener getFormDataListener() {
		return mDataListener;
	}

	List<FormListener> getFormListeners(boolean bCreate) {
		if (bCreate && mListenerTable == null)
			mListenerTable = new ArrayList<FormListener>();
		return mListenerTable;
	}

	void setItemsDataListener(FormItemsDataListener listener) {
		mItemsDataListener = listener;
	}

	FormItemsDataListener getItemsDataListener() {
		return mItemsDataListener;
	}

	// clean up the data and datalistener
	void cleanupListeners() {
		
		if (mDataListener != null) {
			mDataListener.dispose();
			mDataListener = null;
		}
			
		if (mItemsDataListener != null) {
			mItemsDataListener.dispose();
			mItemsDataListener = null;	
		}

		if (mListenerTable != null) {
			mListenerTable.clear();
			mListenerTable = null;
		}
	}

	void resetData() {
		ProtoableNode proto = getProto();
		if (proto == null)
			return;

		Element value = proto.getElement(XFA.VALUETAG, true, 0, false, false);

		// Watson 1061950, a null value is valid
		setElement(value, XFA.VALUETAG, 0);

		// get the content node and set that as the param.
		Element value2 = getElement(XFA.VALUETAG, false, 0, false, false);
		Node content = value2.getOneOfChild();

		notifyPeers(Peer.DESCENDENT_VALUE_CHANGED, "", content);
	}
	
	/**
	 * For use by Gibson for XFAplugins.
	 * @exclude from public api.
	 */
    public Object getCPDField() {
	    return mCPDField;
    }

	/**
	 * For use by Gibson for XFAplugins.
	 * @exclude from public api.
	 */
    public void setCPDField(Object oCPDField) {
        mCPDField = oCPDField;
    }

	// Adobe patent application tracking # B252, entitled METHOD AND SYSTEM TO PERSIST STATE, inventors: Roberto Perelman, Chris Solc, Anatole Matveief, Jeff Young, John Brinkman
	// Adobe patent application tracking # B322, entitled METHOD AND SYSTEM TO MAINTAIN THE INTEGRITY OF A CERTIFIED DOCUMENT WHILE PERSISTING STATE IN A DYNAMIC FORM, inventors: Roberto Perelman, Chris Solc, Anatole Matveief, Jeff Young, John Brinkman
	/**
	 * @see Element#getDeltas(Element, XFAList)
	 * @exclude from public api.
	 */
	public void getDeltas(Element delta, XFAList list) {
		
		if (!isSameClass(delta))
			return;
	
		FormModel model = (FormModel)getModel();
		
		if (model.isLoading()) {
			// restore the value of the field if we are loading
			
			// grab the delta value node
			Element deltaValue =  delta.getElement(XFA.VALUETAG, true, 0, false, false);
			if (deltaValue != null) {
				// get the value node for this field
				Element value = getElement(XFA.VALUETAG, false, 0, false, false);
				
				XFAList list2 = new XFAList();
				value.getDeltas(deltaValue, list2);
				
				// always restore value deltas when loading
				int nLen = list2.length();
				for (int i = 0; i < nLen; i++) {
					Delta valueDelta = (Delta)list2.item(i);
					valueDelta.restore();
				}
			}
			
			// restore validate.disableAll if we are loading
			model.restoreValidateDisableAll(this, delta);
		}
		
		super.getDeltas(delta, list);
	}

	/*CL#708930*/	
	void setFromData(DataNode dataValue /* = null */, boolean bDefault /*=true*/) {
		// null check
		// boolean bDefault = true; //CL#708930 
		if (dataValue == null) {
			bDefault = false; // if we have a null data node that means we are called via an update form peer call
			// in which case we should clear the default flag for this field

			dataValue = getDataNode();

			if (dataValue == null)
				return;
		}

		boolean bIsRichText = false;
		boolean bDataIsNull = dataValue.getIsNull();

		// Get the Value and the Content from the formNode.
		Element value = getElement(XFA.VALUETAG, 0);
		Element valueContent = (Element)value.getOneOfChild();

		// Are we rich text?
		if (valueContent.isSameClass(XFA.EXDATATAG)) {
			bIsRichText = true;	
		
			// Do we need to Coerce the ExDataValue field to be of a different contentType?
			// Are we dealing with Rich Text?

			// Get the contentType of the XFADataValueNode
			String sDataValueContentType = dataValue.getContentType();
			
			// Get the contentType of the ExDataValue node.
			String sContentType = valueContent.getAttribute(XFA.CONTENTTYPETAG).toString();
			
			if (StringUtils.isEmpty(sDataValueContentType)) {
				
				// watson bug 1930441,  don't change the default content type for null fields
				// content type will be updated if the value of the field is updated.

				// default to text/plain
				if (!bDataIsNull)
					sDataValueContentType = STRS.TEXTPLAIN;
				else
					sDataValueContentType = sContentType;
				
			}

			// If the content type of the ExDataValue node is empty, null, or is
			// different from the XFADataValueNode then we want to coerce it into
			// the appropriate contentType based on the contentType of the
			// dataValueNode.
			if (StringUtils.isEmpty(sContentType) ||
				!sDataValueContentType.equals(sContentType)) {
				//set the contenttype of the new exdata element to text/plain
				if (sDataValueContentType.equals(STRS.TEXTPLAIN))
					valueContent.setAttribute(new StringAttr(XFA.CONTENTTYPE, sDataValueContentType), XFA.CONTENTTYPETAG);

				// Are we dealing with Rich Text?
				else if (sDataValueContentType.equals(STRS.TEXTHTML))
					bIsRichText = true;
				else {
					//warning;
					MsgFormatPos oFormatPos = new MsgFormatPos(ResId.IncompatibleContentType);
					oFormatPos.format(sDataValueContentType);
					oFormatPos.format(XFA.EXDATA);
					ExFull oEx = new ExFull(oFormatPos);
					getModel().addErrorList(oEx, LogMessage.MSG_INFORMATION, this);
				}
			}
			else if (sContentType.equals(STRS.TEXTHTML)) {
				bIsRichText = true;
			}
		}
		else if (valueContent.isSameClass(XFA.IMAGETAG)) {
			// Are we dealing with image data, if so ensure the content types are correct
			// and that the attributes are set.
			// NOTE: See Vantive bug #586347 about changes to the following "if" statement
			//       with respect to PA and the XFAPlugin.  The XFAPlugin 6.0 fix broke
			//       functionality for linked images and was removed.  The correct fix
			//       for the XFAPlugin issue will be address again in the 6.1 version.

			// Get the Value and the Content from the formNode.
			valueContent = (Element)value.getOneOfChild();

			// Get the contentType of the XFADataValueNode
			// Watson 1778671.  The default for an image contentType was
			// being set to "text/plain" which doesn't make sense.  Updated
			// code to default to "image/*" if a contentType was not specified.			
			String sDataValueContentType = dataValue.getContentType();
			if (StringUtils.isEmpty(sDataValueContentType))
				sDataValueContentType = "image/*";
				
			if (! (sDataValueContentType.equals(STRS.TEXTPLAIN) ||
				sDataValueContentType.contains("image/"))) {
				String sDataValue = dataValue.getValue(false);
				if (!StringUtils.isEmpty(sDataValue)) {
					//warning;
					MsgFormatPos formatPos = new MsgFormatPos(ResId.IncompatibleContentType);
					formatPos.format(sDataValueContentType);
					formatPos.format(XFA.IMAGE);
					ExFull ex = new ExFull(formatPos);
					getModel().addErrorList(ex, LogMessage.MSG_INFORMATION, this);
				}
			}


			// we want to coerce it into the appropriate contentType based on 
			// the contentType of the dataValueNode.
			valueContent.setAttribute(new StringAttr(XFA.CONTENTTYPE, sDataValueContentType), XFA.CONTENTTYPETAG);

			// get attributes from the data
			
			// set the href attribute on the image element

			// watson 1094235: If the href value is null we need to set the
			// attribute to an empty string otherwise we get into problems
			// when the data has a embed image and the template as a href link.
			String href = null;
			int index = dataValue.findAttr("", XFA.HREF);
			if (index != -1)
				href = dataValue.getAttrVal(index);
			
			valueContent.setAttribute(new StringAttr(XFA.HREF, href != null ? href : ""), XFA.HREFTAG);

			// set the transferEncoding attribute on the image element
			index = dataValue.findAttr("", XFA.TRANSFERENCODING);
			if (index != -1) {
				StringAttr attr = new StringAttr(XFA.TRANSFERENCODING, dataValue.getAttrVal(index));
				valueContent.setAttribute(attr, XFA.TRANSFERENCODINGTAG);
			}
		}

		// Now, We want to place the data node into our form hierarchy so that we'll
		// transparently use it to get/set field values.
		//
		// We will create an TextNode or RichTextNode if we are dealing
		// with RichText (whose parent will be the oValueContent node
		// created above) which will be peered to the XFADataValue XML Dom Node.
		//
		// A simple data node will have a #text child -- returned by
		// singleTextChild() which will contain the dataValueNode value.
		//
		// For complex data nodes (those with data node children)
		//
		//	i.e.   <jf01>You owe <amount>$10</amount></jf01>
		//
		// we'll use the aggregating dom node (returned by getDomPeer())
		//
		// The getValue() method on TextNode has been updated to handle
		// retrieving complex data nodes.
		//
		// Finally, the last line of code where pNewtext is created but
		// doesn't seem to be used, is taken care of by the constructor of
		// TextNode, which sets the parent, model and peer appropriately.
		// It will get cleaned when no longer referenced.

		// if the content of the data value is null set the value to be null one of child
		if (bDataIsNull) {
			setIsNull(true, true);
			return;
		}

		if (bIsRichText) {
			
			Node dvNode = dataValue.getXmlPeer();
			
			// Watson bug 0945102
			// if the data node is an attribute, simply update it with the string value
			ExDataValue exData = (ExDataValue)valueContent;
			if (dataValue.isAttribute()) {
				// use dataValue.getValue() to ensure proper picture formatting is applied				
				exData.setStrValue(dataValue.getValue(), true, bDefault);
			}
			else {
				// html fragment and copy it as a child of the XFADataValue
				Element fvNode = valueContent;
				
				// Import the htmlFragment from the template model to the
				// dataModel
				Node dvNodeValue = dvNode.getFirstXMLChild();
				
				// watson bug 1613046 the data node is not null it is empty.  
				// The null check is done up above
				if (dvNodeValue == null) {
					// use poDataValue->getValue() to ensure proper picture formatting is applied
					exData.setStrValue(dataValue.getValue(), true, bDefault);
				}
				else {
				
					Node importedNode = fvNode.getOwnerDocument().importNode(dvNodeValue, true);
			
					// JavaPort: don't need to append to XML peer as in C++ since ExData is single-DOM

					exData.setContent(importedNode, true);
				}
			}
		}
		else {
			if(getAppModel().getLegacySetting(AppModel.XFA_PATCH_W_2757988))
				((Content)valueContent).setStrValue(dataValue.getValue(), true, bDefault);
			else
				((Content)valueContent).setValue(dataValue.getValue(), true, true, bDefault);	
		}
	}

	public boolean getIsNull() {
		return getIsNull(true);
	}

	private boolean getIsNull(boolean bCheckData) {		
		DataNode dataNode = getDataNode();
		if (bCheckData && dataNode != null)
			return dataNode.getIsNull();
			
		Element value = getElement(XFA.VALUETAG, true, 0, false, false);
		if (value != null) {
			Node oneOfChild = value.getOneOfChild(true, false);
			if (oneOfChild == null)
				return true;
			if (!(oneOfChild instanceof Content))
				return true;
				
			Content content = (Content)oneOfChild;
				
			if (content.getIsNull())
				return true;
				
			if (!content.couldBeNull())
				return false;
		}
		else
			return true;
			
		return false;
	}

	private static class ExecEventRecursionRec {
		protected final FormField formfield;
		protected final String msActivity;

		public ExecEventRecursionRec(FormField formfield, String sActivity) {
			this.formfield = formfield;
			this.msActivity = sActivity;
		}

		public boolean equals(FormField aformfield, String sActivity) {
			return formfield == aformfield && msActivity.equals(sActivity);
		}
	}

	private final static ThreadLocal<List<ExecEventRecursionRec>> execEventRecursionListThreadLocal = new ThreadLocal<List<ExecEventRecursionRec>>() {
		protected List<ExecEventRecursionRec> initialValue() {
			return new ArrayList<ExecEventRecursionRec>();
		}
	};

	/**
	 * @exclude from public api.
	 */
	public void execEvent(String sActivity) {
		
		List<ExecEventRecursionRec> execEventRecursionList = execEventRecursionListThreadLocal.get();
		int isize = execEventRecursionList.size();
		for (int i = 0; i < isize; i++) {
			ExecEventRecursionRec r = execEventRecursionList.get(i);
			if (r.equals(this, sActivity))
				return;
		}
		
		execEventRecursionList.add(new ExecEventRecursionRec(this, sActivity));
		try {
			FormModel formModel = (FormModel) getModel();
			formModel.eventOccurred(sActivity, this);
		}
		finally {
			execEventRecursionList.remove(execEventRecursionList.size() - 1);
		}
	}

	/**
	 * @exclude from public api.
	 */	
	public boolean execValidate() {
		FormModel formModel = (FormModel)getModel();
		FormModel.Validate validate = formModel.getDefaultValidate();

		formModel.validate(validate, this, false, false);

		if (validate != null && validate.getFailCount() > 0)
			return false;

		return true;
	}

    public boolean getRegistered(int eActivity) {
        if (eActivity == XFA.VALIDATETAG)
            return mbValidateRegistered;
        else if (eActivity == XFA.CALCULATETAG)
            return mbCalculateRegistered;
        return false;
    }

	void setRegistered(int eActivity) {
		if (eActivity == XFA.VALIDATETAG)
			mbValidateRegistered = true;
		else if (eActivity == XFA.CALCULATETAG)
		    mbCalculateRegistered = true;
	}

	/**
	 * @exclude from public api.
	 */
	public void notifyPeers(int eventType,
		String arg1,
		Object pArg2) {
		
		// XFA doesn't notify of changes during loads, there is no need and it improves load performance		
		if (getModel() != null && getModel().isLoading())
			return;
		
		// If validate changes at run time and there are no validations are
		// registered for this node, then we need to register it so that validations
		// will fire.
		
		if (!mbValidateRegistered && pArg2 instanceof FormModel.Validate) {
			if (eventType == Peer.DESCENDENT_ATTR_CHANGED ||
				eventType == Peer.DESCENDENT_ADDED) {
				FormModel poFormModel = (FormModel)getModel();
				poFormModel.registerEvents(this, FormModel.XFAEVENTTYPE_VALIDATE);
			}
		}
		
		super.notifyPeers(eventType, arg1, pArg2);
	}

	void updateItemsFromData(Element bindItems, boolean bIsConnectionBind /* =false */) {
		// first make sure field is a list
		Element ui = getElement(XFA.UITAG, true, 0, false, false);
		if (ui == null)
			return;

		Node currentUI = ui.getOneOfChild(true, false);
		if (currentUI == null || !currentUI.isSameClass(XFA.CHOICELIST)) {
			return;
		}

		// verify the binditems node
		if (bindItems == null || !bindItems.isSameClass(XFA.BINDITEMSTAG))
			return;

		// get the data ref details
		String sLabelRef = "";
		String sValueRef = "";
		Attribute labelRef = bindItems.getAttribute(XFA.LABELREFTAG, true, false);
		if (labelRef != null)
			sLabelRef = labelRef.toString();
		Attribute valueRef = bindItems.getAttribute(XFA.VALUEREFTAG, true, false);
		if (valueRef != null)
			sValueRef = valueRef.toString();

		if (sValueRef.length() == 0 && sLabelRef.length() == 0) {
			// nothing to do if both are empty
			return;
		}

		// look for a ref attribute
		String sRef = "";
		Attribute ref = bindItems.getAttribute(XFA.REFTAG, true, false);
		if (ref != null)
			sRef = ref.toString();

		if (sRef.length() == 0 && !bIsConnectionBind) {
			// try the sourceset
			updateItemsFromDatabase(bindItems, sValueRef, sLabelRef);
			return;
		}

		if (sRef.length() == 0) {
			// nothing to look at
			return;
		}

		NodeList refNodes;
		Node dataNode = null;
		if (bIsConnectionBind) {
			Node node = this;
			while (node != null) {
				if ((node instanceof Container) && ((Container) node).getFormInfo() != null)
					dataNode = ((Container) node).getFormInfo().connectionDataNode;
				if (dataNode != null)
					break;
				node = node.getXFAParent();
			}
		} else {
			dataNode = getDataNode();
		}

		if (dataNode != null) {
			refNodes = dataNode.resolveNodes(sRef, true, false, true);
		} else {
			// absolute ref?
			refNodes = resolveNodes(sRef, true, false, true);
		}

		if (refNodes.length() == 0)
			return;

		boolean bMultiColumn = true;
		if (sValueRef.length() == 0 || sLabelRef.length() == 0)
			bMultiColumn = false;

		Field.ItemPair itemPair = new Field.ItemPair();
		// Retrieve the bound and text item lists.
		getItemLists(false, itemPair, bMultiColumn);

		Items displayItems = itemPair.mDisplayItems;
		Items saveItems = itemPair.mSaveItems;
		
		// list items coming from a connection don't have their values persisted out
		// so mark them as non defaults;
		boolean bMarkAsDefault = !bIsConnectionBind;

		// okay - ready to go - clear existing list items first
		if (displayItems != null)
			displayItems.clearItems(bMarkAsDefault);
		if (saveItems != null)
			saveItems.clearItems(bMarkAsDefault);

		FormItemsDataListener listener = getItemsDataListener();

		int nItems = refNodes.length();
		for (int i = 0; i < nItems; i++) {
			Node refNode = (Node)refNodes.item(i);

			String sDisplayText = "";
			String sSaveText = "";

			if (sLabelRef.length() != 0) {
				Node labelNode = refNode.resolveNode(sLabelRef, true, false, false);
				if (labelNode != null && labelNode.isSameClass(XFA.DATAVALUETAG)) {
					sDisplayText = ((DataNode)labelNode).getValue();

					if (listener != null)
						listener.addDataNode(labelNode, true);
				}
			}
			if (sValueRef.length() != 0) {
				Node valueNode = refNode.resolveNode(sValueRef, true, false, false);
				if (valueNode != null && valueNode.isSameClass(XFA.DATAVALUETAG)) {
					sSaveText = ((DataNode)valueNode).getValue();

					if (listener != null)
						listener.addDataNode(valueNode, true);
				}
			}

			if (listener != null)
				listener.addDataNode(refNode, false);

			if (!bMultiColumn) {
				String sText = sDisplayText;
				if (sLabelRef.length() == 0)
					sText = sSaveText;

				sSaveText = sText;
			}

			if (displayItems != null)
				displayItems.addItem(sDisplayText, bMarkAsDefault);
			if (saveItems != null && saveItems != displayItems)
				saveItems.addItem(sSaveText, bMarkAsDefault);

		}

		Node refNode = (Node)refNodes.item(0);
		Node refParent = (Node)refNode.getXFAParent();
		if (refParent != null && listener != null) {
			listener.addDataNode(refParent, false);
		}
	}

	void updateItemsFromDatabase(Element bindItems, String sValueRef, String sLabelRef) {
		// check if it's a database binding
		String sConnection = "";
		Attribute connection = bindItems.getAttribute(XFA.CONNECTIONTAG,
				true, false);
		if (connection != null)
			sConnection = connection.toString();

		if (sConnection.length() == 0)
			return;

		DataModel dataModel = DataModel.getDataModel(getAppModel(), false, false);
		if (dataModel == null)
			return;

		DataModel.SourceSetLink link = dataModel.getSourceSetLink();
		if (link == null || !link.isSource(this, sConnection))
			return;

		List<String> valueData = new ArrayList<String>();
		List<String> labelData = new ArrayList<String>();

		if (sValueRef.length() != 0)
			valueData = link.getColumnData(this, sConnection, sValueRef);
		if (sLabelRef.length() != 0)
			labelData = link.getColumnData(this, sConnection, sLabelRef);

		if (valueData.size() == 0 && labelData.size() == 0)
			return;

		boolean bMultiColumn = true;
		int nItemCount = 0;
		if (valueData.size() != 0 && labelData.size() != 0) {
			nItemCount = valueData.size();
			if (labelData.size() < nItemCount)
				nItemCount = labelData.size();
		} else if (valueData.size() != 0) {
			nItemCount = valueData.size();
			bMultiColumn = false;
		} else {
			valueData = labelData;
			nItemCount = valueData.size();
			bMultiColumn = false;
		}

		// Retrieve the bound and text item lists.
		ItemPair itemPair = new ItemPair();
		getItemLists(false, itemPair, bMultiColumn);

		Items displayItems = itemPair.mDisplayItems;
		Items saveItems = itemPair.mSaveItems;

		// okay - ready to go - clear existing list items first
		if (displayItems != null)
			displayItems.clearItems(true);
		if (saveItems != null)
			saveItems.clearItems(true);

		for (int i = 0; i < nItemCount; i++) {
			String sDisplayText;
			String sSaveText;

			if (bMultiColumn) {
				sDisplayText = labelData.get(i);
				sSaveText = valueData.get(i);
			} else {
				sDisplayText = valueData.get(i);
				sSaveText = valueData.get(i);
			}

			if (displayItems != null)
				displayItems.addItem(sDisplayText, true);
			if (saveItems != null && saveItems != displayItems)
				saveItems.addItem(sSaveText, true);
		}
	}
}
