/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2007 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */
package com.adobe.xfa.formcalc;


import com.adobe.xfa.ut.UnitSpan;


/**
 * This class defines static methods to implement
 * the FormCalc unitvalue and unittype functions.
 *
 *  U N I T   F U N C T I O N S
 *      unitvalue, unittype.
 *
 * @author Mike P. Tardif
 *
 * @exclude from published api.
 */
final class BuiltinUnit {

	/*
	 *  Disallow instances of this class.
	 */
	private BuiltinUnit() {
	}

	/*
	 * UnitValue(s1[, s2])
	 *	where s1 is a unitspan string,
	 *	and s2 is an optional unit string to convert the unitspan value to.
	 * For example
	 *  unitvalue("12in")
	 * returns 12, whereas
	 *  unitvalue("1.0in", "cm")
	 * returns 2.54
	 */
	static void UnitValue(CalcParser oParser, CalcSymbol[] oArgSym) {
		final int nArgs = oArgSym.length;
		CalcSymbol oRetSym = null; 
		try {
			//
			// check if identifier is a keyword.
			//
			//
			// check the number of args vs the number required.
			//
			Builtins.minArgs(nArgs, 1);
			Builtins.maxArgs(nArgs, 2);
			//
			// check for error-valued, return-valued and null-valued args.
			//
			Builtins.limitExceptionArgs(oArgSym);
			Builtins.limitNullArgs(oParser, nArgs, oArgSym);
			//
			// retrieve the given argument(s), and compute unit value.
			//
			UnitSpan oUnit = new UnitSpan(oParser.getString(oArgSym[0]));
			int eCode;
			if (nArgs > 1)
				eCode = UnitSpan.stringToUnit(oParser.getString(oArgSym[1]),
													UnitSpan.INCHES_72K);
			else
				eCode = oUnit.units();
			double d = UnitSpan.valueToUnit(oUnit.valueAsUnit(eCode), eCode);
			oRetSym = new CalcSymbol(d);
		} catch (CalcException e) {
			oRetSym = e.getSymbol();
			if (oRetSym.getType() != CalcSymbol.TypeNull)
				oParser.mbInThrow = true;
		}
		//
		// push the result on the stack.
		//
		oParser.mStack.push(oRetSym);
	}


	/*
	 * UnitType(s1)
	 *	where s1 is a unitspan string.
	 * For example
	 *  unittype("72pt")
	 * returns "pt".
	 */
	static void UnitType(CalcParser oParser, CalcSymbol[] oArgSym) {
		final int nArgs = oArgSym.length;
		CalcSymbol oRetSym = null; 
		try {
			//
			// check the number of args vs the number required.
			//
			Builtins.minArgs(nArgs, 1);
			Builtins.maxArgs(nArgs, 1);
			//
			// check for error-valued, return-valued and null-valued args.
			//
			Builtins.limitExceptionArgs(oArgSym);
			Builtins.limitNullArgs(oParser, nArgs, oArgSym);
			//
			// retrieve the given argument(s), and compute unit type.
			//
			UnitSpan oUnit = new UnitSpan(oParser.getString(oArgSym[0]));
			String sRes = null;
			if (oUnit.units() != UnitSpan.UNIT_UNKNOWN) {
				sRes = UnitSpan.unitToString(oUnit.units());
				int nComma = sRes.indexOf(',');
				if (nComma >= 0)
					sRes = sRes.substring(0, nComma);
			}
			oRetSym = new CalcSymbol(sRes);
		} catch (CalcException e) {
			oRetSym = e.getSymbol();
			if (oRetSym.getType() != CalcSymbol.TypeNull)
				oParser.mbInThrow = true;
		}
		//
		// push the result on the stack.
		//
		oParser.mStack.push(oRetSym);
	}

}
