/*************************************************************************
 *
 *	File: PDF417CompactorManager.java
 *
 **************************************************************************
 * 
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2011 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.xfa.pmp.adobepdf417pmp;

import java.util.ArrayList;
import java.util.List;

/**
 * Ported from PDF417CompactManager.cpp
 */
class PDF417CompactorManager {
	// ////////////////////////////////////////////////////////////////////
	/**
	 * Do the byte compaction. This algorithm examines 12 bytes of the message
	 * at a time. If the 12 bytes can be compacted using text compaction, it
	 * uses text compaction, else byte compaction is used.
	 * 
	 * @param message
	 *            - The input binary message.
	 * @returns A vector of the of the binary encoded message in PDF417 code
	 *          words.
	 */
	// ////////////////////////////////////////////////////////////////////
	static void compact(char[] message, List<Integer> compactedMessage) {
		compactedMessage.add(0); // save room for the size

		int mode = 0; // text

		List<Character> packetVector = new ArrayList<Character>();
		List<Character> messageVector = new ArrayList<Character>();
		boolean canHandleText = true;
		for (int idx = 0; idx < message.length; idx++) {
			char v = message[idx];
			if (canHandleText)
				canHandleText = (canHandleText && PDF417TextCompactor
						.canHandle(v));
			packetVector.add(v);

			if (idx % 12 == 11) {
				switch (mode) {
				case 0: // text
					if (!canHandleText) {
						// need to switch to byte compactor
						// do the text compaction on what we have
						if (messageVector.size() > 0) {
							List<Integer> data = new ArrayList<Integer>();
							PDF417TextCompactor.compact(messageVector, data);
							compactedMessage.addAll(data);
							messageVector.clear();
						}
						mode = 1; // byte
					}
					messageVector.addAll(packetVector);
					packetVector.clear();
					break;

				case 1: // byte
					if (canHandleText) {
						// need to switch to text compactor
						// do the text compaction on what we have
						if (messageVector.size() > 0) {
							List<Integer> data = new ArrayList<Integer>();
							PDF417ByteCompactor.compact(messageVector, data);
							compactedMessage.addAll(data);
							messageVector.clear();
						}
						mode = 0; // text
					} 
					messageVector.addAll(packetVector);
					packetVector.clear();
					break;
				}
				canHandleText = true; // reset the handle of text
			}
		}

		// Handle the last packet
		if (packetVector.size() > 0) {
			switch (mode) {
			case 0: // text
				if (!canHandleText) {
					// need to switch to byte compactor
					// do the text compaction on what we have
					if (messageVector.size() > 0) {
						List<Integer> data = new ArrayList<Integer>();
						PDF417TextCompactor.compact(messageVector, data);
						compactedMessage.addAll(data);
						messageVector.clear();
					}
					mode = 1; // byte
				}
				messageVector.addAll(packetVector);
				packetVector.clear();
				break;

			case 1: // byte
				if (canHandleText) {
					// need to switch to text compactor
					// do the text compaction on what we have
					if (messageVector.size() > 0) {
						List<Integer> data = new ArrayList<Integer>();
						PDF417ByteCompactor.compact(messageVector, data);
						compactedMessage.addAll(data);
						messageVector.clear();
					}
					mode = 0; // text
					
				}
				messageVector.addAll(packetVector);
				packetVector.clear();
				break;
			}
			canHandleText = true; // reset the handle of text
		}

		// compact the last bits
		if (messageVector.size() > 0) {
			List<Integer> data = new ArrayList<Integer>();
			switch (mode) {
			case 0: // text
				PDF417TextCompactor.compact(messageVector, data);
				break;
			case 1: // byte
				PDF417ByteCompactor.compact(messageVector, data);
				break;
			}
			compactedMessage.addAll(data);
		}

		// Set the size
		compactedMessage.set(0, compactedMessage.size());
	}
}
