/*************************************************************************
 *
 *	File: DataMatrixC40Compactor.java
 *
 **************************************************************************
 * 
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2011 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.xfa.pmp.datamatrixpmp;

import java.util.ArrayList;
import java.util.List;

/**
 * Ported from DataMatrixC40Compactor.cpp
 */
class DataMatrixC40Compactor extends DataMatrixBaseCompactor {
	static class C40Encoding {
		int messageValue;
		List<Integer> c40Value = new ArrayList<Integer>();
	}

	static class CharacterSet {
		int messageValue;
		int codeValue;

		CharacterSet(int messageValue, int codeValue) {
			super();
			this.messageValue = messageValue;
			this.codeValue = codeValue;
		}

	}

	static final CharacterSet[] basicSet = { new CharacterSet(32, 3), // space
			new CharacterSet(48, 4), // 0
			new CharacterSet(49, 5), // 1
			new CharacterSet(50, 6), // 2
			new CharacterSet(51, 7), // 3
			new CharacterSet(52, 8), // 4
			new CharacterSet(53, 9), // 5
			new CharacterSet(54, 10), // 6
			new CharacterSet(55, 11), // 7
			new CharacterSet(56, 12), // 8
			new CharacterSet(57, 13), // 9
			new CharacterSet(65, 14), // A
			new CharacterSet(66, 15), // B
			new CharacterSet(67, 16), // C
			new CharacterSet(68, 17), // D
			new CharacterSet(69, 18), // E
			new CharacterSet(70, 19), // F
			new CharacterSet(71, 20), // G
			new CharacterSet(72, 21), // H
			new CharacterSet(73, 22), // I
			new CharacterSet(74, 23), // J
			new CharacterSet(75, 24), // K
			new CharacterSet(76, 25), // L
			new CharacterSet(77, 26), // M
			new CharacterSet(78, 27), // N
			new CharacterSet(79, 28), // O
			new CharacterSet(80, 29), // P
			new CharacterSet(81, 30), // Q
			new CharacterSet(82, 31), // R
			new CharacterSet(83, 32), // S
			new CharacterSet(84, 33), // T
			new CharacterSet(85, 34), // U
			new CharacterSet(86, 35), // V
			new CharacterSet(87, 36), // W
			new CharacterSet(88, 37), // X
			new CharacterSet(89, 38), // Y
			new CharacterSet(90, 39) // Z
	};

	static final CharacterSet[] shift1Set = { new CharacterSet(0, 0), // NULL
			new CharacterSet(1, 1), // SOH
			new CharacterSet(2, 2), // STX
			new CharacterSet(3, 3), // ETX
			new CharacterSet(4, 4), // EOT
			new CharacterSet(5, 5), // ENQ
			new CharacterSet(6, 6), // ACK
			new CharacterSet(7, 7), // BEL
			new CharacterSet(8, 8), // BS
			new CharacterSet(9, 9), // HT
			new CharacterSet(10, 10), // LF
			new CharacterSet(11, 11), // VT
			new CharacterSet(12, 12), // FF
			new CharacterSet(13, 13), // CR
			new CharacterSet(14, 14), // SO
			new CharacterSet(15, 15), // SI
			new CharacterSet(16, 16), // DLE
			new CharacterSet(17, 17), // DC1
			new CharacterSet(18, 18), // DC2
			new CharacterSet(19, 19), // DC3
			new CharacterSet(20, 20), // DC4
			new CharacterSet(21, 21), // NAK
			new CharacterSet(22, 22), // SYN
			new CharacterSet(23, 23), // ETB
			new CharacterSet(24, 24), // CAN
			new CharacterSet(25, 25), // EM
			new CharacterSet(26, 26), // SUB
			new CharacterSet(27, 27), // ESC
			new CharacterSet(28, 28), // FS
			new CharacterSet(29, 29), // GS
			new CharacterSet(30, 30), // RS
			new CharacterSet(31, 31), // US
	};

	static final CharacterSet[] shift2Set = { new CharacterSet(33, 0), // !
			new CharacterSet(34, 1), // "
			new CharacterSet(35, 2), // #
			new CharacterSet(36, 3), // $
			new CharacterSet(37, 4), // %
			new CharacterSet(38, 5), // &
			new CharacterSet(39, 6), // `
			new CharacterSet(40, 7), // (
			new CharacterSet(41, 8), // )
			new CharacterSet(42, 9), // *
			new CharacterSet(43, 10), // +
			new CharacterSet(44, 11), // ,
			new CharacterSet(45, 12), // -
			new CharacterSet(46, 13), // .
			new CharacterSet(47, 14), // /
			new CharacterSet(58, 15), // :
			new CharacterSet(59, 16), // ;
			new CharacterSet(60, 17), // <
			new CharacterSet(61, 18), // =
			new CharacterSet(62, 19), // >
			new CharacterSet(63, 20), // ?
			new CharacterSet(64, 21), // @
			new CharacterSet(91, 22), // [
			new CharacterSet(92, 23), // '\'
			new CharacterSet(93, 24), // ]
			new CharacterSet(94, 25), // ^
			new CharacterSet(95, 26) // _
	};

	static final CharacterSet[] shift3Set = { new CharacterSet(96, 0), // '
			new CharacterSet(97, 1), // a
			new CharacterSet(98, 2), // b
			new CharacterSet(99, 3), // c
			new CharacterSet(100, 4), // d
			new CharacterSet(101, 5), // e
			new CharacterSet(102, 6), // f
			new CharacterSet(103, 7), // g
			new CharacterSet(104, 8), // h
			new CharacterSet(105, 9), // i
			new CharacterSet(106, 10), // j
			new CharacterSet(107, 11), // k
			new CharacterSet(108, 12), // l
			new CharacterSet(109, 13), // m
			new CharacterSet(110, 14), // n
			new CharacterSet(111, 15), // o
			new CharacterSet(112, 16), // p
			new CharacterSet(113, 17), // q
			new CharacterSet(114, 18), // r
			new CharacterSet(115, 19), // s
			new CharacterSet(116, 20), // t
			new CharacterSet(117, 21), // u
			new CharacterSet(118, 22), // v
			new CharacterSet(119, 23), // w
			new CharacterSet(120, 24), // x
			new CharacterSet(121, 25), // y
			new CharacterSet(122, 26), // z
			new CharacterSet(123, 27), // new CharacterSet(
			new CharacterSet(124, 28), // |
			new CharacterSet(125, 29), // )
			new CharacterSet(126, 30), // ~
			new CharacterSet(127, 31) // DEL
	};

	// ////////////////////////////////////////////////////////////////////
	/**
	 * Do the byte compaction.
	 * 
	 * @param message
	 *            - The input binary message.
	 * @returns A vector of the of the binary encoded message in PDF417 code
	 *          words.
	 */
	// ////////////////////////////////////////////////////////////////////
	void compact(char[] message) {
		m_codeWords.clear();
		m_valid = false;

		C40Encoding[] c40Encoding = new C40Encoding[message.length];

		int messageSize = message.length;
		int idx;
		for (idx = 0; idx < messageSize; idx++) {
			int v = message[idx];

			int jdx;
			boolean found = false;
			for (jdx = 0; (jdx < basicSet.length) && !found; jdx++) {
				if (v == basicSet[jdx].messageValue) {
					c40Encoding[idx] = new C40Encoding();
					c40Encoding[idx].messageValue = v;
					c40Encoding[idx].c40Value.add(basicSet[jdx].codeValue);
					found = true;
					continue;
				}
			}

			for (jdx = 0; (jdx < shift1Set.length) && !found; jdx++) {
				if (v == shift1Set[jdx].messageValue) {
					c40Encoding[idx] = new C40Encoding();
					c40Encoding[idx].messageValue = v;
					c40Encoding[idx].c40Value.add(0); // Shift 1
					c40Encoding[idx].c40Value.add(shift1Set[jdx].codeValue);
					found = true;
					continue;
				}
			}

			for (jdx = 0; (jdx < shift2Set.length) && !found; jdx++) {
				if (v == shift2Set[jdx].messageValue) {
					c40Encoding[idx] = new C40Encoding();
					c40Encoding[idx].messageValue = v;
					c40Encoding[idx].c40Value.add(1); // Shift 2
					c40Encoding[idx].c40Value.add(shift2Set[jdx].codeValue);
					found = true;
					continue;
				}
			}

			for (jdx = 0; (jdx < shift3Set.length) && !found; jdx++) {
				if (v == shift3Set[jdx].messageValue) {
					c40Encoding[idx] = new C40Encoding();
					c40Encoding[idx].messageValue = v;
					c40Encoding[idx].c40Value.add(2); // Shift 3
					c40Encoding[idx].c40Value.add(shift3Set[jdx].codeValue);
					found = true;
					continue;
				}
			}

			// Must be in the 128 - 255 range
			if (!found) {
				int v1 = v - 128;

				for (jdx = 0; (jdx < basicSet.length) && !found; jdx++) {
					if (v1 == basicSet[jdx].messageValue) {
						c40Encoding[idx] = new C40Encoding();
						c40Encoding[idx].messageValue = v;
						c40Encoding[idx].c40Value.add(1); // Shift 2
						c40Encoding[idx].c40Value.add(30); // Upper Shift
						c40Encoding[idx].c40Value.add(basicSet[jdx].codeValue);
						found = true;
						continue;
					}
				}

				for (jdx = 0; (jdx < shift1Set.length) && !found; jdx++) {
					if (v1 == shift1Set[jdx].messageValue) {
						c40Encoding[idx] = new C40Encoding();
						c40Encoding[idx].messageValue = v;
						c40Encoding[idx].c40Value.add(1); // Shift 2
						c40Encoding[idx].c40Value.add(30); // Upper Shift
						c40Encoding[idx].c40Value.add(0); // Shift 1
						c40Encoding[idx].c40Value.add(shift1Set[jdx].codeValue);
						found = true;
						continue;
					}
				}

				for (jdx = 0; (jdx < shift2Set.length) && !found; jdx++) {
					if (v1 == shift2Set[jdx].messageValue) {
						c40Encoding[idx] = new C40Encoding();
						c40Encoding[idx].messageValue = v;
						c40Encoding[idx].c40Value.add(1); // Shift 2
						c40Encoding[idx].c40Value.add(30); // Upper Shift
						c40Encoding[idx].c40Value.add(1); // Shift 2
						c40Encoding[idx].c40Value.add(shift2Set[jdx].codeValue);
						found = true;
						continue;
					}
				}

				for (jdx = 0; (jdx < shift3Set.length) && !found; jdx++) {
					if (v1 == shift3Set[jdx].messageValue) {
						c40Encoding[idx] = new C40Encoding();
						c40Encoding[idx].messageValue = v;
						c40Encoding[idx].c40Value.add(1); // Shift 2
						c40Encoding[idx].c40Value.add(30); // Upper Shift
						c40Encoding[idx].c40Value.add(2); // Shift 3
						c40Encoding[idx].c40Value.add(shift3Set[jdx].codeValue);
						found = true;
						continue;
					}
				}

				if (!found) {
					// TO DO ???
				}
			}
		}

		// Concatenate the c40 codewords
		List<Integer> textCodeWords = new ArrayList<Integer>();
		for (idx = 0; idx < messageSize; idx++) {
			int jdx;
			for (jdx = 0; jdx < c40Encoding[idx].c40Value.size(); jdx++)
				textCodeWords.add(c40Encoding[idx].c40Value.get(jdx));
		}

		int c40Size = textCodeWords.size();

		if (c40Size < 3) {
			// For small messages, just use ASCII encoding
			return;
		}

		// The C40 encoding must end on a mod 3 boundary
		int c40End = messageSize - 1;
		while ((c40Size % 3 != 0) && (c40End > 0)) {
			c40Size -= c40Encoding[c40End].c40Value.size();
			c40End--;
		}

		c40End++;

		if (c40End > 0)
			m_codeWords.add(230); // C40 latch

		// Encode the real code words
		for (idx = 0; idx < c40Size; idx += 3) {
			int value16 = textCodeWords.get(idx) * 1600
					+ textCodeWords.get(idx + 1) * 40
					+ textCodeWords.get(idx + 2) + 1;
			m_codeWords.add((value16 / 256));
			m_codeWords.add((value16 % 256));
		}

		// To properly handle the remaining characters, we need to
		// know what symbol size might contain the data.
		int tempSymbolSize = findSymbolSize(m_codeWords.size());
		if (tempSymbolSize < 0)
			return; // Can't fit the data in a symbol

		// How much room is left in the symbol?
		int remainSymbolCodeWords = DataMatrixInfo.DATA_SYMBOLS[tempSymbolSize]
				- m_codeWords.size();

		// How much data remains?
		int remainTextCodeWords = 0;
		for (idx = c40End; idx < messageSize; idx++)
			remainTextCodeWords += c40Encoding[idx].c40Value.size();

		// Rule from standard:
		// If two symbol characters remain and three C40 values remain
		// to be encoded (which may include both data and shift chracters)
		// encode the three C40 values in the last symbol characters.
		// A final Unlatch codeword is not required.
		if (remainTextCodeWords == 0) {
			if (remainSymbolCodeWords > 0) {
				// Unlatch to ASCII.
				m_codeWords.add(254);

				// Add padding.
				DataMatrixPadder.addPadding(m_codeWords, tempSymbolSize);

				m_symbolSize = tempSymbolSize;
				m_valid = true;
			} else {
				// The message just fits!
				m_symbolSize = tempSymbolSize;
				m_valid = true;
			}
		} else
		// Rule from standard:
		// If one symbol character remains and one C40 value
		// (data character) remains to be encoded, the last symbol
		// character is encoded with the data character using
		// ASCII encodation scheme. The Unlatch character is not
		// encoded, but is assumed, before the last symbol character
		// whenever an odd number of symbol characters remain in
		// the symbol in the C40 encodation scheme.
		if ((remainSymbolCodeWords == 1) && (remainTextCodeWords == 1)) {
			// Encode the last character using ASCII, no need to unlatch, or
			// padding.
			if (c40Encoding[c40End].messageValue < 128) {
				m_codeWords.add(c40Encoding[c40End].messageValue + 1);

				m_symbolSize = tempSymbolSize;
				m_valid = true;
			} else {
				// We need to shift the ASCII encoding.
				m_codeWords.add(254); // Unlatch
				m_codeWords.add(235); // Upper shift
				m_codeWords.add(c40Encoding[c40End].messageValue - 127);

				// Find the symbol size that will fit the data.
				m_symbolSize = findSymbolSize(m_codeWords.size());
				if (m_symbolSize == -1)
					return; // Can't fit the data

				// Pad the data. No need to unlatch.
				DataMatrixPadder.addPadding(m_codeWords, m_symbolSize);

				m_valid = true;
			}
		} else
		// Rule from standard:
		// If two symbol characters remain and only one C40 value
		// (data character) remains to be encoded, the first symbol
		// is encoded as an Unlatch character and the last symbol
		// character is encoded with the data character using
		// the ASCII encodation scheme.
		if ((remainSymbolCodeWords == 2) && (remainTextCodeWords == 1)) {
			if (c40Encoding[c40End].messageValue < 128) {
				m_codeWords.add(254); // Unlatch

				// Encode the last character using ASCII encoding
				m_codeWords.add(c40Encoding[c40End].messageValue + 1);

				m_symbolSize = tempSymbolSize;
				m_valid = true;
			} else {
				// We need to shift the ASCII encoding.
				m_codeWords.add(254); // Unlatch
				m_codeWords.add(235); // Upper shift
				m_codeWords.add(c40Encoding[c40End].messageValue - 127);

				// Find the symbol size that will fit the data.
				m_symbolSize = findSymbolSize(m_codeWords.size());
				if (m_symbolSize == -1)
					return; // Can't fit the data

				// Pad the data. No need to unlatch.
				DataMatrixPadder.addPadding(m_codeWords, m_symbolSize);

				m_valid = true;
			}
		} else
		// Rule from standard:
		// If two symbol characters remain and two C40 values
		// remain to be encoded (the first C40 value may be a shift or
		// data character but the second must represent a data character);
		// encode the two remaining C40 values followed by a pad
		// C40 value of 0 (shift 1) in the last two symbol characters.
		// A final Unlatch codeword again is not required.
		if ((remainSymbolCodeWords == 2) && (remainTextCodeWords == 2)) {
			int value16 = textCodeWords.get(c40Size) * 1600
					+ textCodeWords.get(c40Size + 1) * 40 + 0 + 1;
			m_codeWords.add((value16 / 256));
			m_codeWords.add((value16 % 256));

			m_symbolSize = tempSymbolSize;
			m_valid = true;
		} else
		// Use ASCII encoding for the remaining codewords.
		{
			// Encode the last characters as ASCII
			m_codeWords.add(254); // Unlatch
			for (idx = c40End; idx < messageSize; idx++) {
				int v = c40Encoding[idx].messageValue;
				if (v > 127) {
					v = v - 128;
					m_codeWords.add(235); // Upper shift
				}
				m_codeWords.add(v + 1);
			}

			// Find the symbol size that will fit the data.
			m_symbolSize = findSymbolSize(m_codeWords.size());
			if (m_symbolSize == -1)
				return; // Can't fit the data

			// Pad the data. No need to unlatch.
			DataMatrixPadder.addPadding(m_codeWords, m_symbolSize);

			m_valid = true;
		}
	}
}
