/*************************************************************************
 *
 *	File: DataMatrixCompactorManager.java
 *
 **************************************************************************
 * 
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2011 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.xfa.pmp.datamatrixpmp;

import java.util.List;

import com.adobe.xfa.pmp.common.IntegerHolder;

/**
 * Ported from DataMatrixCompactorManager.cpp
 */
class DataMatrixCompactorManager {
	// ////////////////////////////////////////////////////////////////////
	/**
	 * Do the byte compaction. This algorithm examines 12 bytes of the message
	 * at a time. If the 12 bytes can be compacted using text compaction, it
	 * uses text compaction, else byte compaction is used.
	 * 
	 * @param message
	 *            - The input binary message.
	 * @throws DataMatrixEncoderException
	 * @returns A vector of the of the binary encoded message in PDF417 code
	 *          words.
	 */
	// ////////////////////////////////////////////////////////////////////
	static void compact(char[] message, // In: Message to encode in barcode.
			List<Integer> codeWords, // Out: The barcode codewords.
			IntegerHolder symbolSize) throws DataMatrixEncoderException // Out:
																		// The
																		// symbol
																		// size
																		// to
																		// use.
	{
		DataMatrixAsciiCompactor asciiCompactor = new DataMatrixAsciiCompactor();
		asciiCompactor.compact(message);

		DataMatrixByteCompactor byteCompactor = new DataMatrixByteCompactor();
		byteCompactor.compact(message);

		DataMatrixC40Compactor c40Compactor = new DataMatrixC40Compactor();
		c40Compactor.compact(message);

		DataMatrixTextCompactor textCompactor = new DataMatrixTextCompactor();
		textCompactor.compact(message);

		DataMatrixX12Compactor x12Compactor = new DataMatrixX12Compactor();
		x12Compactor.compact(message);

		// Find a valid compactor to start the search with
		DataMatrixBaseCompactor minCompactor = null;
		if (byteCompactor.isValid())
			minCompactor = byteCompactor;
		else if (asciiCompactor.isValid())
			minCompactor = asciiCompactor;
		else if (c40Compactor.isValid())
			minCompactor = c40Compactor;
		else if (textCompactor.isValid())
			minCompactor = textCompactor;
		else if (x12Compactor.isValid())
			minCompactor = x12Compactor;

		if (minCompactor == null)
			throw new DataMatrixEncoderException(
					DataMatrixEncoderErrorCode.MESSAGE_TOO_BIG);

		int minSymbolSize = minCompactor.getSymbolSize();

		// Find the compactor with the best compaction
		if (textCompactor.isValid()) {
			if (textCompactor.getSymbolSize() < minSymbolSize) {
				minCompactor = textCompactor;
				minSymbolSize = minCompactor.getSymbolSize();
			}
		}

		if (c40Compactor.isValid()) {
			if (c40Compactor.getSymbolSize() < minSymbolSize) {
				minCompactor = c40Compactor;
				minSymbolSize = minCompactor.getSymbolSize();
			}
		}

		if (asciiCompactor.isValid()) {
			if (asciiCompactor.getSymbolSize() <= minSymbolSize) {
				minCompactor = asciiCompactor;
				minSymbolSize = minCompactor.getSymbolSize();
			}
		}

		if (x12Compactor.isValid()) {
			if (x12Compactor.getSymbolSize() < minSymbolSize) {
				minCompactor = x12Compactor;
				minSymbolSize = minCompactor.getSymbolSize();
			}
		}

		// Get the codewords
		minCompactor.getCodeWords(codeWords);
		symbolSize.setValue(minSymbolSize);
	}
}
