/*************************************************************************
 *
 *	File: DataMatrixEncoder.java
 *
 **************************************************************************
 * 
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2011 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.xfa.pmp.datamatrixpmp;

import java.awt.image.BufferedImage;
import java.util.ArrayList;
import java.util.List;

import com.adobe.xfa.pmp.common.BarcodeEncoder;
import com.adobe.xfa.pmp.common.BarcodeGenerationParams;
import com.adobe.xfa.pmp.common.IntegerHolder;

/**
 * Ported from DataMatrixEncoder.cpp
 */
public class DataMatrixEncoder implements BarcodeEncoder {
	// ////////////////////////////////////////////////////////////////////
	/**
	 * Encode the message into a DataMatrix symbol.
	 * 
	 * param message
	 *            - The byte message to be placed in the symbol.
	 * param parameters
	 *            - The encoding parameters.
	 * param numberOfParameters
	 *            - The number of encoding parameters.
	 * 
	 *            The encoding parameters are:
	 * 
	 *            Resolution Type: int32. Required: yes. Range: must be greater
	 *            than 0. Description: The DPI resolution. This is used along
	 *            with the XSymWidth and XSymHeight parameters to calculate the
	 *            size of the output image in pixels.
	 * 
	 *            Width Type: double. Required: yes. Range: must be greater than
	 *            0.0. Description: The width of the barcode image in inches.
	 * 
	 *            Height Type: double. Required: yes. Range: must be greater
	 *            than 0.0. Description: The height of the barcode image in
	 *            inches.
	 * 
	 *            XSymWidth Type: int. Required: no. Default: 4. Range: must be
	 *            greater than 0. Description: The size of a module in pixels.
	 */
	// ////////////////////////////////////////////////////////////////////
	public BufferedImage encode(char[] message, // In: The message to be encoded
												// in the barcode.
			BarcodeGenerationParams pmpParams)
			throws DataMatrixEncoderException {

		// Extract Resolution
		double resolution = pmpParams.getResolution();
		if (resolution < 1)
			throw new DataMatrixEncoderException(
					DataMatrixEncoderErrorCode.RESOLUTION);

		// Extract the image width
		int width = (int) (pmpParams.getWidth() * (resolution) + 0.5);

		// Extract the image height
		int height = (int) (pmpParams.getHeight() * (resolution) + 0.5);

		// Extract the module size
		int xSymWidth = (pmpParams.getXSymbolWidth());

		if (xSymWidth < 1)
			throw new DataMatrixEncoderException(
					DataMatrixEncoderErrorCode.XSYMWIDTH);

		// Compact the data
		List<Integer> codeWords = new ArrayList<Integer>();
		IntegerHolder symbolSize = new IntegerHolder(-1);
		DataMatrixCompactorManager.compact(message, codeWords, symbolSize);

		// Add ECC
		DataMatrixReedSolomon.addECC(codeWords, symbolSize.getValue());

		// Create the Matrix

		DataMatrixMatrix matrix = DataMatrixMatrixBuilder.buildMatrix(
				codeWords, symbolSize.getValue());

		// Create the image
		BufferedImage image = DataMatrixImageBuilder.buildImage(matrix, width,
				height, xSymWidth);

		return image;
	}

}
