/*************************************************************************
 *
 *	File: QRCodeEncoder.java
 *
 **************************************************************************
 * 
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2011 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.xfa.pmp.qrcodepmp;

import java.awt.image.BufferedImage;

import com.adobe.xfa.pmp.common.BarcodeEncoder;
import com.adobe.xfa.pmp.common.BarcodeGenerationParams;
import com.adobe.xfa.pmp.qrcodepmp.denso.QRC;

/**
 * Ported from QRCodeEncoder.cpp
 */
public class QRCodeEncoder implements BarcodeEncoder
{
	enum Level
	{

	         LevelH(3),
	         LevelQ(2),
	         LevelM(1),
	         LevelL(0);
	         Level(int level)
	         {
	        	this.level = level; 
	         }
	         int level;
	     
	}

//////////////////////////////////////////////////////////////////////
	/**
	    Encode the message into a QR Code symbol.

	    param message - The byte message to be placed in the symbol.
	    param parameters - The encoding parameters.
	    param numberOfParameters - The number of encoding parameters.

	    The encoding parameters are:

	    Resolution
	        Type: int32.
	        Required: yes.
	        Range: must be greater than 0.
	        Description:
	           The DPI resolution.  This is used along with the 
	           XSymWidth and XSymHeight parameters to calculate
	           the size of the output image in pixels.

	    ECC
	        Type: int32
	        Required: no.
	        Default: 4.
	        Range: 3 (H), 2 (Q), 1 (M), 0 (L)
	        Description:
	            The ECC level to use in the barcode.

	    Width
	        Type: double.
	        Required: yes.
	        Range: must be greater than 0.0.
	        Description:
	            The width of the barcode image in inches.

	    Height
	        Type: double.
	        Required: yes.
	        Range: must be greater than 0.0.
	        Description:
	            The height of the barcode image in inches.

	    XSymWidth
	        Type: int.
	        Required: no.
	        Default: 4.
	        Range: must be greater than 0.
	        Description:
	           The size of a module in pixels.

	*/
	//////////////////////////////////////////////////////////////////////
	public BufferedImage encode(char[] message, BarcodeGenerationParams pmpParams) throws QRCodeEncoderException
	{
	    // Extract Resolution
		double resolution = pmpParams.getResolution();
	    if(resolution < 1)
	        throw( new QRCodeEncoderException(QRCodeEncoderErrorCode.RESOLUTION));

	    // Extract the ECC
		// [todo] change default proto in QRCodeEncoderGenParams to setup ECC default value
		int ecc = Level.LevelM.level;;
		ecc = pmpParams.getEccLevel();
	    if(!((ecc == Level.LevelH.level) ||
	         (ecc == Level.LevelQ.level) ||
	         (ecc == Level.LevelM.level) ||
	         (ecc == Level.LevelL.level)))
			 throw(new QRCodeEncoderException(QRCodeEncoderErrorCode.ECC));

	    // Extract the image width
		int width = (int) (pmpParams.getWidth() * (resolution) + 0.5);

	    // Extract the image height
		int height = (int) (pmpParams.getHeight() * (resolution) + 0.5);

	    // Extract the module size
		int xSymWidth = pmpParams.getXSymbolWidth();;
		
	    if(xSymWidth < 1)
	        throw( new QRCodeEncoderException(QRCodeEncoderErrorCode.XSYMWIDTH));

	    QRC qrc = new QRC();
	    // Use model 2
		qrc.modelNo = QRC.MODEL2;

	    // Set the ECC level
	    qrc.Clvl = ecc;

	    // Automatically set the selection mask.
	    //qrc.Selmsk = -1;
	    
	    qrc.cdata = message;
	   
	    qrc.auto_conv(qrc.qdata, message, qrc.cdata, message.length);

	    // Create the symbol
	    if(qrc.qrcode(qrc.qdata, 0) == 0)
			throw( new QRCodeEncoderException(QRCodeEncoderErrorCode.MESSAGE_TOO_BIG));

	    // Build the image of the symbol
	    QRCodeImageBuilder imageBuilder = new QRCodeImageBuilder();
	    imageBuilder.setQrc(qrc);
		BufferedImage image = imageBuilder.buildImage(width, height, xSymWidth);
		
		return image;
	}

}
