/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2007 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */
package com.adobe.xfa.scripthandler.rhino;


import java.io.IOException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.mozilla.javascript.Context;
import org.mozilla.javascript.Function;
import org.mozilla.javascript.FunctionObject;
import org.mozilla.javascript.IdFunctionObject;
import org.mozilla.javascript.Scriptable;
import org.mozilla.javascript.ScriptableObject;
import org.mozilla.javascript.Undefined;

import com.adobe.xfa.AppModel;
import com.adobe.xfa.Arg;
import com.adobe.xfa.DependencyTracker;
import com.adobe.xfa.Node;
import com.adobe.xfa.Obj;
import com.adobe.xfa.SOMParser;
import com.adobe.xfa.ScriptDebugger;
import com.adobe.xfa.ScriptFuncObj;
import com.adobe.xfa.ut.ExFull;


/**
 * A class to implement naked references in JavaScript.
 * This class is a proxy for all naked references in XFA.
 * <p>
 * Instances of this class are registered as global objects
 * such that any naked reference, including JavaScript
 * variables, is first processed by the get method
 * of this class.
 * 
 * @author Mike Tardif
 */
public class LiveComponent extends ScriptableObject {

    static final long serialVersionUID = 1853010678358666589L;

    private static final String CLASSNAME = "LiveComponent";

    // JavaPort; apparently HashMap is still more efficient that a List
    private static final Map<String, Object> mStandardObjects = new HashMap<String, Object>() {
        private static final long serialVersionUID = 1L; {
    	put("Infinity", null);
    	put("NaN", null);
    	put("Math", null);
        put("Output", null);
        put("StopIteration", null);
        put("undefined", null);
    }};

    private static final Method mInvokeMethod;

    static {
        Method m = null;
        try {
            m = LiveComponent.class.getMethod(
                "invoke", 
                new Class[] { Context.class, Scriptable.class, Object[].class, Function.class });
        }
        catch (NoSuchMethodException ex) {
            assert false;    // not possible
        }
        mInvokeMethod = m;

    }

    private transient AppModel            moAppModel;
    private transient RhinoScriptHandler  moHandler;


    /**
     * Instantiates a LiveComponent object.
     * The lone constructor used by this package to create
     * instances of this class.
     *
     * @param handler a Rhino script handler.
     * @param xfaObject the XFA object associated with this component.
     * Typically this is the application model.
     */
    public LiveComponent(RhinoScriptHandler handler, Obj xfaObject) {
        jsConstructor(handler, xfaObject);
    }

    /**
     * Defines the JavaScript constructor to Rhino.
     * <p>
     * This method is required by the Scriptable interface
     * and must mirror the signature of the class constructor.
     *
     * @param handler a Rhino script handler.
     * @param object the XFA object associated with this component.
	 *
	 * @exclude from published api.
     */
    public void jsConstructor(Object handler, Obj object) {
        assert (handler instanceof RhinoScriptHandler);
        moHandler = (RhinoScriptHandler) handler;
        //
        // Tell the live component the value of "xfa".
        //
        assert (object instanceof AppModel);
        setXFAObjectValue((AppModel) object);
    }

    /**
     * Gets the name of the set of objects implemented by this class.
     * <p>
     * This method is required by the Scriptable interface.
	 *
	 * @exclude from published api.
     */
    public String getClassName() {
        return CLASSNAME;
    }

    /**
     * Gets the value of the named property of this object.
     * @param name the name of the property.
     * @param start the object in which the lookup began.
     * @return the value of the property, or NOT_FOUND
     * if the property is not xfa nor a naked reference to
     * an XFA property.
     * @throws org.mozilla.javascript.EvaluatorException
     * whenever an ExFull was thrown while getting the XFA
     * property.
	 *
	 * @exclude from published api.
     */
    public Object get(String name, Scriptable start) {
        if (name.equals(LiveObject.CLASSNAME))
            return Undefined.instance;
        else if (mStandardObjects.containsKey(name)) {
			// Avoid trying to get Starndard JS objects on the XFA side.
        }
        else if (name.equals("xfa"))
            return moHandler.getAppModelLiveObject();
        else if (moAppModel != null) {
            try {
                ScriptFuncObj scriptFuncObj = moAppModel.getContext().getScriptMethodInfo(name);
                if (scriptFuncObj != null) {
                    //    
                    // JavaPort: This test of the version approximates what is in XFALiveObject::getIDForName.
                    // If the method is not valid for this XFA version, then pretend that the method wasn't found.
                    //    
                    if (! moAppModel.getContext().validateUsageFailedIsFatal(scriptFuncObj.getXFAVersion(), scriptFuncObj.getAvailability())) {
                        return new FunctionObject(name, mInvokeMethod, this);
                    }
                }
                SOMParser oParser = new SOMParser(null);
                List<SOMParser.SomResultInfo> oResult = new ArrayList<SOMParser.SomResultInfo>();
                oParser.resolve((Node) moAppModel.getContext(), name, oResult);            
                if (oResult.size() == 1) {
                    return moHandler.argToVariant(oResult.get(0).value);
                }
            } catch (ExFull oEx) {
                moHandler.throwError(oEx);
            }
        }
        return super.get(name, start);
    }

    /**
     * Puts the value of the named property of the start object.
     * <p>
     * This method may (or not) set the property in the start object.
     * @param name the name of the property.
     * @param start the object whose property is being set.
     * @param value the value to set the property to.
	 *
	 * @exclude from published api.
     */
    public void put(String name, Scriptable start, Object value) {
        if (name.equals(LiveObject.CLASSNAME))
            return;
        else if (value instanceof IdFunctionObject) {
			// Avoid trying to set Starndard JS functions on the XFA side.
        }
        else if (mStandardObjects.containsKey(name)) {
			// Avoid trying to set Starndard JS objects on the XFA side.
        }
        else if (moAppModel != null) {
            try {
                moAppModel.getContext().setScriptProperty(name, moHandler.variantToArg(value), false);
                return;
            } catch (ExFull oEx) {
                // We can't assume the property belonged to XFA, so ignore,
				// except when it's a Rhino exception -- an ExFull whose
				// ResId is 0 -- in which case, we mustn't ignore.
                if (oEx.firstResId() == 0)
                    moHandler.throwError(oEx);
            }
        }
        super.put(name, start, value);
    }

    /**
     * @exclude from published api.
     */
    public static Object invoke(Context cx, Scriptable obj, Object[] args, Function func) {
        LiveComponent oThis = (LiveComponent) obj;
        try {
            DependencyTracker oDependencyTracker = oThis.moHandler.getAppModel().dependencyTracker();
            Arg oRetVal = new Arg();
            String sFunctionName = ((FunctionObject) func).getFunctionName();
            Arg[] oArgs = new Arg[args.length];
            for (int i = 0; i < args.length; i++) {
                oArgs[i] = oThis.moHandler.variantToArg(args[i]);
            }
            oThis.moAppModel.getContext().invokeFunction(oRetVal, sFunctionName, oArgs, oDependencyTracker, false);
            ScriptDebugger oDebugger = oThis.moHandler.getDebugger();
            if (oDebugger != null)
                oDebugger.resolvedValue(sFunctionName, oRetVal);
            return oThis.moHandler.argToVariant(oRetVal);
        } catch (ExFull oEx) {
            oThis.moHandler.throwError(oEx);
        }
        return Undefined.instance;
    }

    void setXFAObjectValue(AppModel oAppModel) {
        moAppModel = oAppModel;
    }

    /**
     * This class is not really serializable, so prevent attempts to serialize from succeeding.
     */
	private void writeObject(java.io.ObjectOutputStream out) throws IOException {
		throw new java.io.NotSerializableException(getClass().getSimpleName());
	}
}
