/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */

package com.adobe.xfa.template.containers;

import com.adobe.xfa.Arg;
import com.adobe.xfa.ArrayNodeList;
import com.adobe.xfa.Attribute;
import com.adobe.xfa.Element;
import com.adobe.xfa.Node;
import com.adobe.xfa.NodeList;
import com.adobe.xfa.StringAttr;
import com.adobe.xfa.ScriptTable;
import com.adobe.xfa.XFA;
import com.adobe.xfa.template.TemplateModel;
import com.adobe.xfa.ut.Peer;
import com.adobe.xfa.data.DataNode;

/**
 * ExclGroup elements are like 'transparent' abstract containers. They can
 * enclose most components, but have no notion of location
 *
 * @exclude from published api.
 */
public class ExclGroup extends Container {

	public ExclGroup(Element parent, Node prevSibling) {
		super(parent, prevSibling, null, XFA.EXCLGROUP, XFA.EXCLGROUP, null,
				XFA.EXCLGROUPTAG, XFA.EXCLGROUP);
	}

	public void execEvent(String sActivity) {
		// form node overwrites this function
	}

	public boolean execValidate() {
		// form node overwrites this function
		return true;
	}

	/**
	 * Get the access/accelerator key (single character)
	 * 
	 * @return single character value of the keyboard access key
	 */
	public String getAccessKey() {
		if (isValidAttr(XFA.ACCESSKEYTAG, false, null)
				&& isPropertySpecified(XFA.ACCESSKEYTAG, true, 0)) {
			Attribute oAttr = getAttribute(XFA.ACCESSKEYTAG, true, false);
			if (oAttr != null && !oAttr.isEmpty()) {
				String sAccessKey = oAttr.toString();
				return sAccessKey;
			}
		}
		return "";
	}

	public String getBackColor() {
		String sRet = "";
		Element pBorderNode = getElement(XFA.BORDERTAG, true, 0, true, false);
		if (pBorderNode != null) {
			Element pFillNode = pBorderNode.getElement(XFA.FILLTAG, true, 0,
					true, false);
			if (pFillNode != null) {
				Element pColorNode = pFillNode.getElement(XFA.COLORTAG, true,
						0, true, false);
				if (pColorNode != null)
					sRet = pColorNode.getAttribute(XFA.VALUETAG).toString();
			}
		}
		return sRet;
	}

	public String getBorderColor() {
		String sRet = "";
		Element pBorderNode = getElement(XFA.BORDERTAG, true, 0, true, false);
		if (pBorderNode != null) {
			Element pEdgeNode = pBorderNode.getElement(XFA.EDGETAG, true, 0,
					true, false);
			if (pEdgeNode != null) {
				Element pColorNode = pEdgeNode.getElement(XFA.COLORTAG, true,
						0, true, false);
				if (pColorNode != null)
					sRet = pColorNode.getAttribute(XFA.VALUETAG).toString();
			}
		}
		return sRet;
	}

	public String getBorderWidth() {
		String sRet = "";
		Element pBorderNode = getElement(XFA.BORDERTAG, true, 0, true, false);
		if (pBorderNode != null) {
			Element pEdgeNode = pBorderNode.getElement(XFA.EDGETAG, true, 0,
					true, false);
			if (pEdgeNode != null)
				sRet = pEdgeNode.getAttribute(XFA.THICKNESSTAG).toString();
		}
		return sRet;
	}

	public DataNode getDataNode() {
		// form node overwrites this function
		return null;
	}

	/**
	 * Get the formatted value of the member considered 'on'
	 * 
	 * @return the value of the member considered to be 'on'
	 */
	public String getFormattedValue() {
		Node pMember = getSelectedMember();
		if (pMember instanceof Field)
			return ((Field) pMember).getFormattedValue();

		return "";
	}

	public String getMandatory() {
		String sRet = "";
		Element pValidateNode = getElement(XFA.VALIDATETAG, true, 0, true,
				false);
		if (pValidateNode != null)
			sRet = pValidateNode.getAttribute(XFA.NULLTESTTAG).toString();

		return sRet;
	}

	public NodeList /* <Field> */ getMembers() {
		NodeList oList = new ArrayNodeList();
		// Only deal with radio buttons for now
		for (Node child = getFirstXFAChild(); child != null; child = child.getNextXFASibling()) {
			if (child instanceof Field) {
				oList.append(child);
			}
		}
		return oList;
	}

	public String getMessage(String aType) {
		String sRet = "";
		Element pValidateNode = getElement(XFA.VALIDATETAG, true, 0, false,
				false);
		if (pValidateNode != null)
			sRet = TemplateModel.getValidationMessage(pValidateNode, aType);

		return sRet;
	}

	/**
	 * Get the raw value of the member considered 'on'
	 * 
	 * @return the value of the member considered to be 'on'
	 */
	public String getRawValue() {
		Node pMember = getSelectedMember();
		if (pMember instanceof Field)
			return ((Field) pMember).getRawValue();

		return "";
	}

	/**
	 * Get the raw typed value of the member considered 'on'
	 * @param oValue Arg to retrieve the typed value
	 */
	public void getTypedRawValue(Arg oValue) {
		Node member = getSelectedMember();
		if ((member != null) && (member instanceof Field))
			((Field)member).getTypedRawValue(oValue);
		else
			oValue.setNull();
	}

	/**
	 * Get the member considered 'on'
	 * 
	 * @return the member considered to be 'on'
	 */
	public Node getSelectedMember() {
		NodeList oMembers = getMembers();

		int nLen = oMembers.length();
		for (int i = 0; i < nLen; i++) {
			Field oMember = (Field) oMembers.item(i);
			if (oMember.isOn())
				return oMember;
		}
		return null;
	}

	/**
	 * @exclude from published api.
	 */
	public ScriptTable getScriptTable() {
		return ExclGroupScript.getScriptTable();
	}
	
	/**
	 * @see Container#isConnectSupported()
	 * @exclude
	 */
	public boolean isConnectSupported() {
		return true;
	}

	public boolean isHeightGrowSupported() {
		return true;
	}

	public boolean isWidthGrowSupported() {
		return true;
	}

	public void setBackColor(String sString) {
		Element pBorderNode = getElement(XFA.BORDERTAG, 0);
		Element pFillNode = pBorderNode.getElement(XFA.FILLTAG, 0);
		Element pColorNode = pFillNode.getElement(XFA.COLORTAG, 0);

		pColorNode.setAttribute(new StringAttr(XFA.VALUE, sString), XFA.VALUETAG);
	}

	public void setBorderColor(String sString) {
		Element pBorderNode = getElement(XFA.BORDERTAG, 0);

		// need to set the 4 edges
		for (int i = 0; i < 4; i++) {
			if (!pBorderNode.isPropertySpecified(XFA.EDGETAG, true, i))
				break;

			Element pEdgeNode = pBorderNode.getElement(XFA.EDGETAG, false, i,
					false, false);

			// set the color
			Element pColorNode = pEdgeNode.getElement(XFA.COLORTAG, 0);
			pColorNode.setAttribute(new StringAttr(XFA.VALUE, sString), XFA.VALUETAG);
		}

		// need to set the 4 corners
		for (int i = 0; i < 4; i++) {
			if (!pBorderNode.isPropertySpecified(XFA.CORNERTAG, true, i))
				break;

			Element pCornerNode = pBorderNode.getElement(XFA.CORNERTAG, false,
					i, false, false);

			// set the color
			Element pColorNode = pCornerNode.getElement(XFA.COLORTAG, 0);
			pColorNode.setAttribute(new StringAttr(XFA.VALUE, sString), XFA.VALUETAG);
		}
	}

	public void setBorderWidth(String sString) {
		// TODO change back to getElement once prot changes are in;
		Element pBorderNode = getElement(XFA.BORDERTAG, 0);

		// need to set the 4 edges
		for (int i = 0; i < 4; i++) {
			if (!pBorderNode.isPropertySpecified(XFA.EDGETAG, true, i))
				break;

			Element pEdgeNode = pBorderNode.getElement(XFA.EDGETAG, false, i,
					false, false);

			// set the thickness
			pEdgeNode.setAttribute(new StringAttr(XFA.THICKNESS, sString), XFA.THICKNESSTAG);
		}

		// need to set the 4 corners
		for (int i = 0; i < 4; i++) {
			if (!pBorderNode.isPropertySpecified(XFA.CORNERTAG, true, i))
				break;

			Element pCornerNode = pBorderNode.getElement(XFA.CORNERTAG, false,
					i, false, false);

			// set the thickness
			pCornerNode.setAttribute(new StringAttr(XFA.THICKNESS, sString), XFA.THICKNESSTAG);
		}
	}

	public void setMandatory(String sString) {
		Element pValidateNode = getElement(XFA.VALIDATETAG, 0);
		pValidateNode.setAttribute(new StringAttr(XFA.NULLTEST, sString), XFA.NULLTESTTAG);
	}

	public void setMessage(String sMessage, String aType) {
		Element pValidateNode = getElement(XFA.VALIDATETAG, 0);
		TemplateModel.setValidationMessage(pValidateNode, sMessage, aType);
	}

	/**
	 * Toggle the a given field member 'on' and the rest 'off' based on the
	 * value
	 * 
	 * @param sValue -
	 *            the value of the member considered to be 'on'
	 */
	public void setRawValue(String sValue) {
		boolean bTurnedOn = false;
		String sOn;

		NodeList oMembers = getMembers();
		int nChildren = oMembers.length();
		for (int i = 0; i < nChildren; i++) {
			Field oField = (Field) oMembers.item(i);
			boolean bSetOn = false;
			// check if it has on an on value
			sOn = oField.getOnValue();
			if (sOn != null && sOn.equals(sValue)) {
				bTurnedOn = true;
				bSetOn = true;
			}
			oField.setOn(bSetOn);
		}

		// if none were turned on based on their on value, turn the first field that has no on value
		// on by setting its value
		if (!bTurnedOn) {
			for (int i = 0; i < nChildren; i++) {
				Field oField = (Field) oMembers.item(i);
				sOn = oField.getOnValue();
				if (sOn == null) {
					oField.setRawValue(sValue);
					break;
				}
			}
		}

		notifyPeers(Peer.VALUE_CHANGED, "", this);
	}

	public void setSelectedMember(String sSOMName) {
		// Given the name, resolve the field contained within this exclgroup
		// and toggle it 'on'. Toggle the rest of the fields 'off'.
		Node oOnNode = resolveNode(sSOMName); // do not change.

		NodeList oMembers = getMembers();

		int nLen = oMembers.length();
		for (int i = 0; i < nLen; i++) {
			Field oMember = (Field) oMembers.item(i);
			boolean bOn = (oOnNode == oMember);
			oMember.setOn(bOn);
		}
		notifyPeers(Peer.VALUE_CHANGED, "", this);
	}

}
