package com.adobe.xfa.text;

import com.adobe.xfa.font.FontInstance;
import com.adobe.xfa.font.FontItem;
import com.adobe.xfa.gfx.GFXDecorationInfo;
import com.adobe.xfa.gfx.GFXGlyphOrientation;
import com.adobe.xfa.ut.CoordPair;
import com.adobe.xfa.ut.FindBugsSuppress;
import com.adobe.xfa.ut.Rect;
import com.adobe.xfa.ut.Storage;
import com.adobe.xfa.ut.UnitSpan;


/**
 * @exclude from published api.
 */

class DispLineWrapped extends DispLine {
	static class CaretInfo extends PosnInfo {
		Rect mCaret;
	}

	static class CharRange {
		CharRange mpoNext;
		float moMinX;
		float moMaxX;

		CharRange () {
			mpoNext = null;
			moMinX = CHAR_RANGE_DEFAULT;
			moMaxX = CHAR_RANGE_DEFAULT;
		}

		CharRange (CharRange source) {	// don't copy next pointer
			moMinX = source.moMinX;
			moMaxX = source.moMaxX;
		}

		static boolean isInitialized (float oValue) {
			return oValue != CHAR_RANGE_DEFAULT;
		}
	}

	private static class Extra {
		@FindBugsSuppress(code="UrF")	// field is never read - incomplete implementation?
		TextContext mpoContext;
		TextPosn mpoEmptyPosn;
//		void mpoOptycaLog;
		int[] mpcFlatText;
		int mnFlatLength;
		float moPrevSpreadInsert;
		int mnRadixCharIndex;
		int mnLastDigitIndex;

		Extra () {
//			mpoContext = null;
//			mpoEmptyPosn = null;
//			mpoOptycaLog = null;
//			mpcFlatText = null;
//			mnFlatLength = 0;
			mnRadixCharIndex = Integer.MAX_VALUE;
			mnLastDigitIndex = Integer.MAX_VALUE;
		}

//		void SetOptycaLog (TextContext poContext, void poOptycaLog) {
//			if ((mpoOptycaLog != null) && (mpoContext != null)) {
//				mpoContext.getWRS().releaseOptycaLog (mpoOptycaLog);
//			}
//			AttachContext (poNewContext);
//			mpoOptycaLog = poNewOptycaLog;
//		}

		void attachContext (TextContext poNewContext) {
			mpoContext = poNewContext;
		}

		void createFlatText (DispLineWrapped poLine) {
			if (mpcFlatText != null) {
				return;
			}
			int i;
			int nPositions = poLine.getPositionCount();
			int nChars = 0;
			for (i = 0; i < nPositions; i++) {
				DispPosn oPosition = poLine.getPosition (i);
				nChars += DispLine.getPositionStreamCount (oPosition);
			}
			assert (nChars > 0);
			mpcFlatText = new int [nChars];
			mnFlatLength = nChars;

			int dest = 0;

			for (i = 0; i < nPositions; i++) {
				DispPosn oPosition = poLine.getPosition (i);
				switch (DispLine.getPositionType (oPosition)) {
					case DispLine.POSN_TYPE_RUN:
						int[] source = poLine.getCharArray();
						int index = oPosition.getMapIndex();
						int limit = index + oPosition.getMapLength();
						while (index < limit) {
							mpcFlatText[dest++] = source[index++];
						}
						break;
					case DispLine.POSN_TYPE_LIGATURE:	// stream count > 1
					case DispLine.POSN_TYPE_MULTIPLE:	// stream count == 1
							TextPosnBase oSourcePosn = new TextPosnBase (oPosition.pp());
							int nCopy = oPosition.getStreamCount();
							while (nCopy-- > 0) {
								mpcFlatText[dest++] = oSourcePosn.nextChar();
							}
							break;
					default:
						assert (false);
						break;
				}
			}

			assert (dest == mnFlatLength);
		}

		void clear () {
//			SetOptycaLog (null, null);
			mpoEmptyPosn = null;

			mpcFlatText = null;
			mnFlatLength = 0;

			moPrevSpreadInsert = 0;
			mnRadixCharIndex = Integer.MAX_VALUE;
			mnLastDigitIndex = Integer.MAX_VALUE;
		}
	}

//----------------------------------------------------------------------
//
//		Class WrappedSpan - Templated helper class to keep
//		track of an object associated with a span of characters in
//		the wrapped text.
//
//----------------------------------------------------------------------
	private static class WrappedSpan extends DispMapSpan {
		private final DispMap moParentMap;
		private int mnParentMapIndex;
		private int mnParentCharStart;
		private int mnParentCharMax;
		private final int mnParentCharLimit;

		WrappedSpan (DispLineWrapped poLine, DispMap oParentMap, int nParentCharStart, int nParentCharLength, DispMapItem poInitial) {
			super (poLine);
			moParentMap = oParentMap;
			mnParentCharLimit = nParentCharStart + nParentCharLength;
			setup (nParentCharStart, poInitial);
		}

		void update (int nParentCharIndex) {
			if (nParentCharIndex >= mnParentCharMax) {
				flush();
				setup (nParentCharIndex, null);
			}
		}

		void finish () {
			flush();
		}

		private void setup (int nParentCharStart, DispMapItem poInitial) {
			mnParentCharStart = nParentCharStart;
			mnParentMapIndex = moParentMap.findItem (mnParentCharStart);
			assert (moParentMap.isValidMapIndex (mnParentMapIndex));
			DispMapItem oParentItem = moParentMap.getItem (mnParentMapIndex);
			if (poInitial == null) {
				copyFrom (oParentItem);
			} else {
				copyFrom (poInitial);
			}
			mnParentCharMax = oParentItem.getMapIndex() + oParentItem.getMapLength();
			if (mnParentCharMax > mnParentCharLimit) {
				mnParentCharMax = mnParentCharLimit;
			}
		}
	}

//----------------------------------------------------------------------
//
//		HitTest - Internal class that performs horizontal
//		hit testing.
//
//----------------------------------------------------------------------
	private static class HitTest {
		private static class HitTestHit {
			final TextPosnBase moPosn = new TextPosnBase();
			float moBestDelta;
		}

		private final DispLineWrapped mpoLine;
		//private TextStream mpoStream;
		private final float moX;
		private final HitTestHit moResult = new HitTestHit();
		private final HitTestHit moTrailing = new HitTestHit();
		private final DispLineWrapped.CaretInfo mCaretInfo = new DispLineWrapped.CaretInfo();

		HitTest (DispLineWrapped poLine, TextStream poStream, float oX) {
			mpoLine = poLine;
			//mpoStream = poStream;
			moX = oX;
		}

		void tryHit (int nGlyphLocIndex, DispPosn oPosition) {
			GlyphLoc oGlyphLoc = mpoLine.getGlyphLoc (nGlyphLocIndex);

			for (int i = 0; i <= oGlyphLoc.getMapLength(); i++) {
				int nCharIndex = oGlyphLoc.getMapIndex() + i;
				DispLine.charToStreamInfo (oPosition, nCharIndex, mCaretInfo);
				for (int nStreamOffset = 0; nStreamOffset < mCaretInfo.auxInfo; nStreamOffset++) {
					tryHit (oGlyphLoc, oPosition, nCharIndex, nStreamOffset);
				}
			}
		}

		void tryHit (GlyphLoc oGlyphLoc, DispPosn oPosition, int nCharIndex, int nStreamOffset) {
			HitTestHit oHit = mpoLine.isValidPosition (nCharIndex) ? moResult : moTrailing;
			int nStreamIndex = DispLine.charToStreamIndex (oPosition, nCharIndex, nStreamOffset);
			TextPosnBase oPosn = new TextPosnBase (oPosition.pp().stream(), nStreamIndex);

			for (int nAffinity = 0; nAffinity < 2; nAffinity++) {
				oPosn.affinity ((nAffinity == 0) ? TextPosnBase.AFFINITY_BEFORE : TextPosnBase.AFFINITY_AFTER);

				int eCaret = mpoLine.getCaretRect (oPosn, true, mCaretInfo);
				if (eCaret != DispLine.CARET_INVALID) {
					float oDelta = Units.toFloat (mCaretInfo.mCaret.left());
					oDelta -= moX;
					if (oDelta < 0) {
						oDelta = -oDelta;
					}

					if ((oHit.moPosn.stream() == null) || (oDelta < oHit.moBestDelta)) {
						oHit.moPosn.associate (oPosition.pp().stream(), nStreamIndex);
						oHit.moPosn.affinity (oPosn.affinity());
						oHit.moPosn.setRTL (mpoLine.getGlyph (oGlyphLoc.getGlyphIndex()).isRTL());
						oHit.moBestDelta = oDelta;
					}
				}
			}
		}

		boolean hasHit () {
			return (moResult.moPosn.stream() != null) || (moTrailing.moPosn.stream() != null);
		}

		int reconcile (TextPosnBase oResult) {
			if (moResult.moPosn.stream() != null) {
				oResult.copyFrom (moResult.moPosn);
				return DispLineWrapped.CARET_PRESENT;
			}

			if (moTrailing.moPosn.stream() != null) {
				oResult.copyFrom (moTrailing.moPosn);
				return DispLineWrapped.CARET_CONDITIONAL;
			}

			return DispLineWrapped.CARET_INVALID;
		}
	}

	//----------------------------------------------------------------------
	//
	//		TextCombCell - Structure that keeps track of information
	//		for a single comb cell, as comb justification is
	//		processed.
	//
	//----------------------------------------------------------------------
	private static class CombCell {
		final int mnCombiningSize;	// number of chars (base + accents)
		int mnBaseGlyphIndex;	// index of base char's glyph
		float moX;				// position of base glyph
		float moNextX;			// position of next base glyph

		CombCell (int nCombiningSize) {
			mnCombiningSize = nCombiningSize;
		}
	}

	static float CHAR_RANGE_DEFAULT = 999999.0f;

	private final DispMapSet moMaps = new DispMapSet();
	private Extra mpoExtra;
	private LineDesc moLineDesc;
	private TextAttr mpoStartAttr;
	private DispLineWrapped mpoOldLine;

	private float moAMin;
	private float moAMax;
	private float moAMaxExtended;
	private int moJFAMin;
	private int moJFAMax;
	private int moJFAMaxExtended;
	private int moBMin;
	private int moBMinExtended;
	private int moBMaxExtended;
	private float moRawWidth;
	private float moRawWidthFull;
	private int moJFRawWidth;
	private LineHeight moHeight = new LineHeight();

	private Storage<CharRange> moCharRanges;
	private CoordPair moXYOrigin = CoordPair.zeroZero();

	private final static UnitSpan gHalfPoint = new UnitSpan (UnitSpan.POINTS_1K, 500);

	DispLineWrapped () {
		setMaps (moMaps);
	}

	void initialize (TextFrame poFrame, LineDesc oLineDesc) {
		super.initialize (poFrame);
		moLineDesc = new LineDesc (oLineDesc);
		mpoOldLine = null;
		if (moCharRanges != null)
			moCharRanges.clear();
		moHeight = new LineHeight (getLegacyLevel());
	}

	void setXYOrigin (CoordPair oNewOrigin) {
		moXYOrigin = oNewOrigin;
		CoordPair oABOrigin = ABXY.toAB (frame().getXYOrigin(), moXYOrigin, frame().getLayoutOrientation());
		moBMin = Units.toInt (oABOrigin.y());
	}

	CoordPair getXYOrigin () {
		return moXYOrigin;
	}

	UnitSpan getAMin () {
		return Units.toUnitSpan (moJFAMin);
	}

	float getAMinFloat () {
		return moAMin;
	}

	UnitSpan getAMax (boolean bExtended) {
		return Units.toUnitSpan (bExtended ? moJFAMaxExtended : moJFAMax);
	}

	UnitSpan getAMax () {
		return getAMax (false);
	}

	float getAMaxFloat (boolean bExtended) {
		return bExtended ? moAMaxExtended : moAMax;
	}

	UnitSpan getAExtent (boolean bExtended) {
		return Units.toUnitSpan ((bExtended ? moJFAMaxExtended : moJFAMax) - moJFAMin);
	}

	UnitSpan getAExtent () {
		return getAExtent (false);
	}

	UnitSpan getBMin () {
		return Units.toUnitSpan (moBMin);
	}

	UnitSpan getBMinExtended (boolean bRelative) {
		return Units.toUnitSpan (bRelative ? moBMinExtended : (moBMin + moBMinExtended));
	}

	UnitSpan getBMax () {
		return getBExtent().add (Units.toUnitSpan (moBMin));
	}

	UnitSpan getBMaxExtended (boolean bRelative) {
		return Units.toUnitSpan (bRelative ? moBMaxExtended : (moBMin + moBMaxExtended));
	}

	UnitSpan getBExtent () {
// If it's not the last line, or is compatibility mode, just return the
// line's full height.
		if ((! isLastLineInStream()) || (getLegacyLevel() == TextLegacy.LEVEL_V6)) {
			return Units.toUnitSpan (moHeight.fullHeight());
		}

// Correct spacing doesn't subtract the line gap on the last line if
// there is a spacing override in effect.
		if ((getLegacyLevel() == TextLegacy.LEVEL_CORRECT_SPACING) && (moHeight.override() > 0)) {
			return Units.toUnitSpan (moHeight.fullHeight());
		}

// Normal last line: do not include line gap.
		return Units.toUnitSpan (moHeight.fullHeight() - moHeight.lineGap());
	}

	UnitSpan getBaselineOffset (boolean bRelative) {
		int oResult = moHeight.textOffset (GFXGlyphOrientation.HORIZONTAL); // TODO:
		if (! bRelative) {
			oResult += moBMin;
		}
		return Units.toUnitSpan (oResult);
	}

	UnitSpan getRawWidth () {
		return Units.toUnitSpan (moJFRawWidth);
	}

	void justify (UnitSpan oMaxWidth) {
// Note: Justification can occur independently of layout.  Typically it
// occurs immediately after layout, but re-layout could be avoided in
// certain situations--mostly a future optimization, though this does
// occur to some extent today.	The main reason this hasn't been done
// more pervasively is because Arabic presents a case where layout and
// justification can happen together.

// First, determine the horizontal margins and absolute alignment.	Note
// that the notions of left and right are swapped for RTL text.
		int eJust = moLineDesc.meJustifyH;
		float oSpecialL = 0;
		float oSpecialR = 0;

		if (isRTL()) {
			oSpecialR = moLineDesc.moSpecial;
		} else {
			oSpecialL = moLineDesc.moSpecial;
		}

		if (eJust == TextAttr.JUST_H_RADIX) {
			eJust = TextAttr.JUST_H_RIGHT;
		}

// Trailing spaces may occur on the right (LTR text), on the left (RTL
// text) or in the middle of the line (BiDi text).	Depending on which
// situation is present, we may need to adjust the width of the line or
// its minimum X.
		boolean bTrailingLeft = false;
		if ((moLineDesc.mnTrailingSpaces > 0) && (getGlyphCount() > 0)) {
			GlyphLoc oGlyphLoc = getOrderedGlyphLoc (0);
			bTrailingLeft = ! isVisualChar (oGlyphLoc.getMapIndex());
			oGlyphLoc = getOrderedGlyphLoc (getGlyphCount() - 1);
			boolean bTrailingRight = ! isVisualChar (oGlyphLoc.getMapIndex());

// Special handling required only if trailing spaces on both sides or
// neither side (i.e., bTrailingLeft == bTrailingRight).
			if (bTrailingLeft == bTrailingRight) {
				if (bTrailingRight) {			// both sides (i.e., all spaces) ...
					bTrailingLeft = isRTL();	// ... treat as trailing left only if RTL text
				} else {						// neither side (trailing is in middle of line)
					setTrailingWidth (0);
					moRawWidth = moRawWidthFull;
					setEndInMiddle (true);
				}
			}
		}

// Determine the true minimum and maximum X values of the text in the
// line.  Also determine how much to spread out each blank if spread
// justification.  Note that this code still supports the JetForm 5
// notion of a text label: no horizontal extent to align in, instead the
// text is aligned at a single point.
		float oMaxUnit = Units.toFloat (oMaxWidth);
		float oShortfall = 0;
		float oSpreadInsert = 0;

		moAMin = 0;

// TBD: vertical orientation?
		if (frame().alignHPoint()) {					// align at a point
			switch (eJust) {
				case TextAttr.JUST_H_CENTRE:
					moAMin -= moRawWidth / 2;
					break;
				case TextAttr.JUST_H_RIGHT:
					moAMin -= moLineDesc.moMarginR + moRawWidth;
					break;
			}											// treat rest as left
		}

		else {											// align within a horizontal extent (XFA)
			moAMin += moLineDesc.moMarginL + oSpecialL;
			oMaxUnit -= moLineDesc.moMarginL + moLineDesc.moMarginR + oSpecialL + oSpecialR;

			if (oMaxUnit > moRawWidth) {				// round-off can cause -ve values
				oShortfall = oMaxUnit - moRawWidth;
			}

			boolean bSpread = false;

			switch (eJust) {
				case TextAttr.JUST_H_CENTRE:
					moAMin += oShortfall / 2;
					break;

				case TextAttr.JUST_H_RIGHT:
					moAMin += oShortfall;
					break;

				case TextAttr.JUST_H_SPREAD:
				case TextAttr.JUST_H_SPREAD_ALL:
					if (moLineDesc.mnInternalSpaces > 0) {
						if (moLineDesc.meJustifyH == TextAttr.JUST_H_SPREAD_ALL) {
							bSpread = true;
						} else if (legacyPositioning()) {
							if (getLastParaLine() == INTERNAL_LINE) {
								bSpread = true;
							}
						} else {
							if (getLastParaLine() <= HARD_NEW_LINE) {
								bSpread = true;
							}
						}
						if (bSpread) {
							oSpreadInsert = oShortfall / moLineDesc.mnInternalSpaces;
						}
					}
					if ((! bSpread) && isRTL()) {		// Last line of RTL spread para ...
						moAMin += oShortfall;			// ... treat as right-aligned
					}
					break;
			}
		}

		moAMax = moAMin + moRawWidth;
		moAMaxExtended = moAMin + moRawWidthFull;

// If there was trailing space on the left, adjust the minimum X to
// include it.
		if (bTrailingLeft) {
			moAMin -= getTrailingWidth();
		}

// If the line has spread justification, insert extra space at each
// blank.  Note that if this is a re-layout, we effectively remove the
// old space and insert the new space.
		if ((oSpreadInsert > 0)
		 || (mpoExtra != null) && ((mpoExtra.moPrevSpreadInsert > 0))) {
			needExtra();

			boolean bPendingSpreadOffset = false;
			float oDelta = oSpreadInsert - mpoExtra.moPrevSpreadInsert;
			float oOffset = 0;

			for (int i = 0; i < getGlyphCount(); i++) {
				Glyph oGlyph = getGlyph (i);
				GlyphExtra oGlyphExtra = forceExtra (i);
				oGlyphExtra.setOffsetX (oGlyphExtra.getOffsetX() + oOffset);

				if (bPendingSpreadOffset) {
					oGlyph.setSpreadOffset (true);
					bPendingSpreadOffset = false;
				}

				GlyphLoc oGlyphLoc = getOrderedGlyphLoc (i);
				int nCharIndex = oGlyphLoc.getMapIndex();
				if (isVisualChar (nCharIndex) && (oGlyphLoc.getMapLength() == 1)) {
					if (getBreakClass (nCharIndex) == TextCharProp.BREAK_SP) {
						oGlyph.setOriginalNextX (oGlyph.getOriginalNextX() + oDelta);
						oOffset += oDelta;
						bPendingSpreadOffset = true;		// next glyph starts new (draw) text run
					}
				}
			}

			mpoExtra.moPrevSpreadInsert = oSpreadInsert;
			moAMax += oShortfall;
			moAMaxExtended += oShortfall;
		}

// Radix justification: If we saw a radix character, align it to the
// right of the radix offset.  Otherwise, align the last digit to the
// left of the radix offset.  If we saw neither, treat as right-aligned/
		else if (moLineDesc.meJustifyH == TextAttr.JUST_H_RADIX) {
			needExtra();

			float oRadixOffset = 0;
			if ((mpoStartAttr != null) && (mpoStartAttr.radixOffsetEnable())) {
				oRadixOffset = Units.toFloat (mpoStartAttr.radixOffset().getLength());
			}
			float oDelta = Units.toFloat (oMaxWidth);
			oDelta -= oRadixOffset;

			if (mpoExtra.mnRadixCharIndex != Integer.MAX_VALUE) {
				GlyphLoc oGlyphLoc = getMappedGlyphLoc (mpoExtra.mnRadixCharIndex);
				Glyph oGlyph = getGlyph (oGlyphLoc.getGlyphIndex());
				oDelta -= oGlyph.getDrawX (this) + moAMin;
			} else if (mpoExtra.mnLastDigitIndex != Integer.MAX_VALUE) {
				GlyphLoc oGlyphLoc = getMappedGlyphLoc (mpoExtra.mnLastDigitIndex);
				Glyph oGlyph = getGlyph (oGlyphLoc.getGlyphIndex());
				oDelta -= oGlyph.getDrawNextX (this) + moAMin;
			}

			if (oDelta != 0.0) {
// TBD: radix alignment for point-justified labels?
				moAMin += oDelta;
				moAMax += oDelta;
				moAMaxExtended += oDelta;
			}
		}

// Comb justification: We want to distribute the characters into comb
// cells.  The sub type (left/centre/right) determines the starting cell.
// Then, each glyph is offset so that it is horizontally centered at the
// centre of its corresponding cell.  This code handles combining accent
// and base characters/glyphs into a single comb cell.
		else if (getCombCells() > 0) {
			int i;

// All comb glyphs require extra information
			preAllocExtra (getGlyphCount());

// Step 1: Run the glyph list to find sequences of glyphs that correspond
// to base plus optional accent character combinations.  The output of
// this step is the array oCellRuns, which initially contains only the
// number of glyphs per cell.
			CombCell[] oCellRuns = new CombCell [getGlyphCount()];
			int nFilledCells = 0;
			int nPrevGlyph = 0;
			int nGlyph = 0;

			while (nGlyph < getGlyphCount()) {
				if (nGlyph > nPrevGlyph) {
					oCellRuns[nFilledCells] = new CombCell (nGlyph - nPrevGlyph);
					nFilledCells++;
				}

				nPrevGlyph = nGlyph;

				int nGroup = getGlyph (nGlyph).getGroup (this);
				if (nGroup == 0) {
					nGlyph = nextGlyphBaseAccentIndex (nGlyph);
				} else {
					for (nGlyph++; nGlyph < getGlyphCount(); nGlyph++) {
						if (getGlyph(nGlyph).getGroup (this) != nGroup) {
							break;
						}
					}
					nGlyph = nextGlyphBaseAccentIndex (nGlyph - 1);
				}
			}

			if (nGlyph > nPrevGlyph) {
				oCellRuns[nFilledCells] = new CombCell (nGlyph - nPrevGlyph);
				nFilledCells++;
			}

// Step 2a: Depending on the particular type of comb justification,
// determine how many empty cells there are on the left of the comb.
			int nMaxCells = getCombCells();
			int nSkip = 0; // number of cells to skip for l/c/r alignment
			if (nMaxCells > nFilledCells) {
				nSkip = nMaxCells - nFilledCells;
			}

			switch (eJust) {
				case TextAttr.JUST_H_LEFT:
				case TextAttr.JUST_H_COMB_LEFT:
					nSkip = 0;
					break;
				case TextAttr.JUST_H_CENTRE:
				case TextAttr.JUST_H_COMB_CENTRE:
					nSkip /= 2;
					break;
			}

			float oCellWidth = oMaxUnit / nMaxCells;
			float oHalfWidth = oCellWidth / 2;

// Step 2b: This step completes the cell array by determining the glyph
// offsets for each cell.  Each iteration determines the cells X offset
// and the "next X offset" of the previous cell.
			int nGlyphIndex = 0;
			for (i = 0; i < nFilledCells; i++) {
				int nBaseIndex = nGlyphIndex;
				Glyph poGlyph = getGlyph (nGlyphIndex);
				if (poGlyph.isRTL()) {
					nBaseIndex = nGlyphIndex + oCellRuns[i].mnCombiningSize - 1;
				}
				oCellRuns[i].mnBaseGlyphIndex = nBaseIndex;

				int nCell = nSkip + i;
				float oCentreX = (oCellWidth * nCell) + oHalfWidth;
				DispRect oBBox = getABGlyphBBox (nBaseIndex);
				float oGlyphWidth = oBBox.xMax - oBBox.xMin;
				float oX = oCentreX - (oGlyphWidth / 2);
				if (i == 0) {
					moAMin = oX;
				}
				oX -= moAMin; // all positions relative to line's min X

				oCellRuns[i].moX = oX;

				if (i > 0) {
					oCellRuns[i-1].moNextX = oX;
				}

				if (i+1 == nFilledCells) { // last one ... compute its next X
					float oNextX = oCellWidth * (nCell + 1) + oHalfWidth;
					if (oNextX > oMaxUnit) {
						oNextX = oMaxUnit;
					}
					oNextX -= moAMin;
					oCellRuns[i].moNextX = oNextX;
				}

				nGlyphIndex += oCellRuns[i].mnCombiningSize;
			}

// Step 3: Run the cell array and adjust the line's glyph objects to
// account for comb spacing.
			nGlyphIndex = 0;
			for (i = 0; i < nFilledCells; i++) {
				CombCell oCell = oCellRuns[i];
				Glyph oBaseGlyph = getGlyph (oCell.mnBaseGlyphIndex);
				float oDelta = oCell.moX - oBaseGlyph.getDrawX (this);

				for (int j = 0; j < oCell.mnCombiningSize; j++) {
					forceExtra (nGlyphIndex);
					getGlyph(nGlyphIndex).setComb (this, oDelta, oCell.moNextX);
					nGlyphIndex++;
				}
			}

			if (nFilledCells == 0) {
				if (eJust == TextAttr.JUST_H_COMB_RIGHT) {
					nSkip--;
				}
				moAMin = (oCellWidth * nSkip) + oHalfWidth;
			}

			moAMax = oMaxUnit;
			moAMaxExtended = oMaxUnit;
		}

		moJFAMin = Units.toInt (moAMin);
		moJFAMax = Units.forceInt (moAMax);
		moJFAMaxExtended = Units.toInt (moAMaxExtended);
	}

	void updateSubsettedChars () {
		if (getGlyphCount() == 0) {
			return;
		}

// If we have bypassed Optyca, we can make some simplifying assumptions
// for a more performant algorithm.  In particular, there is a 1:1
// mapping between characters and glyphs that preserves order.	Also any
// non-Unicode font will be rendered by character.	We can process the
// run map sequentially and then the characters in each run (more
// efficient than the lookups involved if we start from glyphs).
//		if (hasBIDI()) {
//			for (int nRunIndex = 0; nRunIndex < getRunCount(); nRunIndex++) {
//				DispRun oRun = getRun (nRunIndex);
//
//				TextAttr poAttr = oRun.getAttr();
//				if (poAttr == null) {
//					continue; // no attribute present: skip this run
//				}
//
//				FontInstance oFontInstance = poAttr.fontInstance();
//				if (oFontInstance == null) {
//					continue; // no font instance available: skip this run
//				}
//
//				FontItem poFontItem = oFontInstance.getFontItem();
//				if (poFontItem == null) {
//					continue; // invalid font item: skip this run
//				}
//
//				int nRunStart = oRun.getMapIndex();
//				if ((oRun.getMapLength() == 1) && (tabAt (nRunStart) != null)) {
//					RenderGlyph oRenderGlyph = new RenderGlyph();
//					getRenderChar (poAttr, oFontInstance, getGlyphLoc (nRunStart), getGlyph (nRunStart), oRenderGlyph, poFontItem);
//					poFontItem.addSubsettedGlyphs (oRenderGlyph.mnGlyphID, oRenderGlyph.mcGlyph);	// TODO: poFontItem is a return pointer in C++
//				} else {
//					boolean bIsUnicodeFont = isUnicodeFont (poAttr);
//					int nRunLimit = nRunStart + oRun.getMapLength();
//					for (int nCharIndex = nRunStart; nCharIndex < nRunLimit; nCharIndex++) {
//						poFontItem.addSubsettedGlyphs (getGlyph(nCharIndex).getGlyph(), bIsUnicodeFont ? '\0' : getChar (nCharIndex));
//					}
//				}
//			}
//		}

// Did not bypass Optyca: The order may have changed.  In addition,
// Optyca may have substituted glyphs for contextual forms, ligatures,
// and so on.  Therefore, we must run the glyph map and pass only glyph
// IDs to the subsetting method.
//		else {
			for (int i = 0; i < getGlyphCount(); i++) {
				GlyphLoc oGlyphLoc = getGlyphLoc (i);

				TextAttr poAttr = getMappedAttr (oGlyphLoc.getMapIndex());
				if (poAttr == null) {
					continue; // no attribute present: skip this run
				}

				FontInstance oFontInstance = poAttr.fontInstance();
				if (oFontInstance == null) {
					continue; // no font instance available: skip this run
				}

				FontItem poFontItem = oFontInstance.getFontItem();
				if (poFontItem == null) {
					continue; // invalid font item: skip this run
				}

				RenderGlyph oRenderGlyph = new RenderGlyph();
				getRenderChar (poAttr, oFontInstance, oGlyphLoc, getGlyph (oGlyphLoc.getGlyphIndex()), oRenderGlyph);
				poFontItem.addSubsettedGlyphs (oRenderGlyph.mnGlyphID, oRenderGlyph.mcGlyph);	// TODO: poFontItem is a return pointer in C++
			}
//		}
	}

// Watson 1260903:	Work-around for printer-based font limitations.  Moved
// previous implementation to a private method.  If we're using a printer font (XDC Font)
// then glyph IDs have no meaning.	Printer fonts can only be used with character
// data.  If there is no character that can be gleaned for the glyph, override the
// given font item with a non-printer font of the same typeface that will allow glyph usage.
// Force the Unicode encoding of the FontItem in order to allow the widest selection of glyphs.
	// FIXME: oFontInstance, oGlyphLoc, oGlyph, oRenderGlyph passed by reference in C++
	// JavaPort: Signature changed to return poFontItem instead of passing it by value
	FontItem getRenderChar (TextAttr poAttr, FontInstance oFontInstance, GlyphLoc oGlyphLoc, Glyph oGlyph, RenderGlyph oRenderGlyph, FontItem poFontItem) {
		getRenderChar (poAttr, oFontInstance, oGlyphLoc, oGlyph, oRenderGlyph);
		if (oRenderGlyph.mcGlyph == '\0') {
			FontItem poUnicodeFont = poFontItem.getUnicodeFont();
			if (poUnicodeFont != null) {
				poFontItem = poUnicodeFont;	// TODO: need to return this (put in RenderGlyph?)
			}
		}
		
		return poFontItem;
	}

	TextPosn getStartPosition () {
		if (getPositionCount() > 0) {
			return getPosition(0).pp();
		} else if ((mpoExtra != null) && (mpoExtra.mpoEmptyPosn != null)) {
			return (mpoExtra.mpoEmptyPosn);
		}

		return null;
	}

	TextPosnBase getEndPosition () {
		TextPosnBase oResult = null;

		if (getPositionCount() > 0) {
			DispPosn oMappedPosn = getPosition (getPositionCount() - 1);
			oResult = new TextPosnBase (oMappedPosn.pp().stream(), oMappedPosn.pp().index() + oMappedPosn.getMapLength());
		} else if ((mpoExtra != null) && (mpoExtra.mpoEmptyPosn != null)) {
			oResult = new TextPosnBase (mpoExtra.mpoEmptyPosn);
		}

		return oResult;
	}

	TextAttr getStartAttr () {
		return mpoStartAttr;
	}

	int getHorizontalJustification () {
		return moLineDesc.meJustifyH;
	}

	DispRect getXYGlyphBBox (int nGlyphIndex, boolean bBaselineShift) {
		return ABXY.toXY (moXYOrigin, getABGlyphBBox (nGlyphIndex, bBaselineShift), frame().getLayoutOrientation());
	}

	DispRect getABGlyphBBox (int nGlyphIndex, boolean bBaselineShift) {
		DispRect oResult = new DispRect();

		int nGlyphLocIndex = getGlyphLocOrder (nGlyphIndex);
		GlyphLoc oGlyphLoc = getGlyphLoc (nGlyphLocIndex);

		int nEmbedIndex = findObject (oGlyphLoc);
		if (nEmbedIndex >= 0) {	// TODO: nEmbedIndex
			DispEmbed oMapEmbed = getEmbed (nEmbedIndex);
			TextEmbed poEmbed = oMapEmbed.getEmbed();
			UnitSpan oHeight = poEmbed.height();

			oResult.xMax = Units.toFloat (poEmbed.width());

			switch (poEmbed.embedAt()) {
				case TextEmbed.EMBED_AT_TOP:
					oResult.yMin = Units.toFloat (moHeight.before() - moHeight.textOffset (GFXGlyphOrientation.HORIZONTAL));
					oResult.yMax = oResult.yMin + Units.toFloat (oHeight);
					break;
				case TextEmbed.EMBED_AT_BOTTOM:
					oResult.yMax = Units.toFloat (moHeight.descent());
					oResult.yMin = oResult.yMax - Units.toFloat (oHeight);
					break;
				default:
					oResult.yMin = -Units.toFloat (oHeight);
			}
		}

		else {
			Glyph oGlyph = getGlyph (nGlyphIndex);
			int eGlyphOrientation = oGlyph.getOrientation();
			//boolean bHorizontal = GFXGlyphOrientation.usesHorizontalGlyphs (eGlyphOrientation);

			int nCharIndex = oGlyphLoc.getMapIndex();
			DispRun oRun = getMappedRun (nCharIndex);
			TextAttr poAttr = oRun.getAttr();
			float oShift = 0;
			boolean bFound = false;

			if (poAttr != null) {								// TODO:
//				FontInstance oJFInst = poAttr.fontInstance();
//
//				if (! oJFInst.isNull()) {
//					FontBBox oFontBBox;
//					if (oJFInst.getGlyphBBox (oGlyph.getGlyph(), oFontBBox, bHorizontal)) {
//						oResult.moLeft = oFontBBox.mnMinX;
//						oResult.yMax = oFontBBox.mnMinY;
//						oResult.xMax = oFontBBox.mnMaxX;
//						oResult.yMax = oFontBBox.mnMaxY;
//						bFound = true;
//					}
//				}
//
//				if (! bFound) {
//					TextDispFontWrapper poFont = Display().getFontMap().lookupFont (oRun.fontID());
//					if ((poFont != null) && poAttr.sizeEnable()) {
//						try {
//							CTFontDict poDict = poFont.getFontDict();
//							UnitSpan oSourceSize (poAttr.size());
//							UnitSpan oPTSize (UnitSpan.POINTS_1K, oSourceSize.units(), oSourceSize.value());
//							BRVRealCoord dSize = oPTSize.value() / (BRVRealCoord) 1000.0;
//							BRVCoordMatrix oMtx = {dSize, 0, 0, dSize, 0, 0};
//							CCTFontInstance oGeneratedInst (poDict, oMtx, bHorizontal ? kCTLeftToRight : kCTTopToBottom);
//
//							BRVRealCoordRect oRect;
//							oGeneratedInst.getBBox (oGlyph.getGlyph(), oRect);
//							oResult.moLeft = oRect.xMin;
//							oResult.yMax = oRect.yMin;
//							oResult.xMax = oRect.xMax;
//							oResult.yMax = oRect.yMax;
//
//							bFound = true;
//						}
//						catch (9) {
//						}
//						catch (13) {
//// Although we do the same thing, we keep the catch
//// blocks for different exceptions in case we need to
//// do something different in the future.
//						}
//					}
//				}

				if (bBaselineShift) {
					if (poAttr.baselineShiftEnable() && (! poAttr.baselineShift().isNeutral())) {
						oShift = Units.toFloat (poAttr.baselineShift().applyShift (UnitSpan.ZERO, Units.toUnitSpan (moHeight.textOffset (GFXGlyphOrientation.HORIZONTAL))));
					}
				}
			}

			if (! bFound) {
				oResult.xMin = 0;
				oResult.xMax = oGlyph.getDrawNextX (this) - oGlyph.getDrawX (this);
				oResult.yMin = Units.toFloat (moHeight.before() - moHeight.textOffset (eGlyphOrientation));
				oResult.yMax = Units.toFloat (moHeight.descent());
			}

			oResult.yMin += oShift;
			oResult.yMax += oShift;
		}

		return oResult;
	}

	DispRect getABGlyphBBox (int nGlyphIndex) {
		return getABGlyphBBox (nGlyphIndex, false);
	}


	CharRange getCharExtent (int nPositionIndex, int nCharIndex, int nStreamOffset) {
		getCharOffset (nPositionIndex, nCharIndex, nStreamOffset, TextPosnBase.AFFINITY_AFTER);

		DispPosn oPosition = getPosition (nPositionIndex);
		if ((nStreamOffset + 1) < getCharStreamCount (oPosition)) {
			getCharOffset (nPositionIndex, nCharIndex, nStreamOffset + 1, TextPosnBase.AFFINITY_BEFORE);
		} else {
			if ((nCharIndex + 1) >= (oPosition.getMapIndex() + oPosition.getMapLength())) {
				nPositionIndex++;
			}
			getCharOffset (nPositionIndex, nCharIndex + 1, 0, TextPosnBase.AFFINITY_BEFORE);
		}

		CharRange poRange = getCharRange (nCharIndex);
		for (; ; ) {
			assert (poRange != null);
			if (nStreamOffset == 0) {
				CharRange oResult = new CharRange (poRange);	// TODO: is the copy necessary?
				oResult.mpoNext = null;							// just to keep these hidden
				return oResult;
			}
			poRange = poRange.mpoNext;
			nStreamOffset--;
		}
	}

	void preAllocOffsets () {
		if (moCharRanges == null) {
			moCharRanges = new Storage<CharRange>();
		}
		int size = getCharCount();
		moCharRanges.setSize (size);
		for (int i = 0; i < size; i++) {
			moCharRanges.set (i, new CharRange());
		}
	}

	boolean gfxDraw (DrawParm oParm) {
		if (getGlyphCount() == 0) {
			return true;
		}

		if ((! isFirstLineInStream()) && isFirstParaLine()) {
			oParm.driver().paraHint();
		}

		oParm.setDispHeight (moHeight);

//		Glyph oFirstGlyph = getGlyph (0);

		DrawAttr oCurrentAttr = new DrawAttr (this, oParm);
		oCurrentAttr.drawBackground();

		DrawRun oRun = new DrawRun (this, oParm);

		boolean bTruncate = false;
		UnitSpan oTruncAMin = null;
		UnitSpan oTruncAMax = null;
		boolean bFits = true;

		if (oParm.truncate() != null) {
			oTruncAMin = Units.toUnitSpan (oParm.truncateAMin() - moAMin);
			oTruncAMax = Units.toUnitSpan (oParm.truncateAMax() - moAMin);
			bTruncate = true;
		}

		RenderInfo oRenderInfo = new RenderInfo();
		oRenderInfo.mbAttrChange = true;
//		boolean bWordFlushing = false;

		int[] pcText = null;
		if (hasAXTEMappings()) {
			needExtra();
			mpoExtra.createFlatText (this);
			pcText = mpoExtra.mpcFlatText;
		} else {
			pcText = getCharArray();
		}

		boolean popRenderContext = false;
		if (oParm.charIndex() < 0) {
			oParm.driver().setUnicodeChars (pcText);
		} else {
			oParm.driver().pushRenderContext (pcText, oParm.charIndex());
			popRenderContext = true;
		}

		try {
			while (oCurrentAttr.prepareGlyph (oRenderInfo))
			{
				int nGlyphIndex = oRenderInfo.mpoGlyphLoc.getGlyphIndex();
				Glyph oGlyph = getGlyph (nGlyphIndex);
				float oX = oGlyph.getDrawX (this) + moAMin;
				boolean bSuppressGlyph = false;

// Note: if the glyph starts to the left of the truncation rectangle,
// it's out.	However, if it ends beyond the right, we allow a 1/2
// threshold on the last visible character.  This is for compatibility
// with 6.0 that used rounded font metrics.  See displineraw.cpp for the
// word wrap algrithm inplementation of this threshold.  Watson 1241063:
// Simplified the right side test which was trying to handle only the
// right-most glyph that was a non-trailing space.	This didn't work for
// RTL text; even a round-off error of 1/1000pt could cause an RTL glyph
// to be truncated completely.
				if (bTruncate) {
					if (Units.toUnitSpan (oGlyph.getDrawX (this)).lt (oTruncAMin)) {	// TODO: avoid conversion each time ?
						bSuppressGlyph = true;
					} else {
						UnitSpan oDelta = Units.toUnitSpan(oGlyph.getDrawNextX(this)).subtract (oTruncAMax);
						if (oDelta.value() > 0) {
							if (oDelta.gt (gHalfPoint)) {
								bSuppressGlyph = true;
							}
						}
					}
					if (bSuppressGlyph) {
						if ((oRenderInfo.mpoGlyphLoc.getMapLength() > 1) || isVisualChar (oRenderInfo.mpoGlyphLoc.getMapIndex())) {
							bFits = false;
						}
					}
				}

				DispRect oBBox = getABGlyphBBox (nGlyphIndex);
				float oAMin = oBBox.xMin + oX;
				float oAMax = oBBox.xMax + oX;

				if ((oAMin <= oParm.invalidAMax()) && (oAMax >= oParm.invalidAMin()) && (! bSuppressGlyph)) {
					float oY = oGlyph.getDrawY (this) + Units.toFloat (oCurrentAttr.getBaseline());	// TODO: avoid repeated conversions?
					DispEmbed poDispEmbed = isObject (oRenderInfo.mpoGlyphLoc);
					boolean bRenderGlyph = false;
					boolean bRenderObject = false;

					if (poDispEmbed == null) {
						bRenderGlyph = true;
					} else {
						DispTab poDispTab = tabAt (oRenderInfo.mpoGlyphLoc.getMapIndex());
						if ((poDispTab == null) || legacyPositioning()) {
							bRenderObject = true;
						} else {
							bRenderGlyph = renderTabLeader (oParm, poDispTab, oRenderInfo.mpoGlyphLoc, oCurrentAttr, oRun);
						}
					}
					if (bRenderObject) {
						oRun.flush();
//						bWordFlushing = false;

						TextEmbed poEmbed = poDispEmbed.getEmbed();
						UnitSpan oTop = UnitSpan.ZERO;
						UnitSpan oHeight = poEmbed.height();

						switch (poEmbed.embedAt()) {
							case TextEmbed.EMBED_AT_BASELINE:
								oTop = Units.toUnitSpan (oY);
								oTop = oTop.subtract (oHeight);
								break;
							case TextEmbed.EMBED_AT_TOP:
								oTop = Units.toUnitSpan (moHeight.before());
								break;
							case TextEmbed.EMBED_AT_BOTTOM:
								oTop = Units.toUnitSpan (moHeight.before() + moHeight.spacing());
								oTop = oTop.subtract (oHeight);
								break;
						}

// TBD: vertical text
						CoordPair oOffset = new CoordPair (Units.toUnitSpan (oX), oTop);
						oParm.driver().pushOffset (false, oOffset);
						try {
							poEmbed.gfxDraw (oParm.env()); // TBD: clipped drawing methods
						} finally {
							oParm.driver().popOffset();
						}

						oRenderInfo.mbAttrChange = true;
						oRun.forceAttrChange();
					}

					if (bRenderGlyph) {
						if (! oRenderInfo.mbFlushBefore) {
							if (/* (oParm.opt() == TextPrefOpt.OPT_CHAR) || */ (optycaJustify()) || (oGlyph.isSpreadOffset()) || (oGlyph.isComb()) || (poDispEmbed != null)) {
								oRenderInfo.mbFlushBefore = true;
							}
						}

						if (! oRenderInfo.mbFlushBefore) {
//						if (oParm.opt() == TextPrefOpt.OPT_WORD) {
//							if (getBreakClass (oRenderInfo.mpoGlyphLoc.getMapIndex()) == TextCharProp.BREAK_SP) {
//								bWordFlushing = true; // spaces go with preceding word
//							}
//							else if (bWordFlushing) { // word char and was in spaces after word ...
//								oRenderInfo.mbFlushBefore = true; // ... flush previous word (and following spaces)
//								bWordFlushing = false;
//							}
//						}
						}

						if (oRenderInfo.mbFlushBefore) {
							oRun.flush();
						}

						oRun.addGlyph (oCurrentAttr, oRenderInfo, oX, oY, oGlyph.isRTL());

						if (oRenderInfo.mbFlushAfter) {
							oRun.flush();
						}
					}
				}
			}

			oRun.flush();
			oRun.clear();			// clears any clip rectangle
			oCurrentAttr.flush();	// puts out any decoration lines (after clear clip rect)
		} finally {
			if (popRenderContext) {
				oParm.driver().popRenderContext();
			}
		}

		return bFits;
	}

	DispLineWrapped getOldLine () {
		return mpoOldLine;
	}

	void setOldLine (DispLineWrapped poOld) {
		mpoOldLine = poOld;
	}

	Rect diff (DispLineWrapped poCompare, Rect oABDiff) {
		Rect oThisDiff = null;

// TBD: may want to replace calls to GetLineExtent() below with access to
// cached line extent information for performance.
		if (poCompare == null) {
			oThisDiff = getABLineExtent();
		}

// Note: Used to compare GetBMax() instead of FullHeight() values.
// However, with the change in handling last line line-spacing, the
// GetBMax() comparison always failed on the last line, because the old
// last line had already been marked as no longer being the last line.
// FullHeight() is not influenced by the last line status.
		else if ((getXYOrigin() != poCompare.getXYOrigin()) || (moHeight.fullHeight() != poCompare.moHeight.fullHeight())) {
			oThisDiff = getABLineExtent().union (poCompare.getABLineExtent());
		}

		else {
			int nCompareGlyphs = poCompare.getGlyphCount();
			int nGlyphMax = getGlyphCount();
			if (nGlyphMax > nCompareGlyphs) {
				nGlyphMax = nCompareGlyphs;
			}

			int i;
			for (i = 0; i < nGlyphMax; i++) {
				if (diffGlyphs (poCompare, i, i)) {
					break;
				}
			}

			if ((i < getGlyphCount()) || (i < nCompareGlyphs)) {
				int nDiffStart = i;
				DispRect oDiffRect = new DispRect();

				if (nDiffStart >= nCompareGlyphs) {
					oDiffRect = getABSubExtent (nDiffStart, getGlyphCount() - 1);
				}

				else if (nDiffStart >= getGlyphCount()) {
					oDiffRect = poCompare.getABSubExtent (nDiffStart, nCompareGlyphs - 1);
				}

				else {
					int j = nCompareGlyphs;
					i = getGlyphCount();

					while ((i > nDiffStart) && (j > nDiffStart)) {
						i--;
						j--;
						if (diffGlyphs (poCompare, i, j)) {
							break;
						}
					}
					oDiffRect = getABSubExtent (nDiffStart, i);
					DispRect oDiffCompare = poCompare.getABSubExtent (nDiffStart, j);

					if (oDiffCompare.xMin < oDiffRect.xMin) {
						oDiffRect.xMin = oDiffCompare.xMin;
					}
					if (oDiffCompare.yMax < oDiffRect.yMax) {
						oDiffRect.yMax = oDiffCompare.yMax;
					}
					if (oDiffCompare.xMax > oDiffRect.xMax) {
						oDiffRect.xMax = oDiffCompare.xMax;
					}
					if (oDiffCompare.yMax > oDiffRect.yMax) {
						oDiffRect.yMax = oDiffCompare.yMax;
					}
				}

				oThisDiff = new Rect (Units.toUnitSpan (oDiffRect.xMin),
									  Units.forceUnitSpan (oDiffRect.xMax),
									  Units.toUnitSpan (oDiffRect.yMax),
									  Units.forceUnitSpan (oDiffRect.yMax));
			}
		}

		if (oThisDiff == null) {
			return oABDiff;
		}

		oThisDiff = oThisDiff.add (new CoordPair (UnitSpan.ZERO, Units.toUnitSpan (moBMin)));
		return (oABDiff == null) ? oThisDiff : oABDiff.union (oThisDiff);
	}

	int getCaretRect (TextPosnBase oPosn, boolean bAB, CaretInfo caretInfo) {
		int eCaret = findPosition (oPosn, caretInfo);

		if (eCaret != CARET_PRESENT) {
			if (! isEmptyPosition (oPosn)) {
				return eCaret;
			}
			eCaret = CARET_PRESENT;
		}

		float oA = getCharOffset (caretInfo.posnIndex, caretInfo.index, caretInfo.auxInfo, oPosn.affinity());
		UnitSpan oAUnit = Units.toUnitSpan (oA);

		caretInfo.mCaret = new Rect (oAUnit, UnitSpan.ZERO, oAUnit, getBExtent());

		if (! bAB) {
			caretInfo.mCaret = ABXY.toXY (moXYOrigin, caretInfo.mCaret, frame().getLayoutOrientation());
		}

		return eCaret;
	}

	int validateCaretPosn (TextPosnBase oPosn, boolean bAB) {
		PosnInfo findInfo = new PosnInfo();
		int eCaret = findPosition (oPosn, findInfo);

		if (eCaret == CARET_INVALID) {
			if (isEmptyPosition (oPosn)) {
				eCaret = CARET_PRESENT;
			}
		}

		return eCaret;
	}

	int getCaretPosn (TextStream poStream, UnitSpan oSearch, TextPosnBase oResult) {
		return getCaretPosn (poStream, oSearch, oResult, false);
	}

	int getCaretPosn (TextStream poStream, UnitSpan oSearch, TextPosnBase oResult, boolean bAllowDescendents) {
		int eResult = CARET_INVALID;

// The line is totally empty; use the line's "empty" position
		if (getGlyphCount() == 0) {
			if ((mpoExtra != null) && (mpoExtra.mpoEmptyPosn != null) && (mpoExtra.mpoEmptyPosn.stream() == poStream)) {
				oResult.copyFrom (mpoExtra.mpoEmptyPosn);
				eResult = CARET_PRESENT;
			}
		}

// Normal line: Currently this is a fairly brute force implementation,
// running through every single glyph in the line.
		else {
			float oX = Units.toFloat (oSearch);
			//float oXRelative = oX - moAMin;
			HitTest oHitTest = new HitTest (this, poStream, oX);
			int i;

			oResult.associate (null);

// This loop iterates over the glyphs in the line, using the HitTest
// object to record how close the text positions in this glyph are,
			for (i = 0; i < getGlyphCount(); i++) {
				int nGlyphLocIndex = getGlyphLocOrder (i);
				GlyphLoc oGlyphLoc = getGlyphLoc (nGlyphLocIndex);

				int nCharIndex = oGlyphLoc.getMapIndex();
				DispPosn oPosition = getMappedPosition (nCharIndex);

				if ((oPosition.pp().stream() == poStream) || (bAllowDescendents && oPosition.pp().stream().isDescendentOf (poStream))) {
					oHitTest.tryHit (nGlyphLocIndex, oPosition);
				}
			}

// In obscure circumstances, it is passible for no position to be found
// in the above loop, but a valid position present.  This happens if
// we're looking for the parent stream in a stream hierarchy, the line
// starts with--or within--a nested field, and the field is the last item
// in the parent stream.  In such a case, there is no glyph corresponding
// to (a position in) the parent stream.  However, the line's position
// map will end with an empty position that identifies the parent stream.
// The code below detects that situation, and if present, allows
// positioning after the nested field.
			if ((! oHitTest.hasHit()) && (poStream != null)) { // TBD optimize this loop not to test all glyphs
				DispPosn poFieldPosn = null;
				for (i = 1; i < getPositionCount(); i++) { // note: start at 1
					DispPosn oTest = getPosition (i);
					if ((oTest.getMapLength() == 0) && (oTest.pp().stream() == poStream)) {
						poFieldPosn = getPosition (i - 1);
						break;
					}
				}

				if ((poFieldPosn != null) && (poFieldPosn.getMapLength() > 0)) {
					int nAfterCharIndex = poFieldPosn.getMapIndex() + poFieldPosn.getMapLength();
					oHitTest.tryHit (getMappedGlyphLoc (nAfterCharIndex - 1), poFieldPosn, nAfterCharIndex, 0);
				}
			}

			eResult = oHitTest.reconcile (oResult);
			if (eResult != CARET_INVALID) {
				checkAXTELigature (oResult, true);
			}
		}

		return eResult;
	}

	int getCaretStartEnd (TextStream poStream, boolean bEnd, boolean bVisual, TextPosnBase oResult) {
		int eCaret = CARET_INVALID;

// If this line is empty, return the only position present.
		if ((getPositionCount() == 0) && (mpoExtra != null) && (mpoExtra.mpoEmptyPosn != null)) {
			oResult.copyFrom (mpoExtra.mpoEmptyPosn);
			eCaret = CARET_PRESENT;
		}

// Visual start/end: we cheat and defer to the graphical caret position
// search for the location nearest to the start or end X extent of the
// line.
		else if (bVisual) {
			UnitSpan oSearch;
			if (bEnd == isRTL()) {								// start/LTR or end/RTL ...
				oSearch = Units.toUnitSpan (moJFAMin);			// ... use left side
			} else {											// end/LTR or start/RTL ...
				oSearch = Units.toUnitSpan (moJFAMaxExtended);	// ... use right side
			}
			eCaret = getCaretPosn (poStream, oSearch, oResult, true);
			if ((eCaret != CARET_INVALID) && (oResult.stream() != poStream)) {
				TextPosnBase oPath = new TextPosnBase();
				oResult.stream().isDescendentOf (poStream, oPath);
				if (bEnd != oResult.isRTL()) {
					oPath.nextUserPosn();
				}
				oResult.copyFrom(oPath);
			}
		}

// Logical start/end: Run through the position map, looking at position
// objects that have the same stream as the one in question.  If looking
// for the start, we can stop on the first one.  If looking for the end,
// we need to find the last one with the desired stream.
		else {
			int nStreamIndex = 0;
			int nCharIndex = 0;

			for (int i = 0; i < getPositionCount(); i++) {
				DispPosn oPosition = getPosition (i);

				if (oPosition.pp().stream() == poStream) {
					if (bEnd) {
						int nTestCharIndex = oPosition.getMapIndex() + oPosition.getMapLength();
						boolean bAtEnd = false;
						if (! isValidPosition (nTestCharIndex)) {	// don't position after last char
							if (oPosition.getMapLength() == 0) {
								nTestCharIndex = 0;
							}
							bAtEnd = true;
						}

						if (nTestCharIndex > nCharIndex) {
							nCharIndex = nTestCharIndex;
							int nStreamOffset = getPositionStreamCount (oPosition);
							if (bAtEnd && (nStreamOffset > 0)) {
								nStreamOffset--;
							}
							nStreamIndex = charToStreamIndex (oPosition, oPosition.getMapIndex(), nStreamOffset);
							eCaret = CARET_PRESENT;
						}
					}

					else {											// start
						int nTestCharIndex = oPosition.getMapIndex();
						if (isValidPosition (nTestCharIndex)) {
							nStreamIndex = oPosition.pp().index();
							eCaret = CARET_PRESENT;
							break;									// as soon as we find one we can get out
						}
					}
				}
			}

			if (eCaret != CARET_INVALID) {
				oResult.associate (poStream, nStreamIndex);
			}
		}

		return eCaret;
	}

	int getCaretLeftRight (TextPosnBase oPosn, boolean bRight, TextPosnBase oResult) {
		int c = '\0';
		int eCaret = CARET_PRESENT;
		int eItem = TextItem.UNKNOWN;

//		if (bypassOptyca()) {
			oResult.copyFrom (oPosn);
			oResult.setRTL (false);
			eItem = bRight ? oResult.nextUserPosnType (false)
						   : oResult.prevUserPosnType (false);
			if (eItem == TextItem.UNKNOWN)
				return CARET_INVALID;
//		} else {
//			eCaret = display().getWRS().getCaretLeftRight (this, oPosn, bRight, oResult);
//			if (eCaret == CARET_INVALID)
//				return CARET_INVALID;
//			if (pcChar != NULL) {
//				TextPosnBase oTest = new TextPosnBase (oResult);
//				eItem = (bRight == oResult.IsRTL()) ? oTest.NextUserPosnType (FALSE)
//													: oTest.PrevUserPosnType (FALSE);
//			}
//		}

		switch (eItem) {
			case TextItem.CHAR:
				c = (bRight == oResult.isRTL()) ? oResult.nextChar (true) : oResult.prevChar (true);
				break;
			case TextItem.PARA:
				c = '\n';
				break;
			default:
				c = Pkg.EMBED_OBJ_CHAR;
				break;
		}

		PosnInfo findInfo = new PosnInfo();
		eCaret = findPosition (oResult, findInfo);
		if ((eCaret == CARET_INVALID) && (! isEmptyPosition (oResult))) {
			c = '\0';
		}

		return c;
	}

	PosnInfo getCaretGlyph (int nCharIndex, int nStreamOffset, int eAffinity) {
		PosnInfo result = new PosnInfo();

// Watson 1189350: Previous implementation simply called
// GetMappedGlyphLoc() on the final character index in the line.
// However, in cases where one character generates multiple glyphs, this
// returns the left-most one, which is appropriate only for RTL text.  For
// LTR text, we start by trying the right-most glyph location and do the
// old algorithm only if it turns out to be an RTL glyph.
		if (nCharIndex >= getCharCount()) {
			int nIndex = getGlyphCount() - 1;
			GlyphLoc oGlyphLoc = getGlyphLoc (nIndex); // last if multiple glyphs for char
			result.index = oGlyphLoc.getGlyphIndex();
			Glyph oGlyph = getGlyph (result.index);

			if (oGlyph.isRTL()) {
				oGlyphLoc = getMappedGlyphLoc (getCharCount() - 1); // first if multiple glyphs for char
				result.index = oGlyphLoc.getGlyphIndex();
				result.auxInfo = 0;
			} else {
				result.auxInfo = 100;
			}
		}

		else {
			int nCharInner = 0;

			if (nStreamOffset > 0) {
				charToStreamInfo (nCharIndex, result);
				if (nStreamOffset > result.auxInfo) {
					nStreamOffset = result.auxInfo;
				}

				if (result.auxInfo <= 1) {
					nCharInner = 100;
				} else {
					double nRatio = ((double) (nStreamOffset - 1)) / ((double) (result.auxInfo - 1));
					nCharInner = (int) Math.round (100.0 * (2.0 + nRatio) / 3.0); // match WRServices ratio
				}
			}

			else if ((eAffinity == TextPosnBase.AFFINITY_BEFORE) && (nCharIndex > 0)) {
// Watson 1175482: Previous version of this code always moved to the end
// of the previous character for AFFINITY_BEFORE.  Now do it only if
// there is a direction change between the two characters.	The old
// implementation was sometimes leading to incorrect results with Thai
// text in WRServices.
				GlyphLoc oGlyphLoc = getMappedGlyphLoc (nCharIndex);
				result.index = oGlyphLoc.getGlyphIndex();
				Glyph oGlyph = getGlyph (result.index);
				GlyphLoc oPrevGlyphLoc = getMappedGlyphLoc (nCharIndex - 1);
				Glyph oPrevGlyph = getGlyph (oPrevGlyphLoc.getGlyphIndex());

				if (oGlyph.isRTL() != oPrevGlyph.isRTL()) {
					nCharIndex--;
					nCharInner = 100;
				}
			}

//			if (BypassOptyca()) {
				result.index = nCharIndex;
				result.auxInfo = nCharInner;
//			}
//
//			else {
// TBD: does not always handle the compounding of AXTE ligatures and
// WRServices.	Fortunately, these cases should not currently occur.
//				Display().getWRS().getCaretGlyph (this, nCharIndex, nCharInner, result.index, result.auxInfo);
//			}
		}

		return result;
	}

	void checkAXTELigature (TextPosnBase oPosn, boolean bForward) {
		PosnInfo info = new PosnInfo();
		int eCaret = findPosition (oPosn, info);
		if (eCaret == CARET_INVALID) {
			return;
		}

		if (info.auxInfo == 0) {	// stream offset
			return;
		}

		DispPosn oMapPosn = getMappedPosition (info.index);
		if (bForward) {
			oPosn.index (oMapPosn.pp().index() + oMapPosn.getStreamCount());
		} else {
			oPosn.index (oMapPosn.pp().index());
		}
	}

	LineHeight dispHeight () {
		return moHeight;
	}

	boolean isAtStart (TextPosnBase oPosn) {
		TextPosnBase oResult = new TextPosnBase();
		if (getCaretStartEnd (oPosn.stream(), false, false, oResult) == CARET_INVALID) {
			return false;
		}
		oResult.tighten (true);

		TextPosnBase oCurrent = new TextPosnBase (oPosn);
		oCurrent.tighten (true);

		return (oCurrent.stream() == oResult.stream()) && (oCurrent.index() == oResult.index());
	}

//	void Invalidate (Rect oInvalid, boolean bEraseBkgnd) {
//		CoordPair oOffset (GetAMin(), GetBMin());
//		ABXY.toXY (moXYOrigin, Frame().getLayoutOrientation(), oOffset);
//		Frame().invalidateArea (oInvalid + oOffset, bEraseBkgnd);
//	}

//	boolean GetInvalidationRect (TextPosnBase oStart, TextPosnBase oEnd, UnitSpan oLeft, UnitSpan oRight) {
//		boolean bStarted = false;
//		boolean bPrevStreamMatch = false;
//		int nRangeStart = oStart.index();
//		int nRangeEnd = oEnd.index();
//		float oAMin = 0;
//		float oAMax = 0;
//
//		for (int i = 0; i < GetPositionCount(); i++) {
//			DispMapPosition oPosition = getPosition (i);
//			if (oPosition.stream() == oStart.stream()) {
//				int nCharIndex = oPosition.getMapIndex();
//				int nCharMax = nCharIndex + oPosition.getMapLength();
//
//				while ((nCharIndex <= nCharMax) && (i < getPositionCount())) {
//					PosnInfo posnInfo = CharToStreamInfo (oPosition, nCharIndex);
//					if (nCharIndex >= nCharMax) {
//						posnInfo.auxInfo = 1;
//					}
//
//					for (int nStreamOffset = 0; nStreamOffset < posnInfo.auxInfo; nStreamOffset++) {
//						int nStreamIndex = CharToStreamIndex (oPosition, nCharIndex, nStreamOffset);
//						float oOffset1 = getCharOffset (i, nCharIndex, nStreamOffset, TextPosnBase.AFFINITY_BEFORE);
//						float oOffset2 = getCharOffset (i, nCharIndex, nStreamOffset, TextPosnBase.AFFINITY_AFTER);
//						boolean bCheckOffset1 = false;
//						boolean bCheckOffset2 = false;
//
//						if (nStreamIndex < nRangeStart) {
//							if (! bStarted) {
//								oAMin = oOffset1;
//								oAMax = oOffset1;
//								bPrevStreamMatch = true;
//								bCheckOffset2 = true;
//							}
//						} else if (nStreamIndex > nRangeEnd) {
//							if (bStarted || bPrevStreamMatch) {
//								bCheckOffset1 = true;
//								bCheckOffset2 = true;
//								bStarted = true;
//							}
//						} else if (nStreamIndex <= nRangeEnd) {
//							if (bStarted) {
//								bCheckOffset1 = true;
//							} else {
//								if ((nStreamIndex == nRangeStart) || (! bPrevStreamMatch)) {
//									oAMin = oOffset1;
//									oAMax = oOffset1;
//								}
//								bStarted = true;
//							}
//							bCheckOffset2 = true;
//						}
//
//						if (bCheckOffset1) {
//							if (oOffset1 < oAMin) {
//								oAMin = oOffset1;
//							} else if (oOffset1 > oAMax) {
//								oAMax = oOffset1;
//							}
//						}
//
//						if (bCheckOffset2) {
//							bCheckOffset2 = true;
//							if (oOffset2 < oAMin) {
//								oAMin = oOffset2;
//							}
//							else if (oOffset2 > oAMax) {
//								oAMax = oOffset2;
//							}
//						}
//
//						if (nStreamIndex >= nRangeEnd) {
//							i = GetPositionCount();
//							break;
//						}
//					}
//					nCharIndex++;
//				}
//			}
//		}
//
//		if (bStarted) {
//			oLeft = oAMin.unitSpan();
//			oRight = oAMax.unitSpan();
//		}
//
//		return bStarted;
//	}

	void fill (DispLineRaw poSource, int nStart, int nLength) {
		TextAttr poStartAttr = null;
		boolean bHasObjects = false;
//		boolean bCopyGlyphs = bypassOptyca();		// TODO:
//		boolean bCopyGlyphs = true;

		int nVisual = (nLength > moLineDesc.mnTrailingSpaces) ? (nLength - moLineDesc.mnTrailingSpaces) : 0;
		if (nVisual < nLength) {
			TextAttr poTrailingAttr;
			poTrailingAttr = poSource.getMappedAttr (nStart + nVisual);
			if ((poTrailingAttr != null) && (poTrailingAttr.invisibleEnable()) && (poTrailingAttr.invisible()) && (poTrailingAttr.invisChar() != '\0')) {
				nVisual = nLength;
			}
		}

		if (nLength == 0) {
// TBD: it was not the intent to mask poStartAttr at the outer scope.
// However, this bug crept in and fixing it causes text shifts.
			TextAttr poStartAttr2 = getLastAttr();
			if (poStartAttr2 != null) {
				moHeight.accumulate (poStartAttr, (poStartAttr2.layoutOrientation() == TextAttr.ORIENTATION_HORIZONTAL) ? GFXGlyphOrientation.HORIZONTAL : GFXGlyphOrientation.VERTICAL, isFirstLineInStream());
			}

			if ((mpoExtra == null) || (mpoExtra.mpoEmptyPosn == null)) {
				needExtra();
				mpoExtra.mpoEmptyPosn = new TextPosn();
			}
			mpoExtra.mpoEmptyPosn.associate (stream(), Integer.MAX_VALUE, TextPosn.POSN_BEFORE);
		}

		else {
			if (mpoExtra != null) {
				mpoExtra.mpoEmptyPosn = null;
			}

			preAllocChars (nLength);

			DispPosn oParentStart = poSource.getMappedPosition (nStart);
			int nIndex = charToStreamIndex (oParentStart, nStart, 0);
			DispPosn oInitialPosn = new DispPosn (oParentStart.pp().stream(), nIndex, oParentStart.pp().position());
			oInitialPosn.setStreamCount (oParentStart.getStreamCount());

			WrappedSpan oPosition = new WrappedSpan (this, poSource.getPositionMap(), nStart, nLength, oInitialPosn);
			WrappedSpan oRun = new WrappedSpan (this, poSource.getRunMap(), nStart, nLength, null);

			TextAttr poPrevAttr = null;
			boolean bHasSingleColour = true;
			boolean bHasDecoration = false;

			poStartAttr = oRun.r().getAttr();

			if ((poStartAttr.charSpacingEnable() && (poStartAttr.charSpacing().getLengthValue() != 0))
			 || (poStartAttr.wordSpacingEnable() && (poStartAttr.wordSpacing().getLengthValue() != 0))) {
//				bCopyGlyphs = false;
			}

// Watson 1397698: If the line starts with a nested field, there will be
// an empty position in the parent line.  This must be "copied" to the
// wrapped line, otherwise the position map will be set up wrong and it
// can eventually lead to an infinite loop in Designer.
			if (oParentStart.getMapLength() == 0) {
				oPosition.update (nStart);
			}

			int nLimit = nStart + nLength;
			for (int i = nStart; i < nLimit; i++) {
				int c = poSource.getChar (i);
				int eBreak = poSource.getBreakData (i);

				if (moLineDesc.meJustifyH == TextAttr.JUST_H_RADIX) {
					needExtra();
					if (mpoExtra.mnRadixCharIndex == Integer.MAX_VALUE) {
						if (poStartAttr != null) {
							if ((! legacyPositioning()) && (poStartAttr.radixPos() != Integer.MAX_VALUE)) {
								if (i == poStartAttr.radixPos()) {
									mpoExtra.mnRadixCharIndex = getCharCount();
								}
							} else if (c == poStartAttr.radixChar()) {
								mpoExtra.mnRadixCharIndex = getCharCount();
							}
						}
					}
					if (TextCharProp.getBreakClass (eBreak) == TextCharProp.BREAK_NU) {
						mpoExtra.mnLastDigitIndex = getCharCount();
					}
				}

				oPosition.update (i);
				oRun.update (i);

				if (Decoration.hasDecoration (oRun.r().getAttr())) {
					bHasDecoration = true;
				}

				DispEmbed poDispEMbed = poSource.isObject (i);
				if (poDispEMbed != null) {
//					bCopyGlyphs = false; // in case tab
					add (new DispEmbed (poDispEMbed), getCharCount());
					bHasObjects = true;
					if (poSource.tabAt (i) != null) {
						moHeight.accumulate (oRun.r().getAttr(), oRun.r().glyphOrientation(), isFirstLineInStream());
					} else {
						TextEmbed poEmbed = poDispEMbed.getEmbed();
						if (poEmbed.enforceHeight()) {
							moHeight.cancelOverride();
						}
						UnitSpan oHeight = poEmbed.height();
						if (poEmbed.embedAt() == TextEmbed.EMBED_AT_BOTTOM) {
							moHeight.accumulateSize (oHeight);
						} else {
							moHeight.accumulateAscent (oHeight);
						}
					}
				} else {
					moHeight.accumulate (oRun.r().getAttr(), oRun.r().glyphOrientation(), isFirstLineInStream());
				}

				addChar (c, eBreak);

				TextAttr poThisAttr = oRun.r().getAttr();
				if ((poThisAttr != poPrevAttr) && (poThisAttr != null) && (poPrevAttr != null)) {
					if (testColourChange (poPrevAttr, poThisAttr)) {
						bHasSingleColour = false;
					}
					if ((poThisAttr.charSpacingEnable() && (poThisAttr.charSpacing().getLengthValue() != 0))
					 || (poThisAttr.wordSpacingEnable() && (poThisAttr.wordSpacing().getLengthValue() != 0))) {
//						bCopyGlyphs = false;
					}
				}
				poPrevAttr = poThisAttr;
			}

			oPosition.finish();
			oRun.finish();

			reconcileBaselineShifts();

			setHasDecoration (bHasDecoration);
			setHasSingleColour (bHasSingleColour);

			TextStream poRootStream = frame().getStream();
			TextStream poLastStream = getPosition (getPositionCount()-1).pp().stream();
			if (poLastStream != poRootStream) {
				DispPosn oPath = new DispPosn();
				poLastStream.isDescendentOf (poRootStream, oPath.pp());
				oPath.pp().position (TextPosnBase.POSN_BEFORE);
				oPath.pp().nextUserPosn();
				if (! oPath.pp().nextUserPosn()) {
					add (oPath, getCharCount(), 0);
				}
			}
		}

		setVisualCharCount (nVisual);
		recordStartAttr (poStartAttr);

		format (poSource, nStart);

		if (bHasObjects) {
			for (int i = 0; i < getGlyphCount(); i++) {
				GlyphLoc oGlyphLoc = getGlyphLoc (i);
				int nCharIndex = oGlyphLoc.getMapIndex();
				DispEmbed poDispEmbed = isObject (nCharIndex);
				if (poDispEmbed != null) {
//					Glyph oGlyph = getGlyph (oGlyphLoc.getGlyphIndex());
//					poDIspEmbed.addOwner (this, Units.toUnitSpan (oGlyph.getDrawX (this)), Units.toUnitSpan (oGlyph.getDrawY (this)));
					if (tabAt (nCharIndex) != null) {
						setChar (nCharIndex, '\t', TextCharProp.defaultSpace);
					}
				}
			}
		}

		moHeight.reconcile();

		float oRawWidth = moLineDesc.moMarginL + moLineDesc.moSpecial + moRawWidth + moLineDesc.moMarginR;
		moJFRawWidth = Units.forceInt (oRawWidth);

		Rect oExtent = getABLineExtent();

		moBMinExtended = Units.toInt (oExtent.top());
		moBMaxExtended = Units.toInt (oExtent.bottom());
		UnitSpan oExtendedHeight = Units.toUnitSpan (moBMaxExtended - moBMinExtended);
		if (isFirstLineInStream()) {
			if (moHeight.adjustLineSpacing (oExtent.top())) {	// don't clip rogue ascenders on first line
				moBMinExtended = 0;								// in case line pushed down
				moBMaxExtended = Units.toInt (oExtendedHeight);
			}
		}
	}

//	void fill (TextPosnBase oPosn, TextLayoutLine poLayoutLine) {
//// Normal case: non-empty line.
//// Set up the initial display run object in a templated DispLineSpan
//// object that will update the display run map as content is processed.
//		TextAttr poPrevAttr = poStartAttr;
//		TextDispRun oDispRunSource (Frame(), poStartAttr);
//		DispLineSpan oDispRun (this, oDispRunSource);
//		boolean bHasDecoration = TextDecoration.hasDecoration (poStartAttr);
//		boolean bHasSingleColour = true;
//
//// Similarly, set up the initial position object that will update the
//// line's position map as content is processed.
//		DispLineSpan oDispPosn (this, oPosn);
//
//// Prepare to process the line's content.
//		PreAllocChars (nChars);
//		PreAllocGlyphs (nGlyphs);
//		SetVisualCharCount (nChars - moLineDesc.mnTrailingSpaces);
//
//		while (GetCharCount() < nChars) {
//// This loop process the Unicode content only.	It populates the line's
//// Unicode character array, the break class array, the display run map
//// and the position map.
//			TextItemCode eItem = oPosn.next (true);
//			switch (eItem) {
//				case TEXT_ITEM_PARA:
//				case TEXT_ITEM_CHAR: {
//						if (oDispPosn.index() + oDispPosn.length() != oPosn.index()) {
//							oDispPosn.reset (oPosn);
//						}
//
//						UniChar c = '\n';
//						if (eItem == TEXT_ITEM_PARA) {
//							oPosn.next();
//						}
//						else {
//							c = oPosn.nextChar();
//						}
//						if (c == '\n') {
//							c = ' '; // TBD: other special character checks?
//						}
//						AddChar (c, TextCharProp.getCharProperty (c));
//
//						break;
//					}
//				case TEXT_ITEM_ATTR: {
//						oDispRun.flush();
//						TextAttr poAttr = oPosn.nextAttr();
//						oDispRun.setAttr (poAttr);
//						if (TextDecoration.hasDecoration (poAttr)) {
//							bHasDecoration = true;
//						}
//						if (TestColourChange (poPrevAttr, poAttr)) {
//							bHasSingleColour = false;
//						}
//						poPrevAttr = poAttr;
//						moHeight.accumulate (poAttr, GFXGlyphOrientation.HORIZONTAL, IsFirstLineInStream()); // TBD:
//						break;
//					}
//				default:
//					assert (false); // TBD: can other cases happen?
//					oPosn.next();
//			}
//
//// Finish off maps.
//			oDispRun.flush();
//			oDispPosn.flush();
//
//// The following loop builds the glyphs for the line by iterating over
//// the line's runs, and the glyphs within each run.
//			TextPtr poUseAttr;
//			int nGlyphRunCount = poLayoutLine.getRunCount();
//			for (int nGlyphRun = 0; nGlyphRun < nGlyphRunCount; nGlyphRun++) {
//				TextGlyphRun poGlyphRun = poLayoutLine.getRun (nGlyphRun);
//				int nGlyphCount = poGlyphRun.getGlyphCount();
//				int nGlyph;
//
//				TextAttr poAttr = poGlyphRun.getAttr();
//				assert (poAttr != null);
//				TextAttr poFlatAttr = poAttr.conditionalFlattenBaselineShift();	// TODO: this is important for relative measurements
//				if (poFlatAttr != null) {
//					poUseAttr = poFlatAttr;
//				}
//				else {
//					poUseAttr.attach (poAttr);
//				}
//
//				FontInstance oFontInstance = poUseAttr.fontInstance();
//				assert (! oFontInstance.isNull());
//
//				boolean bIsShifted = poGlyphRun.isShifted();
//				CoordPair oShift = poGlyphRun.getShift();
//				boolean bIsRTL = poGlyphRun.isRTL();
//
//				if (poGlyphRun.isCharRun()) {
//// Process the glyphs in the run to generate display glyph objects for
//// this line.
//// Character run: need to look up both the width and glyph ID for each
//// character.  Fortunately this goes through the font service cache for
//// performance.
//					boolean bHorizontal = UsesHorizontalGlyphs (poGlyphRun.getGlyphOrientation());
//					for (nGlyph = 0; nGlyph < nGlyphCount; nGlyph++) {
//						UniChar c = poGlyphRun.getGlyph (nGlyph);
//						GlyphID nGlyphID;
//						oFontInstance.getGlyphID (c, nGlyphID, bHorizontal);
//						oLineWidth = AddGlyph (nGlyphID, oFontInstance.getCharWidth (c, bHorizontal), oLineWidth, oShift, bIsShifted && (nGlyph == 0), bIsRTL);
//					}
//				}
//
//				else {
//// Glyph ID run: have the glyph IDs already.	Note that the XTG font
//// service doesn't support the lookup of glyph width given glyph ID, so
//// we have to call CoolType directly (not cached).
//					FontInstance oNonConst = ((FontInstance) (oFontInstance));
//					CCTFontInstance oCTInstance (oNonConst.getCTFontInstance()); // why isn't this a const method?
//					for (nGlyph = 0; nGlyph < nGlyphCount; nGlyph++) {
//						GlyphID nGlyphID = poGlyphRun.getGlyph (nGlyph);
//						oLineWidth = AddGlyph (nGlyphID, oCTInstance.getWidth (nGlyphID), oLineWidth, oShift, bIsShifted && (nGlyph == 0), bIsRTL);
//					}
//				}
//			}
//
//// Check whether this line has a complex mapping.  Simple mapping is
//// defined as 1:1 character to glyph mapping with strict left-to-right
//// order.
//			boolean bComplexMapping = false;
//			int nMappings = poLayoutLine.getMappingCount();
//			for (i = 0; i < nMappings; i++) {
//				int nCharStart;
//				int nGlyphStart;
//				int nCharLength;
//				int nGlyphLength;
//
//				poLayoutLine.getMapping (i, nCharStart, nGlyphStart, nCharLength, nGlyphLength);
//
//				if ((nCharStart != i) || (nGlyphStart != i) || (nCharLength != 1) || (nGlyphLength != 1)) {
//					bComplexMapping = true;
//					break;
//				}
//			}
//
//			if (bComplexMapping) {
//// If there is a complex mapping, the line will need a glyph location
//// (TextGlyphLoc) map.	This maps character indexes to glyphs.	In
//// addition, there will need to be a glyph order array, to map from glyph
//// indexes to glyph location indexes.
//				for (i = 0; i < nMappings; i++) {
//					int nCharStart;
//					int nGlyphStart;
//					int nCharLength;
//					int nGlyphLength;
//
//					poLayoutLine.getMapping (i, nCharStart, nGlyphStart, nCharLength, nGlyphLength);
//
//					if (nCharLength == 1) {
//// Currently AXTE supports only 1:n and n:1 mappings.  If this mapping
//// has inly one character, create a glyph location for each of the
//// character's glyphs (typically there's only one glyph).  TBD: need to
//// make sure that this is consistent with what is generated from a
//// WRServices layout.
//						boolean bMultiple = nGlyphLength > 1;
//						for (int j = 0; j < nGlyphLength; j++) {
//							int nGlyphIndex = nGlyphStart + j;
//							GlyphLoc oGlyphLoc (nGlyphIndex);
//							Add (oGlyphLoc, nCharStart, 1);
//							if (bMultiple) {
//								Glyph (nGlyphIndex).setInMultiple (true);
//							}
//						}
//					}
//
//					else if (nGlyphLength == 1) {
//// Multiple characters, single glyph (e.g., ligature).	This situation is
//// represented by a single glyph location.
//						GlyphLoc oGlyphLoc (nGlyphStart);
//						Add (oGlyphLoc, nCharStart, nCharLength);
//					}
//
//					else {
//						assert (false);
//					}
//				}
//
//// This fills in the glyph order array for mapping from glyph indexes to
//// glyph location indexes.
//				PopulateGlyphLocOrder();
//			}
//
//// Next, determine the visible width (the line width excluding trailing
//// spaces).  TBD: need to validate this against RTL or BIDI text.
//			oVisibleWidth = oLineWidth;
//
//			for (i = GetGlyphCount(); i > 0; ) {
//				i--;
//				DispMapGlyphLoc oGlyphLoc = GetOrderedGlyphLoc (i);
//				if (oGlyphLoc.getMapLength() > 1) {
//					break;
//				}
//				if (IsVisualChar (oGlyphLoc.getMapIndex())) {
//					break;
//				}
//				Glyph oGlyph = GetGlyph (oGlyphLoc.getMapIndex());
//				oVisibleWidth = oGlyph.getOriginalX();
//			}
//
//			SetHasDecoration (bHasDecoration);
//			SetHasSingleColour (bHasSingleColour);
//		}
//
//// Update line height information.
//		RecordStartAttr (poStartAttr);
//		ReconcileBaselineShifts();
//		moHeight.reconcile();
//
//// Update various width-related members.
//		SetWidth (oLineWidth);
//		SetTrailingWidth (oLineWidth - oVisibleWidth);
//		moRawWidth = oVisibleWidth;
//		moRawWidthFull = oLineWidth;
//		moAMax = oVisibleWidth;
//		moAMaxExtended = oLineWidth;
//		moJFAMax = moAMax.force();
//		moJFAMaxExtended = moAMaxExtended;
//		moBMinExtended = 0;
//		moBMaxExtended = GetBExtent();
//
//		SetLayoutLine (true);
//	}

	void compose (TextLayout poLayout, boolean bAllowCharGlyphs) {
		UnitSpan oHeight = getBExtent();
		UnitSpan oAscent = Units.toUnitSpan (moHeight.textOffset (GFXGlyphOrientation.HORIZONTAL));
		UnitSpan oDescent = oHeight.subtract (oAscent);

		LayoutRenderer oRenderer = new LayoutRenderer (poLayout, this, oAscent, oDescent, bAllowCharGlyphs);

		UnitSpan oLargeNeg = new UnitSpan (UnitSpan.INCHES_72K, -0xFFFFFF);
		UnitSpan oLargePos = new UnitSpan (UnitSpan.INCHES_72K, 0xFFFFFF);
		Rect oInvalid = new Rect (oLargeNeg, oLargeNeg, oLargePos, oLargePos);

		TextDrawInfo oDrawInfo = new TextDrawInfo (oRenderer);
		DrawParm oParm = new DrawParm (oDrawInfo);
		oParm.setDriver (oRenderer.driver());
		oParm.setInvalid (oInvalid);
//		oParm.setOpt (TextPrefOpt.OPT_LINE);

		gfxDraw (oParm);

		oRenderer.finish();
	}

	void clear () {
		super.clear();
		moMaps.clear();
		moHeight.detach();

		mpoStartAttr = null;

		moAMin = 0;
		moAMax = 0;
		moAMaxExtended = 0;
		moJFAMin = 0;
		moJFAMax = 0;
		moJFAMaxExtended = 0;
		moBMin = 0;
		moBMinExtended = 0;
		moBMaxExtended = 0;
		moRawWidth = 0;
		moRawWidthFull = 0;
		moXYOrigin = CoordPair.zeroZero();

		moHeight.reset();

		for (int i = 0; i < moCharRanges.size(); i++) {
			CharRange oRange = moCharRanges.get (i);
			oRange.moMinX = CHAR_RANGE_DEFAULT;
			oRange.moMaxX = CHAR_RANGE_DEFAULT;
			oRange.mpoNext = null;
		}

		if (mpoExtra != null) {
			mpoExtra.clear();
		}
	}

//	void getOptycaLog () {
//		return (mpoExtra == null) ? null : mpoExtra.mpoOptycaLog;
//	}

//	void setOptycaLog (void poOptycaLog) {
//		NeedExtra();
//		mpoExtra.setOptycaLog (Display().getContext(), poOptycaLog); // Display() not valid in our destructor
//	}

	void textDebug (int lineNumber) {
		TextContext context = display().getContext();
		if (context.debug()) {
			int i;

			StringBuilder output = new StringBuilder (Integer.toString (lineNumber));
			output.append (": Line left: ");
			output.append (getAMin().toString());
			output.append (", right: ");
			output.append (getAMax().toString());
			output.append (", top: ");
			output.append (getBMin().toString());
			output.append (", bottom: ");
			output.append (getBMax().toString());
			output.append (", objects: ");
			output.append (Integer.toString (getEmbedCount()));
			output.append (", characters: ");
			output.append (Integer.toString (getCharCount()));
			context.debug (output.toString());

			for (i = 0; i < getEmbedCount(); i++) {
				output.delete (0, output.length());
				DispEmbed oMapEmbed = getEmbed (i);
				TextEmbed poEmbed = oMapEmbed.getEmbed();
				output.append ("  Object ");
				output.append (Integer.toString (i));
				output.append (": width: ");
				output.append (poEmbed.width().toString());
				output.append (", height: ");
				output.append (poEmbed.height().toString());
				context.debug (output.toString());
			}

			output.delete (0, output.length());
			output.append ("  Text: ");
			for (i = 0; i < getCharCount(); i++) {
				int c = getChar (i);
				if ((c >= ' ') && (c < 0x7F)) {
					output.append ((char) c);
				} else {
					output.append (Units.hexToString (c));
				}
			}
			context.debug (output.toString());

			for (i = 0; i < getGlyphCount(); i++) {
				output.delete (0, output.length());
				Glyph oGlyph = getGlyph (i);
				GlyphLoc oGlyphLoc = getOrderedGlyphLoc (i);

				String sTypeface = "?";
				String sEncoding = "?";
				TextAttr poAttr = getMappedAttr (oGlyphLoc.getMapIndex());
				if (poAttr != null) {
					sTypeface = poAttr.typeface();
					sEncoding = poAttr.encoding();
				}

				int milliPointOffset = Units.toInt (oGlyph.getDrawX (this));
				UnitSpan unitOffset = new UnitSpan (UnitSpan.POINTS_1K, milliPointOffset);
				int milliPointWidth = Units.toInt (oGlyph.getDrawNextX (this) - oGlyph.getDrawX (this));
				UnitSpan unitWidth = new UnitSpan (UnitSpan.POINTS_1K, milliPointWidth);

				output.append ("   Glyph: ");
				output.append (Integer.toString (oGlyph.getGlyph()));
				output.append (", index: ");
				output.append (Integer.toString (oGlyphLoc.getMapIndex()));
				output.append (" (");
				output.append (Integer.toString (oGlyphLoc.getMapLength()));
				output.append ("), x: ");
				output.append (unitOffset.toString());
				output.append (", width: ");
				output.append (unitWidth.toString());
				output.append (", font: ");
				output.append (sTypeface);
				output.append (" (");
				output.append (sEncoding);
				output.append (")");
				context.debug (output.toString());
			}
		}
	}

	private void needExtra () {
		if (mpoExtra == null) {
			mpoExtra = new Extra();
		}
	}

	private boolean isEmptyPosition (TextPosnBase oPosn) {
		return (getPositionCount() == 0)
			&& (mpoExtra != null)
			&& (mpoExtra.mpoEmptyPosn != null)
			&& (oPosn.stream() == mpoExtra.mpoEmptyPosn.stream())
			&& (oPosn.index() == mpoExtra.mpoEmptyPosn.index());
	}

	private void format (DispLineRaw poSourceLine, int nCharOffset) {
		DispMapSet maps = getMaps();
		FormatInfo formatInfo = poSourceLine.getFormatInfo();
		MappingManager mappingManager = formatInfo.getMappingManager();
		AFERun afeRun = formatInfo.getAFERun();

		int glyphCount = getCharCount();
		int glyphOffset = nCharOffset;
		if (mappingManager != null) {
			mappingManager.applyToWrappedLine (this, afeRun, nCharOffset);
		}
		if ((poSourceLine.getGlyphLocMap() != null)
		 && (poSourceLine.getGlyphLocMap().size() > 0)
		 && (maps.moGlyphLocMap != null)
		 && (maps.moGlyphLocMap.size() > 0)) {
			glyphCount = maps.moGlyphLocMap.size();
			glyphOffset = poSourceLine.getGlyphLocMap().findItem (nCharOffset);
		}
		preAllocGlyphs (glyphCount, false);

		if (mappingManager != null) {
			populateGlyphLocOrder (glyphCount);
		}

		GlyphMaker glyphMaker = new GlyphMaker (this);
		assert (afeRun != null);

		for (int i = 0; i < glyphCount; i++) {											// i counts in glyph order
			int glyphLocIndex = getGlyphLocOrder (i);									// glyphLocIndex is in char order
			GlyphLoc glyphLoc = getGlyphLoc (glyphLocIndex);
			AFEElement afeElement = afeRun.getElement (glyphLocIndex + glyphOffset);	// from parent line, char order
			glyphMaker.addGlyph (afeElement, glyphLoc.getMapIndex());
		}

		glyphMaker.applyWidths();
		moRawWidth = getWidth() - getTrailingWidth();
		moRawWidthFull = getWidth();
	}

//	private float AddGlyph (GlyphID nGlyphID, float oGlyphWidth, float oLineWidth, CoordPair oShift, boolean bIsShifted, boolean bIsRTL) {
//		Glyph oGlyph = new Glyph();
//		oGlyph.setGlyph (nGlyphID);
//
//		float oX = oLineWidth;
//		if (bIsShifted && (oShift.X().value() != 0)) {
//			oX += float (oShift.X());
//		}
//		oGlyph.setOriginalX (oX);
//
//		float oNextX (oX + oGlyphWidth);
//		oGlyph.setOriginalNextX (oNextX);
//
//		int nGlyphIndex = GetGlyphCount();
//		Add (oGlyph);
//
//		if (bIsRTL || bIsShifted || (oShift.Y().value() != 0)) {
//			Glyph oAddedGlyph = Glyph (nGlyphIndex);
//			GlyphExtra oGlyphExtra = ForceExtra (GetGlyphCount() - 1);
//			oAddedGlyph.setShifted (true);
//			if (oShift.Y().value() != 0) {
//				oGlyphExtra.setY (oShift.Y());
//			}
//			oAddedGlyph.setRTL (bIsRTL);
//			if (bIsRTL) {
//				oGlyphExtra.setRTLWidth (oGlyphExtra.getWidth());
//			}
//		}
//
//		return oNextX;
//	}

	private void populateGlyphLocOrder (int glyphCount) {
		if (getGlyphLocMap().size() > 0) {			// if there isn't an LTR mapping ...
			allocateGlyphLocOrder (glyphCount); 	// ... need to map glyph indexes to mapped locations
			for (int i = 0; i < glyphCount; i++) {
				setGlyphLocOrder (getGlyphLoc(i).getGlyphIndex(), i);
			}
		}
	}

	@FindBugsSuppress(pattern="FE_FLOATING_POINT_EQUALITY")
	private boolean diffGlyphs (DispLineWrapped poCompare, int nThisIndex, int nCompareIndex) {
// First do obvious checks for diggerent glyphs or positions/
		Glyph oThisGlyph = getGlyph (nThisIndex);
		Glyph oCompareGlyph = poCompare.getGlyph (nCompareIndex);

// TBD: shouldn't moAMin be tested once outside this method?
		if ((oThisGlyph.getGlyph() != oCompareGlyph.getGlyph())
		 || ((oThisGlyph.getDrawX (this) + moAMin) != (oCompareGlyph.getDrawX (poCompare) + poCompare.moAMin))
		 || (oThisGlyph.getDrawY (this) != oCompareGlyph.getDrawY (poCompare))) {
			return true;
		}

// Next, set up for attribute comparisons.
		GlyphLoc oThisGlyphLoc = getOrderedGlyphLoc (nThisIndex);
		GlyphLoc oCompareGlyphLoc = poCompare.getOrderedGlyphLoc (nCompareIndex);
		int nThisRunIndex;
		int nCompareRunIndex;
		nThisRunIndex = getRunMap().findItem (oThisGlyphLoc.getMapIndex());
		nCompareRunIndex = poCompare.getRunMap().findItem (oCompareGlyphLoc.getMapIndex());

// If the attribute pointers are different, the attributes may still be
// the same, so we'll have to do a more expensive comparison (provided
// that one isn't NULL).  If both are NULL, we can do no further
// checking.
		TextAttr poThisAttr = getRun (nThisRunIndex).getAttr();
		TextAttr poCompareAttr = poCompare.getRun (nCompareRunIndex).getAttr();
		if (poThisAttr != poCompareAttr) {
			if ((poThisAttr == null) || (poCompareAttr == null)) {
				return true;
			}
			if (! poThisAttr.equals (poCompareAttr)) {
				return true;
			}
		}
		if (poThisAttr == null) {
			return false; // both NULL at this point
		}

// Decoration isn't normally applied to trailing spaces.  So, if we type
// a non-space after such a space, we'll need to treat the previous space
// as different if there is any decoration in effect.
		if (oThisGlyphLoc.getMapLength() > 1) {
			return false; // ligature is not a trailing space
		}
		boolean bThisVisual = isVisualChar (oThisGlyphLoc.getMapIndex());
		boolean bCompareVisual = poCompare.isVisualChar (oCompareGlyphLoc.getMapIndex());
		if (bThisVisual == bCompareVisual) {
			return false; // both trailing or both not trailing
		}

// If we got here, exactly one of the glyphs is a trailing space.  If
// there is any decoration in effect, we'll need to treat the glyphs as
// different.
		if ((GFXDecorationInfo.extractDecoration (poThisAttr.underline()) != GFXDecorationInfo.decorateNone)
		 || (GFXDecorationInfo.extractDecoration (poThisAttr.strikeout()) != GFXDecorationInfo.decorateNone)
		 || (GFXDecorationInfo.extractDecoration (poThisAttr.overline()) != GFXDecorationInfo.decorateNone)) {
			return true;
		}

		return false;
	}

//	private Rect getXYLineExtent () {
//		return ABXY.toXY (moXYOrigin, getABLineExtent(), frame().getLayoutOrientation());
//	}

	private Rect getABLineExtent () {
		DispRect oExtent = null;
		if (getGlyphCount() > 0) {
			oExtent = getABSubExtent (0, getGlyphCount() - 1);
		} else {
			oExtent = new DispRect();
		}
		oExtent.yMin = Units.toFloat (getBMin());	// TODO: Moved outside else block ...
		oExtent.yMax = Units.toFloat (getBMax());	// ... until we can get individual glyph bboxes for height

		UnitSpan oLeft = Units.toUnitSpan (oExtent.xMin);
		UnitSpan oTop = Units.toUnitSpan (oExtent.yMin);
		UnitSpan oRight = Units.forceUnitSpan (oExtent.xMax);
		UnitSpan oBottom = Units.forceUnitSpan (oExtent.yMax);

		if (getAMin().lt (oLeft)) {
			oLeft = getAMin();
		}
		if (getAMax().gt (oRight)) {
			oRight = getAMax();
		}

		return new Rect (oLeft, oTop, oRight, oBottom);
	}

//	private DispRect getXYSubExtent (int nFirst, int nLast) {
//		return ABXY.toXY (moXYOrigin, getABSubExtent (nFirst, nLast), frame().getLayoutOrientation());
//	}

	private DispRect getABSubExtent (int nFirst, int nLast) {
		DispRect oExtent = new DispRect();
		boolean bFirst = true;
		float oHorzOffset = Units.toFloat (moHeight.textOffset (GFXGlyphOrientation.HORIZONTAL));
		float oVertOffset = Units.toFloat (moHeight.textOffset (GFXGlyphOrientation.VERTICAL));

		for (int i = nFirst; i <= nLast; i++) {
			Glyph oGlyph = getGlyph (i);
			float oX = oGlyph.getDrawX (this);
			float oNextX = oGlyph.getDrawNextX (this);
			float oY = oGlyph.getDrawY (this);
			DispRect oGlyphExtent = getABGlyphBBox (i, true);

			if (oGlyphExtent.xMin >= 0) {
				oGlyphExtent.xMin = oX;
			} else {
				oGlyphExtent.xMin += oX;
			}

			oGlyphExtent.xMax += oX;
			if (oGlyphExtent.xMax < oNextX) {
				oGlyphExtent.xMax = oNextX;
			}

			if (GFXGlyphOrientation.usesHorizontalGlyphs (oGlyph.getOrientation())) {
				oY += oHorzOffset;
			} else {
				oY += oVertOffset;
			}

			oGlyphExtent.yMin += oY;
			oGlyphExtent.yMax += oY;

			if (bFirst) {
				oExtent.xMin = oGlyphExtent.xMin;
				oExtent.yMin = oGlyphExtent.yMin;
				oExtent.xMax = oGlyphExtent.xMax;
				oExtent.yMax = oGlyphExtent.yMax;
				bFirst = false;
			} else {
				if (oGlyphExtent.xMin < oExtent.xMin) {
					oExtent.xMin = oGlyphExtent.xMin;
				}
				if (oGlyphExtent.yMin < oExtent.yMin) {
					oExtent.yMin = oGlyphExtent.yMin;
				}
				if (oGlyphExtent.xMax > oExtent.xMax) {
					oExtent.xMax = oGlyphExtent.xMax;
				}
				if (oGlyphExtent.yMax > oExtent.yMax) {
					oExtent.yMax = oGlyphExtent.yMax;
				}
			}
		}

		oExtent.xMin += moAMin;
		oExtent.xMax += moAMin;

		if (oExtent.yMin > 0) {
			oExtent.yMin = 0;
		}
		float oBMax = Units.toFloat (getBExtent());
		if (oExtent.yMax < oBMax) {
			oExtent.yMax = oBMax;
		}

		return oExtent;
	}

	private float getCharOffset (int nPositionIndex, int nCharIndex, int nStreamOffset, int eAffinity) {
		int nActualChar = nCharIndex;
		int nActualStream = nStreamOffset;

		if (eAffinity == TextPosnBase.AFFINITY_BEFORE) {
			if (nStreamOffset > 0) {
				nActualStream--;
			} else if (nCharIndex > 0) {
				nActualChar--;
				nActualStream = 0;
				boolean bBackupPosition = false;
				if (nPositionIndex >= getPositionCount()) {
					bBackupPosition = true;
				} else {
					DispPosn oPosition = getPosition (nPositionIndex);
					if (nActualChar < oPosition.getMapIndex()) {
						bBackupPosition = true;
					}
				}
				if (bBackupPosition) {
					assert (nPositionIndex > 0);
					DispPosn oPrevPosition = getPosition (nPositionIndex - 1);
					if (getPositionType (oPrevPosition) == POSN_TYPE_LIGATURE) {
						nActualStream = oPrevPosition.getStreamCount() - 1;
					}
				}
			} else {
				eAffinity = TextPosnBase.AFFINITY_AFTER;
			}
		}

		if (moCharRanges == null) {						// TODO: need to understand the incremental allocation of this, drawing vs. other ops
			moCharRanges = new Storage<CharRange>();
		}
		if (nActualChar >= moCharRanges.size()) {
			int oldSize = moCharRanges.size();
			moCharRanges.setSize (getCharCount() + 1);	// alloc all to avoid many reallocs
			int newSize = moCharRanges.size();
			for (int i = oldSize; i < newSize; i++) {
				moCharRanges.set (i, new CharRange());
			}
		}

		CharRange poRange = moCharRanges.get (nActualChar);
		for (; nActualStream > 0; nActualStream--) {
			CharRange poNext = poRange.mpoNext;
			if (poNext == null) {
				poNext = new CharRange();
				poRange.mpoNext = poNext;
			}
			poRange = poNext;
		}
		boolean useMax = eAffinity == TextPosnBase.AFFINITY_BEFORE;
		float oX = useMax ? poRange.moMaxX : poRange.moMinX;

		if (! CharRange.isInitialized (oX)) {
			oX = 0;

			if (getGlyphCount() == 0) {
				oX = moAMin;
			}

			else {
				PosnInfo posnInfo = getCaretGlyph (nCharIndex, nStreamOffset, eAffinity);

				Glyph oGlyph = getGlyph (posnInfo.index);
				if (posnInfo.auxInfo >= 100) {
					oX = oGlyph.getOriginalNextX();
				} else {
					oX = oGlyph.getOriginalX() + (oGlyph.getWidth (this) * posnInfo.auxInfo / 100.0f);
				}
				oX += oGlyph.getOffsetX (this) + moAMin;
				if (useMax) {
					poRange.moMaxX = oX;
				} else {
					poRange.moMinX = oX;
				}
			}
		}

		return oX;
	}
	
	// FIXME: oFontInstance, oGlyphLoc, oGlyph, oRenderGlyph passed by reference in C++
	private void getRenderChar (TextAttr poAttr, FontInstance oFontInstance, GlyphLoc oGlyphLoc, Glyph oGlyph, RenderGlyph oRenderGlyph) {
		oRenderGlyph.mnGlyphID = oGlyph.getGlyph();
		oRenderGlyph.mcGlyph = '\0';
		if ((oGlyphLoc.getMapLength() == 1) && (! oGlyph.isInMultiple()) && (! oGlyph.isAXTELigature())) {
			oRenderGlyph.mcGlyph = getChar (oGlyphLoc.getMapIndex());
		}

// Invalid font instance/item: cannot do any font-based testing of the
// glyph, so use character mode.
		if (oFontInstance == null) {
			return;
		}
		FontItem poFontItem = oFontInstance.getFontItem();
		if (poFontItem == null) {
			return ;
		}

// A tab is stored as an embedded object character and its glyph ID is
// unpredictable.  Reconcile tabs here.
		if ((oGlyphLoc.getMapLength() == 1) && (tabAt (oGlyphLoc.getMapIndex()) != null)) {
			oRenderGlyph.mcGlyph = ' ';
//			if (! oFontInstance.getGlyphID (' ', oRenderGlyph.mnGlyphID, GFXGlyphOrientation.usesHorizontalGlyphs (oGlyph.getOrientation()))) {
//				return;
//			}
		}

// Acrobat cannot handle Unicode.  Therefore characters in a unicode font
// must go out in glyph mode.
		if (isUnicodeFont (poAttr)) {
			oRenderGlyph.mcGlyph = '\0';
			return;
		}

// If AXTE formatted the line, glyph mode is used only for NotDef glyphs.
//		if (bypassOptyca()) {													// TODO:
//			if (oRenderGlyph.mnGlyphID == poFontItem.getNotDefGlyphID()) {
//				oRenderGlyph.mcGlyph = '\0';
//			}
//			return;
//		}

// Ligature or other character combination: glyph mode.
		if (oGlyphLoc.getMapLength() > 1) {
			oRenderGlyph.mcGlyph = '\0';
			return;
		}

// Check for Optyca digit substitution (not currently enabled).  If it
// substituted digit glyphs, we'll have to use glyph mode.
		if ((oRenderGlyph.mcGlyph >= '0') && (oRenderGlyph.mcGlyph <= '9')) {
//			int eDigits = frame().getDigits (poAttr);
//			if ((eDigits != TextAttr.DIGITS_ARABIC) && (eDigits != TextAttr.DIGITS_LOCALE)) {
//				oRenderGlyph.mcGlyph = '\0';
//				return;
//			}
		}

// In some configurations, it is possible that the notdef glyph ID gets
// stored, even though the metrics are correct.  In such a case, use
// character mode.
		if (oRenderGlyph.mnGlyphID == poFontItem.getNotDefGlyphID()) {
			return;
		}

// Cannot look up the glyph: not wise to put out characters.
//		int nBaseGlyph = 0;			// TODO: need more support from AFE to do all the tests
//		if (! oFontInstance.getGlyphID (oRenderGlyph.mcGlyph, nBaseGlyph, UsesHorizontalGlyphs (oGlyph.getOrientation()))) {
//			oRenderGlyph.mcGlyph = '\0';
//			return oRenderGlyph;
//		}
		if (oGlyph.renderByGlyphID()) {
			oRenderGlyph.mcGlyph = '\0';
			return;
		}

// Stored glyph matches default glyph (no glyph substitution): use
// character mode.
//		if (oRenderGlyph.mnGlyphID == nBaseGlyph) {
//			return;
//		}

// For XDC fonts, we must use character mode.
//		UniChar cRenderChar;
//		if (poFontItem.getDisplayableChar (oRenderGlyph.mnGlyphID, cRenderChar)) {
//			oRenderGlyph.mcGlyph = cRenderChar;
//			return oRenderGlyph;
//		}

// Otherwise, must have a substituted glyph: use glyph mode.
//		oRenderGlyph.mcGlyph = '\0';
	}

	private boolean renderTabLeader (DrawParm oParm, DispTab poDispTab, GlyphLoc poGlyphLoc, DrawAttr oDrawAttr, DrawRun oRun) {
		int nCharIndex = poGlyphLoc.getMapIndex();
		TextAttr poAttr = getMappedAttr (nCharIndex);

		if ((poAttr == null) || (! poAttr.leaderPatternEnable())) {
			return true;
		}

		switch (poAttr.leaderPattern()) {
			case TextAttr.LEADER_PATTERN_RULE: {
				LeaderRule oFiller = new LeaderRule (oParm, oDrawAttr);
				return renderTabLeader (oFiller, poAttr, poDispTab, poGlyphLoc, oRun);
			}
			case TextAttr.LEADER_PATTERN_DOTS:
			case TextAttr.LEADER_PATTERN_USE_CONTENT: {
				LeaderContent oFiller = new LeaderContent (oParm, nCharIndex);
				return renderTabLeader (oFiller, poAttr, poDispTab, poGlyphLoc, oRun);
			}
		}

		return true;
	}

	private boolean renderTabLeader (LeaderFill oFiller, TextAttr poAttr, DispTab poDispTab, GlyphLoc poGlyphLoc, DrawRun oRun) {
		if (! oFiller.setup (this, poAttr, poDispTab, poGlyphLoc)) {
			return true;
		}

		oRun.flush();

		return oFiller.render();
	}

	private void reconcileBaselineShifts () {
		if (moHeight.hasBaselineShift()) {
			for (int i = 0; i < getCharCount(); i++) {
				if (isObject (i) == null) {
					DispRun oRun = getMappedRun (i);
					moHeight.accumulateBaselineShift (oRun.getAttr());
				}
			}
		}
	}

	private void recordStartAttr (TextAttr poStartAttr) {
		if (poStartAttr != null) {
			if (isFirstParaLine() && poStartAttr.spaceBeforeEnable()) {
				moHeight.before (Units.toInt (poStartAttr.spaceBefore().getLength()));
			}
			if (isLastParaLine() && poStartAttr.spaceAfterEnable()) {
				moHeight.accumulateAfter (poStartAttr.spaceAfter().getLength());
			}
			mpoStartAttr = poStartAttr;
		}
	}

//----------------------------------------------------------------------
//
//		TestColourChange - Helper function to compare two
//		attributes for colour or transparency change.
//
//----------------------------------------------------------------------
	private static boolean testColourChange (TextAttr poPrevAttr, TextAttr poThisAttr) {
		if ((poPrevAttr == poThisAttr) || (poPrevAttr == null) || (poThisAttr == null)) {
			return false;
		}

		if (poThisAttr.colourEnable() && poPrevAttr.colourEnable()) {
			if (! poThisAttr.colour().equals (poPrevAttr.colour())) {
				return true;
			}
		}
		if (poThisAttr.colourBgEnable() && poPrevAttr.colourBgEnable()) {
			if (! poThisAttr.colourBg().equals (poPrevAttr.colourBg())) {
				return true;
			}
		}
		if (poThisAttr.transparentEnable() && poPrevAttr.transparentEnable()) {
			if (poThisAttr.transparent() != poPrevAttr.transparent()) {
				return true;
			}
		}

		return false;
	}

//	private static int resolveLineBreak (int eLineEndState) {
//		switch (eLineEndState) {
//			case TextLayoutLine.LINE_END_LAST_WORD:
//				return DispLine.LINE_BREAK_HYPHEN_SUPPRESS;
//			case TextLayoutLine.LINE_END_HYPHEN:
//				return DispLine.LINE_BREAK_HYPHEN;
//			case TextLayoutLine.LINE_END_EMERGENCY:
//				return DispLine.LINE_BREAK_FORCED;
//		}
//		return DispLine.LINE_BREAK_NORMAL;
//	}

	private CharRange getCharRange (int index) {
		return moCharRanges.get (index);
	}
}
