package com.adobe.xfa.text;

import com.adobe.xfa.gfx.GFXEnv;
import com.adobe.xfa.ut.Rect;

/**
 * Properties for rendering text.
 * <p>
 * The number of properties available for the rendering of text has been
 * growing as more features are added to AXTE.	This class caches all of
 * those properties.  Instead ofusing forever growing parameter lists,
 * the client creates an instance of this class, populates on the
 * properties relevant to its usage, and then calls the appropriate
 * rendering method.
 * </p>
 * <p>
 * Currently, the following properties are supported.  All are optional,
 * except for the graphics environment.
 * </ul>
 * <li>
 * <b>Graphic Enviornment:</b> Graphic environment in which rendering is
 * to occur.
 * </li>
 * <li>
 * <b>Invalidation rectangle:</b> On a repaint, this is the damage area
 * of the rendering target that needs to be updated.  It is specified in
 * text object-relative coordinates.  It can also be used for coarse
 * control over what gets rendered.  In order to be drawn in a rendering
 * call, a glyph must overlap the invalidation rectangle.  The default
 * is a degenerate rectangle at (0,0).	This is defaulted at draw time
 * according to the invalid default setting (see below).
 * </li>
 * <li>
 * <b>Page rectangle:</b> Describes the extents of the page in
 * page-absolute coordinates.  Typically, the left and top are zero, and
 * the right and bottom are the width and height of the page,
 * respectively.  The default is a degenerate page at (0,0).  The page
 * rectangle is currently used only for aligning leader patterns.  Text
 * outside the page rectangle will be rendered if it meets other
 * constraints.
 * </li>
 * <li>
 * <b>Primary selection:</b> Pointer to a text selection ({@link
 * TextSelection}) object, describing a range of selected text along
 * with foreground and background colours, that may require different
 * rendering from the rest of the text.  Default is the null pointer,
 * indicating there is no primary selection in effect.	Note that, in
 * bidirectional text, selections may not be visually contiguous.
 * </li>
 * <li>
 * <b>Secondary selection:</b> A second, lower priority selection.
 * Default is the null pointer, indicating there is no secondary
 * selection in effect.  Note that the primary and secondary selections
 * may have different ranges and colours.  If both are specified and
 * overlap, text in the overlap area is rendered with the primary
 * selection properties.  a secondary selection might be used to
 * highlight a fillable field within flowing form letter content.
 * </li>
 * <li>
 * <b>Invalid default:</b> An enumeration that describes how to to
 * determine a default value for the invalidation rectangle if none is
 * provided.
 * </li>
 * <li>
 * <b>Truncate:</b> If true, glyphs must be entirely inside the text
 * object's extent in order to be renered.	If a glyph straddles the
 * extend boundary or is entirely outside the extent, it will not be
 * rendered.  If there is also an invalidation rectangle, either
 * specified or defaulted, the glyph must satisfy the invalidation
 * constraints as well in order to be rendered.  If this property is
 * false, no truncation is in effect.
 * </li>
 * <li>
 * <b>Obtimization level:</b> This property is largely deprecated, as
 * all "real" rendering targets support the maximum level of render
 * optimization.  It is used only in some test environments.  Text is
 * rendered line by line.  Within each line, text is normally rendered
 * in maximal runs.  A run is rendered by explicitly positioining and
 * then putting out one or more glyphs, with an implicit update after
 * each.  The default value for this property is OPT_LINE, which
 * instructs AXTE generate maximal runs for the line.  Other values are
 * OPT_WORD, which causes an additional position/run break at the start
 * of each word; and OPT_CHAR, which causes a position/run break on each
 * glyph.
 * </li>
 * <ul>
 * @exclude from published api.
 */

public class TextDrawInfo {
/**
 * Use the text display's runtime extent.  Note that some trailing
 * spaces may not be included in the runtime extent.
 */
	public static final int INVALID_DEFAULT_RUNTIME_EXTENT = 0;
/**
 * Use the declared size of the text object.  If text overflows the
 * declared size, it may not be drawn.
 */
	public static final int INVALID_DEFAULT_DECLARED_SIZE = 1;
/**
 * Use the computed size of the text object.  This will include any
 * overflowing text and trailing spaces.
 */
	public static final int INVALID_DEFAULT_AUGMENTED_SIZE = 2;

	private final GFXEnv moGfxEnv;
	private Rect moInvalid;
	private Rect moPage;
	private TextSelection mpoPrimary;
	private TextSelection mpoSecondary;
	private int meInvalidDefault;
	private boolean mbTruncate;
//	private int meOpt;

/**
 * Constructor.
 * All properties except graphic environment are set to defaults as
 * described above.
 * @param oGfxEnv - Graphic environment to use for rendering.
 */
	public TextDrawInfo (GFXEnv oGfxEnv) {
		moGfxEnv = oGfxEnv;
//		mpoPrimary = null;
//		mpoSecondary = null;
		meInvalidDefault = INVALID_DEFAULT_AUGMENTED_SIZE;
//		mbTruncate = false;
//		meOpt = TextPrefOpt.OPT_LINE;
	}

/**
 * Retrieve the graphic environment.
 * @return Graphic environment reference cached in this object.
 */
	public GFXEnv getGfxEnv () {
		return moGfxEnv;
	}

/**
 * Retrieve the invalidation rectangle.
 * @return Invalidation rectangle.	For more information, please see
 * above.
 */
	public Rect getInvalid () {
		return moInvalid;
	}

/**
 * Set the invalidation rectangle.
 * @param oInvalid - Invalidation rectangle.  For more information,
 * please see above.
 */
	public void setInvalid (Rect oInvalid) {
		moInvalid = oInvalid;
	}

/**
 * Retrieve the page rectangle.
 * @return Page rectangle.	For more information, please see above.
 */
	public Rect getPage () {
		return moPage;
	}

/**
 * Set the page rectangle.
 * @param oPage - Page rectangle.  For more information, please see
 * above.
 */
	public void setPage (Rect oPage) {
		moPage = oPage;
	}

/**
 * Retrieve the primary selection.
 * @return Primary selection.  For more information, please see above.
 */
	public TextSelection getPrimary () {
		return mpoPrimary;
	}

/**
 * Set the primary selection.
 * @param poPrimary - Primary selection.  For more information, please
 * see above.
 */
	public void setPrimary (TextSelection poPrimary) {
		mpoPrimary = poPrimary;
	}

/**
 * Retrieve the secondary selection.
 * @return Secondary selection.  For more information, please see above.
 */
	public TextSelection getSecondary () {
		return mpoSecondary;
	}

/**
 * Set the secondary selection.
 * @param poSecondary - Secondary selection.  For more information,
 * please see above.
 */
	public void setSecondary (TextSelection poSecondary) {
		mpoSecondary = poSecondary;
	}

/**
 * Retrieve the default invalidation rectangle setting.
 * @return Default invalidation rectangle setting.	For more
 * information, please see above.
 */
	public int getInvalidDefault () {
		return meInvalidDefault;
	}

/**
 * Set the default invalidation rectangle setting.
 * @param eInvalidDefault - Default invalidation rectangle setting.  For
 * more information, please see above.
 */
	public void setInvalidDefault (int eInvalidDefault) {
		meInvalidDefault = eInvalidDefault;
	}

/**
 * Retrieve the truncate flag.
 * @return Truncate flag.  For more information, please see above.
 */
	public boolean getTruncate () {
		return mbTruncate;
	}

/**
 * Set the truncate flag.
 * @param bTruncate - Truncate flag.  For more information, please see
 * above.
 */
	public void setTruncate (boolean bTruncate) {
		mbTruncate = bTruncate;
	}

/**
 * Retrieve the optimization level.
 * @return Optimization level.	For more information, please see above.
 */
//	public int getOpt () {
//		return (meOpt == TextPrefOpt.OPT_UNKNOWN) ? TextPrefOpt.OPT_LINE : meOpt;
//	}

/**
 * Set the optimization level.
 * @param eOpt - Optimization level.  For more information, please see
 * above.
 */
//	public void setOpt (int eOpt) {
//		meOpt = eOpt;
//	}
}
