package com.adobe.xfa.text;

import com.adobe.xfa.gfx.GFXEnv;
import com.adobe.xfa.ut.Storage;

/**
 * The client creates an instance of this class to manage the process of
 * generating text stream content from rendering and content.  It also
 * describes the rendering of text after user edits have been made.  The
 * client builds the layout object by adding one or more lines of runs.
 * A layout object instance corresponds to the content of one frame.
 * <p>
 * The rendering-based client loads text layout objects from its
 * persistence format according to its own needs, or on demand as
 * editing events cause AXTE to demand frame content.  Editing itself
 * occurs on a text stream object through the AXTE API.  When the user
 * directs AXTEs client to save data, the client would run through
 * loaded frames and generate new persistence data for them.
 * </p>
 * @exclude from published api.
 */

public class TextLayout {
	private final Storage<TextLayoutLine> moLines = new Storage<TextLayoutLine>();
	private boolean mbIsSuspect;

/**
 * Default constructor.
 * <p>
 * The layout is initially empty of lines.
 */
	public TextLayout () {
	}

/**
 * Copy Constructor.
 * <p>
 * Initializes this layout to be a copy of the given source layout.
 * Note that lines are copied by reference only.
 * @param oSource - Source layout to copy.
 */
	public TextLayout (TextLayout oSource) {
		copyFrom (oSource);
	}

/**
 * Reset the layout object to its initial state.
 * <p>
 * The client typically calls this method after one stream generation
 * operation, in order to perform a second one.  There is no harm in
 * calling Reset() repeatedly or calling it on a just-created layout
 * object.
 */
	public void reset () {
		moLines.setSize (0);
	}

/**
 * Add a line to the layout object.
 * @param poLine - Pointer to line object to add to the layout object.
 * Note that AXTE always caches its own copies of objects.	In other
 * words, a clone of the given line will be added to the layout object.
 * The client is free to delete the line identified by this parameter as
 * soon as the call returns.
 * @return Pointer to the copy actually added.	This is for caller
 * information only and must not be altered.
 */
	public TextLayoutLine addLine (TextLayoutLine poLine) {
		TextLayoutLine poCopy = new TextLayoutLine (poLine);
		moLines.add (poCopy);
		return poCopy;
	}

/**
 * Add a line to the layout object, by reference.
 * @param poLine - Pointer to line object to add to the layout object.
 * This is assumed to be a reference counted object.  The layout object
 * will add its own reference to the given line.
 */
	public void addLineRef (TextLayoutLine poLine) {
		moLines.add (poLine);
	}

/**
 * Return the number of lines in the layout.
 * @return Number of lines in the layout.
 */
	public int getLineCount () {
		return moLines.size();
	}

/**
 * Get one line from the layout object.
 * @param nIndex - Index of line to retrieve.
 * @return Pointer to the requested line
 */
	public TextLayoutLine getLine (int nIndex) {
		return moLines.get (nIndex);
	}

/**
 * Populate the given displayable stream and its display with
 * information from this layout object.
 * <p>
 * This call is rather destructive for the target stream, which loses
 * all its content and display.  In addition, any position and range
 * objects associated with the stream become detached from it.
 * </p>
 * <p>
 * This method first removes the old Unicode content and the stream's
 * display if there was one.  Upon return, the stream will contain both
 * Unicode content and a new display that effectively contains the run
 * content as added.  Once the operation is complete, the client then
 * uses the resulting stream for all interaction.  At the end of editing
 * the client extracts the modified rendering from the stream.
 * </p>
 * <p>
 * A successful operation implicitly ends with a call to the Reset()
 * method.
 * </p>
 * <p>
 * Note that this method generates a single-frame displayable stream.
 * For a multi-frame sparse stream, the client doesn't use this method.
 * Instead, it creates multiple layout instances, one for each frame
 * that needs to be loaded, typically under sparse stream call-back
 * control.
 * </p>
 * @param oTarget - Target stream to receive this layout objects
 * content.  This is an instance of the abstract TextDispStr class,
 * allowing the client to use an instance of any of the displayable
 * stream types.  Note that the XTG Font Service instance associated
 * with this stream must be the same as that associated with the fonts
 * in all the runs added to the rendering.
 * @param poGfxEnv - Abstract graphic environment for the stream/display
 * to use.
 * @return - Pointer to the newly created display for the stream.  Note
 * that this can be queried from the stream at any time.
 */
	public TextDisplay generateStream (TextDispStr oTarget /* , TextScroller poScroller */ , GFXEnv poGfxEnv) {
//		TextDisplay poDisplay = oTarget.loadLayout (this /* , poScroller */ , poGfxEnv);
//		Reset();
//		return poDisplay;
		return null;	// TODO:
	}

/**
 * Populate this layout object with the text rendering from the given
 * displayable stream.
 * <p>
 * Any previously existing content in this layout object is first
 * removed.  The resulting layout will contain original runs wherever
 * possible.  Where edits have occurred, new runs may be created.
 * @param oSource - Input displayable stream to extract the layout from.
 * If the stream doesnt have a display, one is created.
 * @param bAllowCharGlyphs - If false, all glyphs in the output runs are
 * specified by glyph IDs.	If true, AXTE will attempt to convert from
 * glyph IDs back to original characters where reasonable.
 * @param poGfxEnv - Optional abstract graphic environment.  The client
 * need not provide either scroller or environment if the stream already
 * has a display.  If the stream does not have a display, one will be
 * created with the scroller and/or environment provided.
 */
	public void format (TextDispStr oSource, boolean bAllowCharGlyphs /* , TextScroller poScroller */ , GFXEnv poGfxEnv) {
		reset();
//		oSource.format (this, bAllowCharGlyphs /* , poScroller */ , poGfxEnv);	// TODO:
	}

	public boolean isSuspect () {
		return mbIsSuspect;
	}

	public void setIsSuspect (boolean bIsSuspect) {
		mbIsSuspect = bIsSuspect;
	}

/**
 * Assignment operator.
 * <p>
 * Copies the content of the given source layout to this layout,
 * replacing all of its data members.  Note that lines are copied by
 * reference only.
 * @param oSource - Source layout to copy.
 */
	public void copyFrom (TextLayout oSource) {
		if (this != oSource) {
			reset();
			int nLines = oSource.moLines.size();
			moLines.setSize (nLines);
			for (int i = 0; i < nLines; i++) {
				moLines.set (i, oSource.getLine (i));
			}
			mbIsSuspect = oSource.mbIsSuspect;
		}
	}

/**
 * Create a correctly-typed copy of this layout object.
 * <p>
 * Clients don't generally extend this class, but if they do, they can
 * implement this method to ensure that a correctly-typed copy gets
 * created if a copy of the layout is made.  AXTE currently doesn't
 * create copies of layout objects, but may in the future.
 * @return Pointer to cloned layout object.
 */
	public TextLayout cloneLayout () {
		return new TextLayout (this);
	}
}
