package com.adobe.xfa.text;


/**
 * Base class for AXTE classes which must track legacy positioning or
 * spacing mode.
 * <p>
 * AXTE underwent a major rework between versions 6 and 7 to support
 * complex script languages.  During this time, a number of text layout
 * and positioning bugs were fixed, often inadvertently.  In addition,
 * more standard layout practices were adopted.
 * </p>
 * <p>
 * Consequently, version 7 did not lay out text the same as version 6. 
 * This proved unacceptable for some customers and AXTE was modified to
 * support a legacy positioning (version&nbsp;6) mode.  In this mode,
 * implements the many facets of the old behaviour: vertical and
 * horizontal positioning, line breaking rules and font metric fudging.
 * </p>
 * <p>
 * In spite of all the effort in version 7 to support correct and
 * standard layout practices, a bug crept in and was not detected soon
 * enough to provent customers from creating forms with the incorrect
 * behaviour.  Therefore, we have intentionally not correct that bug,
 * lest customers see yet another layout change.  The problem occurs
 * only when line spacing is overridden to something smaller than the
 * default and can lead to vertical alignment problems or the last line
 * of text being clipped.  While it mak not make sense to fix this
 * problem for forms customers, it leads to problems with other clients
 * of AXTE.
 * </p>
 * <p>
 * This class allows the derivative to be flagged by the client as
 * supporting one of the three behaviours described above.
 * </p>
 * @exclude from published api.
 */
public class TextLegacy {
/**
 * Legacy behaviour level enumeration.
 * The level can be discribed as one of the following:
 * <ul>
 * <li>
 * LEVEL_V6 - Use version 6 layout behaviour.
 * </li>
 * <li>
 * LEVEL_NORMAL (default) - Normal forms behaviour.  Layout follows
 * generally accepted behaviour, except for incorrect determination of
 * line height of the last line when a line spacing override is in
 * effect that reduces spacing smaller than the font size.
 * </li>
 * <li>
 * LEVEL_CORRECT_SPACING - Same as normal behaviour, except that it
 * corrects for the line spacing bug.
 * </li>
 * </ul>
 */
	public static final int LEVEL_V6 = -1;
	public static final int LEVEL_NORMAL = 0;
	public static final int LEVEL_CORRECT_SPACING = 1;

	private int meLevel;

/**
 * Default constructor.
 * The default legacy level is LEVEL_NORMAL;
 */
	public TextLegacy () {
		meLevel = LEVEL_NORMAL;
	}

/**
 * Query the current level.
 * @return The current spacing level.
 */
	public int getLegacyLevel () {
		return meLevel;
	}

/**
 * Set a new legacy spacing level.
 * @param eLevel - New level.
 */
	public void setLegacyLevel (int eLevel) {
		meLevel = eLevel;
	}

/**
 * Query whether version 6 layout is in effect.
 * @return True if the level is currently LEVEL_V6; false otherwise.
 */
	public boolean hasLegacyPositioning () {
		return meLevel == LEVEL_V6;
	}

/**
 * Query whether normal forms layout is in effect.
 * @return True if the level is currently LEVEL_NORMAL; false otherwise.
 */
	public boolean hasNormalPositioning () {
		return meLevel == LEVEL_NORMAL;
	}

/**
 * Query whether normal layout with correct spacing handling is in
 * effect.
 * @return True if the level is currently LEVEL_CORRECT_SPACING; false
 * otherwise.
 */
	public boolean hasCorrectPositioning () {
		return meLevel == LEVEL_CORRECT_SPACING;
	}
}
