package com.adobe.xfa.text;

import com.adobe.xfa.ut.UnitSpan;

/**
 * The text panel class represents the geometry and eventually layout of
 * a single panel.
 * <p>
 * The client creates all panel instances, either to pass to the sparse
 * stream API or as a result of its implementation of the sparse stream
 * event handling methods.  The client must populate its created panels
 * with geometry information.  A future implementation may allow the
 * client to set text attributes for a panel as well.
 * </p>
 * <p>
 * Once the client passes a panel instance to the sparse stream, AXTE
 * takes over and populates its layout.  The layout may come from a
 * client-supplied layout object, or it may be generated by AXTE as the
 * result of a reflow operation.
 * </p>
 * <p>
 * Panels are reference counted objects, allowing the client to retain
 * handles to objects that are managed by AXTE.
 * </p>
 * <p>
 * While panels have width and height geometry, it is up to the client
 * to decide how to map panels into its rendering space.  AXTE treats
 * the top-left corner of each panel as its origin.
 * </p>
 * @exclude from published api.
 */

public class TextPanel extends TextFrame {
	private final static UnitSpan DEFAULT_SIZE = new UnitSpan (UnitSpan.INCHES_72K, 72000);
	private UnitSpan moWidth;
	private UnitSpan moHeight;

/**
 * Default constructor.
 * <p>
 * Construct a panel with default dimensions of 1" by 1".
 */
	public TextPanel () {
		moWidth = DEFAULT_SIZE;
		moHeight = DEFAULT_SIZE;
	}

/**
 * Copy constructor.
 * @param oSource - Source panel to copy.
 */
	public TextPanel (TextPanel oSource) {
		moWidth = oSource.moWidth;
		moHeight = oSource.moHeight;
	}

/**
 * Construct a panel with given width and height.
 * @param oWidth - Width of the new panel.
 * @param oHeight - Height of the new panel.
 */
	public TextPanel (UnitSpan oWidth, UnitSpan oHeight) {
		moWidth = oWidth;
		moHeight = oHeight;
	}

/**
 * Set the width of this panel.
 * <p>
 * If the width changes for a panel in a sparse stream, it causes reflow
 * of text from that panel on.
 * @param oWidth - New width for the panel.
 */
	public void setWidth (UnitSpan oWidth) {
		if (oWidth == moWidth) {
			return;
		}
		moWidth = oWidth;
		reflowFromHere();
	}

/**
 * Return the width of the panel.
 * @return Width of the panel.
 */
	public UnitSpan getWidth () {
		return moWidth;
	}

/**
 * Set the height of this panel.
 * <p>
 * If the height changes for a panel in a sparse stream, it causes reflow
 * of text from that panel on.
 * @param oHeight - New height for the panel.
 */
	public void setHeight (UnitSpan oHeight) {
		if (oHeight == moHeight) {
			return;
		}
		moHeight = oHeight;
		reflowFromHere();
	}

/**
 * Return the height of the panel.
 * @return Height of the panel.
 */
	public UnitSpan getHeight () {
		return moHeight;
	}

/**
 * Set the dimensions of this panel.
 * <p>
 * This allows the caller to change both settings in a single call.  If
 * either dimension changes for a panel in a sparse stream, it causes
 * reflow of text from that panel on.
 * @param oWidth - New width for the panel.
 * @param oHeight - New Height for the panel.
 */
	public void setDimensions (UnitSpan oWidth, UnitSpan oHeight) {
		if ((oWidth == moWidth) && (oHeight == moHeight)) {
			return;
		}
		moWidth = oWidth;
		moHeight = oHeight;
		reflowFromHere();
	}

/**
 * Assignment operator.
 * @param oSource - Source panel to copy.
 */
	public void copyFrom (TextPanel oSource) {
		moWidth = oSource.moWidth;
		moHeight = oSource.moHeight;
	}

// Inherited from class TextFrame.
	public UnitSpan minWidth () {
		return moWidth;
	}

	public UnitSpan minHeight () {
		return moHeight;
	}

	public UnitSpan maxWidth () {
		return moWidth;
	}

	public UnitSpan maxHeight () {
		return moHeight;
	}

	public TextFrame cloneFrame () {
		return new TextPanel (this);
	}
}
