package com.adobe.xfa.text;

import com.adobe.xfa.ut.UnitSpan;

/**
 * A text region is a displayable stream (see class TextDispStr) that
 * has both minimum and maximum height and width.  The region's
 * graphical footprint will occupy at least its minimum height and
 * width.  Text content may caus it to grow to its maximums, but never
 * larger.	If the text exceeds the space available, scrolling may be
 * required or displayed content may be truncated.	The application may
 * change the minimum and maximum dimensions through method calls,
 * causing the region to be redisplayed if a text display has already
 * been created.
 * @exclude from published api
 */

public class TextRegion extends TextDispStr {

/**
 * Default constructor.
 * <p>
 * The text stream contains no content and has no pool/mapping
 * assocotiation.  The minimum and maximum width and height are all
 * defaulted to one inch.
 */
	public TextRegion () {
		setTraditionalFrame (new RegionFrame());
	}

/**
 * Copy constructor with graphics source information.
 * <p>
 * Copy all stream content from the source stream, using the optional
 * graphic attribute pool.	Also copies the minimum and maximum width
 * and height.	The display is not automatically created.
 * @param oSource - Source text stream to copy content from.
 * @param poPool - Graphic attribute pool to use.
 */
//	public TextRegion (TextRegion oSource, GFXAttrPool poPool) {
//		TextDispStr = oSource, poPool;
//		setTraditionalFrame (new RegionFrame (oSource.getRegionFrame()));
//	}

/**
 * Construct text block from text stream and graphics source information.
 * <p>
 * Copy all stream content from the source stream, using the optional
 * graphic attribute pool.	The display is not automatically created.
 * @param oSource - Source text stream to copy content from.
 * @param poPool - Graphic attribute pool to use.
 */
//	public TextRegion (TextStream oSource, GFXAttrPool poPool) {
//		TextDispStr = oSource, poPool;
//		setTraditionalFrame (new RegionFrame);
//	}

/**
 * Constructor with source text string.
 * <p>
 * Create a text stream whose initial content is copied from the given
 * string.	The minimum and maximum width and height are all
 * defaulted to one inch.	The text stream initially has no attribute
 * pool association.	The display is not automatically created.
 * @param sSource - String whose contents are to be copied to the text
 * stream.
 */
	public TextRegion (String sSource) {
		super (sSource);
		setTraditionalFrame (new RegionFrame());
	}

/**
 * Return the minimum width of the text region.
 * <p>
 * This provides the same information as the virtual MinWidth() method
 * in the base class, TextDispStr.
 * @return Minimum width of the text region.  For information on special
 * values, please see the base class (TextDispStr) documentation.
 */
	public UnitSpan getMinWidth () {
		return minWidth();
	}

/**
 * Set the minimum width of the text region.
 * <p>
 * This method allows the caller to change the minimum width of the text
 * region, possibly triggering a redisplay.
 * @param oNewWidth - New minimum width of the text region.  For
 * information on special values, please see the base class
 * (TextDispStr) documentation.
 */
	public void setMinWidth (UnitSpan oNewWidth) {
		if (oNewWidth.equals (minWidth())) {
			return;
		}

		getRegionFrame().setMinWidth (oNewWidth);
		if (! enforceJustifyWidth()) {
			updateDisplay (true);
		}
	}

/**
 * Return the minimum height of the text region.
 * <p>
 * This provides the same information as the virtual Minheight() method
 * in the base class, TextDispStr.
 * @return Minimum height of the text region.  For information on
 * special values, please see the base class (TextDispStr)
 * documentation.
 */
	public UnitSpan getMinHeight () {
		return minHeight();
	}

/**
 * Set the minimum height of the text region.
 * <p>
 * This method allows the caller to change the minimum height of the
 * text region, possibly triggering a redisplay.
 * @param oNewHeight - New minimum height of the text region.  For
 * information on special values, please see the base class
 * (TextDispStr) documentation.
 */
	public void setMinHeight (UnitSpan oNewHeight) {
		if (oNewHeight.equals (minHeight())) {
			return;
		}

		getRegionFrame().setMinHeight (oNewHeight);
		if (! enforceJustifyHeight()) {
			updateDisplay (true);
		}
	}

/**
 * Return the maximum width of the text region.
 * <p>
 * This provides the same information as the virtual MaxWidth() method
 * in the base class, TextDispStr.
 * @return Maximum width of the text region.  For information on special
 * values, please see the base class (TextDispStr) documentation.
 */
	public UnitSpan getMaxWidth () {
		return maxWidth();
	}

/**
 * Set the maximum width of the text region.
 * <p>
 * This method allows the caller to change the maximum width of the text
 * region, possibly triggering a redisplay.
 * @param oNewWidth - New maximum width of the text region.  For
 * information on special values, please see the base class
 * (TextDispStr) documentation.
 */
	public void setMaxWidth (UnitSpan oNewWidth) {
		if (oNewWidth.equals (maxWidth())) {
			return;
		}

// Optimization for XFA behaviour: XFA often creates objects that
// supports growth, but then "locks them down" once it knows their size.
// Rather than reformatting on lock down, we detect that condition here
// and possibly avoid an expensive operation.  The strategy is to
// reformat if the max width increases or it becomes smaller than the
// currently formatted width.
		boolean bUpdateDisplay = false;
		if (oNewWidth.value() < 0) {
			bUpdateDisplay = true; // gone to unlimited (i.e., larger)
		} else if ((! unlimitedWidth()) && (oNewWidth.gt (maxWidth()))) {
			bUpdateDisplay = true; // got larger
		} else if (oNewWidth.lt (extent().width())) {
			bUpdateDisplay = true; // too small now
		}

		getRegionFrame().setMaxWidth (oNewWidth);

		if (bUpdateDisplay) {
			updateDisplay();
		}
	}

/**
 * Return the maximum height of the text region.
 * <p>
 * This provides the same information as the virtual Maxheight() method
 * in the base class, TextDispStr.
 * @return Maximum height of the text region.  For information on
 * special values, please see the base class (TextDispStr)
 * documentation.
 */
	public UnitSpan getMaxHeight () {
		return maxHeight();
	}

/**
 * Set the maximum height of the text region.
 * <p>
 * This method allows the caller to change the maximum height of the
 * text region, possibly triggering a redisplay.
 * @param oNewHeight - New maximum height of the text region.  For
 * information on special values, please see the base class
 * (TextDispStr) documentation.
 */
	public void setMaxHeight (UnitSpan oNewHeight) {
		if (oNewHeight.equals (maxHeight())) {
			return;
		}

// Optimization for XFA behaviour: XFA often creates objects that
// supports growth, but then "locks them down" once it knows their size.
// Rather than reformatting on lock down, we detect that condition here
// and possibly avoid an expensive operation.  The strategy is to
// reformat if the max height increases or it becomes smaller than the
// currently formatted height.
		boolean bUpdateDisplay = false;
		if (oNewHeight.value() < 0) {
			bUpdateDisplay = true; // gone to unlimited (i.e., larger)
		} else if ((! unlimitedHeight()) && (oNewHeight.gt (maxHeight()))) {
			bUpdateDisplay = true; // got larger
		} else if (oNewHeight.lt (extent().height())) {
			bUpdateDisplay = true; // too small now
		}

		getRegionFrame().setMaxHeight (oNewHeight);

		if (bUpdateDisplay) {
			updateDisplay (true);
		}
	}

/**
 * Set the justfiy height of the text region.
 * <p>
 * This method sets the height to use during justification calculations
 */
	public void setJustifyHeight (UnitSpan oJustifyHeight) {
		if (enforceJustifyHeight() && (oJustifyHeight.equals (justifyHeight()))) {
			return;
		}

		getRegionFrame().setAlignmentHeight (oJustifyHeight);
		updateDisplay (true);
	}

/**
 * Set the justfiy width of the text region.
 * <p>
 * This method sets the height to use during justification calculations
 */
	public void setJustifyWidth (UnitSpan oJustifyWidth) {
		if (enforceJustifyWidth() && (oJustifyWidth.equals (justifyWidth()))) {
			return;
		}

		getRegionFrame().setAlignmentWidth (oJustifyWidth);
		updateDisplay (true);
	}

/**
 * Set the justfiy extents of the text region.
 * <p>
 * This method sets the height to use during justification calculations
 */
	public void setJustifyExtents (UnitSpan oJustifyWidth, UnitSpan oJustifyHeight) {
		if (enforceJustifyHeight() && oJustifyHeight.equals (justifyHeight())
		 && enforceJustifyWidth() && oJustifyWidth.equals (justifyWidth())) {
			return;
		}

		getRegionFrame().setAlignmentWidth (oJustifyWidth);
		getRegionFrame().setAlignmentHeight (oJustifyHeight);
		updateDisplay (true);
	}

/**
 * Query whether the stream enforces text region size at data entry time.
 * <p>
 * @return TRUE if the graphical size is being enforced; FALSE if not.
 * Note that the constructor defaults this to TRUE.
 */
	public boolean enforceSize () {
		return getRegionFrame().testFitSize();
	}

/**
 * Change the value of the EnforceSize flag.
 * <p>
 * @param bNewEnforce - TRUE causes the text region to respect its
 * maximum size; FALSE allows the text content to exceed the maximum
 * size of the region.	Note that the constructor defaults this to TRUE.
 */
	public void enforceSize (boolean bNewEnforce) {
		getRegionFrame().setTestFitSize (bNewEnforce);
	}

/**
 * Assign this text region's content from the given text region.
 * <p>
 * Replace this stream's content with a copy of the content of the given
 * stream.	The graphic source information is <b>not</b> copied.  In
 * other words, fonts will be re-mapped in this stream's font service
 * and attributes will be re-pooled in any attribute pool associated
 * with this stream.  The minimum and maximum width and height of the
 * source stream are also copied.	The display pointer is not copied,
 * as each stream has its own display; nor is a display automatically
 * created.
 * @param oSource - Stream containing source content to copy.
 */
	public void copyFrom (TextRegion oSource) {
		if (this != oSource) {
			super.copyFrom (oSource);
			getRegionFrame().copyFrom (oSource.getRegionFrame());
		}
	}

/**
 * Assign this text region's content from the given base class stream.
 * <p>
 * Replace this stream's content with a copy of the content of the given
 * stream.	The graphic source information is <b>not</b> copied.  In
 * other words, fonts will be re-mapped in this stream's font service
 * and attributes will be re-pooled in any attribute pool associated
 * with this stream.
 * @param oSource - Stream containing source content to copy.
 */
// Javaport: not needed if method merely calls super.
//	public void copyFrom (TextStream oSource) {
//		super.copyFrom (oSource);
//	}

/**
 * Compare text regions for equality.
 * <p>
 * Compare this stream against the one passed on the parameter oCompare
 * for equality.  The graphics sources of the streams are not compared.
 * To be equal, the streams' content must match in all aspects: raw
 * text, attributes, embedded field content, and so on.  The display
 * does not participate in the comparison.	In addition, the text
 * regions' minimum and maximum width and height must match.
 * @param object - Stream to compare against
 * @return TRUE if the streams are equal; FALSE otherwise.
 */
	public boolean equals (Object object) {		
		
		if (object == this)
			return true;
		
		if (!super.equals(object))
			return false;
		
		TextRegion cmp = (TextRegion) object;
		
		return getRegionFrame().equals (cmp.getRegionFrame());
	}

	public int hashCode() {
		int hash = 61;
		hash = (hash * 31) ^ super.hashCode();
		hash = (hash * 31) ^ getRegionFrame().hashCode();
		return hash;
	}

/**
 * Compare text streams for inequality.
 * <p>
 * Compare this stream against the one passed on the parameter oCompare
 * for inequality.	The graphics sources of the streams are not
 * compared.  This is the exact opposite of the equality comparison.
 * The display does not participate in the comparison.
 * @param oCompare - Stream to compare against
 * @return TRUE if the streams are unequal; FALSE otherwise.
 */
	public boolean notEqual (TextRegion oCompare) {
		return ! equals (oCompare);
	}

	private RegionFrame getRegionFrame () {
		return (RegionFrame) (getTraditionalFrame());
	}
}

/*
 * RegionFrame - Class to represent geometry of a text region.
 */

class RegionFrame extends TraditionalFrame {
	private UnitSpan moMinWidth;
	private UnitSpan moMinHeight;
	private UnitSpan moMaxWidth;
	private UnitSpan moMaxHeight;

	private UnitSpan moAlignmentWidth;
	private UnitSpan moAlignmentHeight;

	private boolean mbTestFitSize;

	private final static UnitSpan DEFAULT_SIZE = new UnitSpan (UnitSpan.INCHES_72K, 72000);

	RegionFrame () {
		mbTestFitSize = true;
	}

	RegionFrame (RegionFrame oSource) {
		copyFrom (oSource);
	}

	void setMinWidth (UnitSpan oMinWidth) {
		moMinWidth = oMinWidth;
	}

	void setMinHeight (UnitSpan oMinHeight) {
		moMinHeight = oMinHeight;
	}

	void setMaxWidth (UnitSpan oMaxWidth) {
		moMaxWidth = oMaxWidth;
	}

	void setMaxHeight (UnitSpan oMaxHeight) {
		moMaxHeight = oMaxHeight;
	}

	void setAlignmentWidth (UnitSpan oAlignmentWidth) {
		moAlignmentWidth = oAlignmentWidth;
	}

	void setAlignmentHeight (UnitSpan oAlignmentHeight) {
		moAlignmentHeight = oAlignmentHeight;
	}

	void setTestFitSize (boolean bTestFitSize) {
		mbTestFitSize = bTestFitSize;
	}

	public UnitSpan minWidth () {
		return (moMinWidth == null) ? DEFAULT_SIZE : moMinWidth;
	}

	public UnitSpan minHeight () {
		return (moMinHeight == null) ? DEFAULT_SIZE : moMinHeight;
	}

	public UnitSpan maxWidth () {
		return (moMaxWidth == null) ? DEFAULT_SIZE : moMaxWidth;
	}

	public UnitSpan maxHeight () {
		return (moMaxHeight == null) ? DEFAULT_SIZE : moMaxHeight;
	}

	UnitSpan alignmentWidth () {
		return moAlignmentWidth;
	}

	boolean enforceAlignmentWidth () {
		return moAlignmentWidth != null;
	}

	UnitSpan alignmentHeight () {
		return moAlignmentHeight;
	}

	boolean enforceAlignmentHeight () {
		return moAlignmentHeight != null;
	}

	boolean testFitSize () {
		return mbTestFitSize;
	}

	boolean enforceDisplayExtent () {
		return true;
	}

	void copyFrom (RegionFrame oSource) {
		if (this != oSource) {
			moMinWidth = oSource.moMinWidth;
			moMinHeight = oSource.moMinHeight;
			moMaxWidth = oSource.moMaxWidth;
			moMaxHeight = oSource.moMaxHeight;

			moAlignmentWidth = oSource.moAlignmentWidth;
			moAlignmentHeight = oSource.moAlignmentHeight;
			mbTestFitSize = oSource.mbTestFitSize;
		}
	}

	boolean equals (RegionFrame oCompare) {
		return compareUnits (moMinWidth, oCompare.moMinWidth)
			&& compareUnits (moMinHeight, oCompare.moMinHeight)
			&& compareUnits (moMaxWidth, oCompare.moMaxWidth)
			&& compareUnits (moMaxHeight, oCompare.moMaxHeight)
			&& compareUnits (moAlignmentWidth, oCompare.moAlignmentWidth)
			&& compareUnits (moAlignmentHeight, oCompare.moAlignmentHeight);
	}

	private static boolean compareUnits (UnitSpan a, UnitSpan b) {
		if (a == b) {
			return true;
		}
		if ((a == null) || (b == null)) {
			return false;
		}
		return a.equals (b);
	}
}
