/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2006 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */
package com.adobe.xfa.ut.trace;

import java.util.Collection;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;


/**
 * store all the instances of Trace when the are constructed
 * 
 * @exclude from published api.
 */
final class GlobalTraceStore {
	
	private final static GlobalTraceStore mInstance = new GlobalTraceStore();
	
	private final ConcurrentMap<String, Trace> mTraceAreas = new ConcurrentHashMap<String, Trace>();
	private volatile boolean mbEnabled;
	
	// add a new trace section
	void addTrace(Trace trace) {
		// ensure unique names are used -- but allow an explicitly-constructed
		// Trace to replace a place holder Trace.
		
		Trace existingTrace = mTraceAreas.putIfAbsent(trace.getName(), trace);
		
		if (existingTrace != null && existingTrace.isPlaceHolder() && !trace.isPlaceHolder()) {
			existingTrace.setHelpId(trace.getHelpId());
			existingTrace.setPlaceHolder(false);
			return;
		}
	}

	// remove a trace section
	void removeTrace(Trace trace) {
		mTraceAreas.remove(trace.getName());
	}

	// get the global variable
	static GlobalTraceStore getStore() {
		return mInstance;
	}

	// Get a Collection of all the trace sections
	Collection<Trace> getTraceSections() {
		return mTraceAreas.values();
	}

	// verify that a named trace section exists.  If it doesn't, add it.
	void ensureTraceExists(String sName) {
		if (!mTraceAreas.containsKey(sName)) {
			addTrace(new Trace(sName));	// add a place holder Trace
		}
	}

	void enable() {
		mbEnabled = true;
	}
	
	boolean isEnabled() {
		return mbEnabled;
	}
	
	// Prevent instantiation
	protected GlobalTraceStore() {
	}
}