/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2008 Adobe Systems Incorporated
 * All Rights Reserved
 *
 * Notice: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */

package com.adobe.xfa.wspolicy;

import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;

import com.adobe.xfa.AppModel;
import com.adobe.xfa.Document;
import com.adobe.xfa.Element;
import com.adobe.xfa.Node;
import com.adobe.xfa.dom.DOM;
import com.adobe.xfa.dom.NamespaceContextImpl;
import com.adobe.xfa.ut.ExFull;
import com.adobe.xfa.ut.FindBugsSuppress;
import com.adobe.xfa.ut.ObjectHolder;
import com.adobe.xfa.ut.ResId;
import com.adobe.xfa.ut.StringUtils;

/**
 * @exclude from published api.
 */
public class Policy extends Element {
	// WS-Policy
	public static final String aWSP_NAMESPACE_URI = "http://schemas.xmlsoap.org/ws/2004/09/policy";
	public static final String aWSP_NAMESPACE_PREFIX = "wsp";
	public static final String aWSP_POLICYURIS = "PolicyURIs";
	public static final String aWSP_POLICY = "Policy";
	public static final String aWSP_POLICYREFERENCE = "PolicyReference";
	public static final String aWSP_URIATTR = "URI";
	public static final String aWSP_ALL = "All";
	public static final String aWSP_EXACTLYONE = "ExactlyOne";

// WS-SecurityPolicy
	public static final String aSP_NAMESPACE_URI = "http://schemas.xmlsoap.org/ws/2005/07/securitypolicy";
	public static final String aSP_NAMESPACE_PREFIX = "sp";
	public static final String aSP_TRANSPORTBINDING = "TransportBinding";
	public static final String aSP_TRANSPORTTOKEN = "TransportToken";
	public static final String aSP_HTTPSTOKEN = "HttpsToken";
	public static final String aSP_HTTPBASICAUTHENTICATION = "HttpBasicAuthentication";
	public static final String aSP_HTTPDIGESTAUTHENTICATION = "HttpDigestAuthentication";
	public static final String aSP_REQUIRECLIENTCERTIFICATE = "RequireClientCertificate";
	public static final String aSP_USERNAMETOKEN = "UsernameToken";
	public static final String aSP_HASHPASSWORD = "HashPassword";
	public static final String aSP_X509TOKEN = "X509Token";
	public static final String aSP_SUPPORTINGTOKENS = "SupportingTokens";

// WS-SecurityUtility
	public static final String aWSU_NAMESPACE_URI = "http://docs.oasis-open.org/wss/2004/01/oasis-200401-wsswssecurity-utility-1.0.xsd";
	public static final String aWSU_NAMESPACE_PREFIX = "wsu";
	public static final String aWSU_ID = "Id";

	public static final int WSP_PRIMITIVE_ASSERTION = 0;
	public static final int WSP_ALL_ASSERTION = 1;
	public static final int WSP_EXACTLY_ONE_ASSERTION = 2;

	private static XPathFactory mXPathFactory;

	private Element moDomNode;
	private int meAssertionType;
	private org.w3c.dom.Document mShadowDoc;
	private org.w3c.dom.Node mShadowNode;

	public final Policy getFirstPolicyChild () {
		return getNextPolicySibling (getFirstXMLChild());
	}

	public final Policy getNextPolicySibling () {
		return getNextPolicySibling (getNextXMLSibling());
	}

// Construct an effective policy for a host node.
// Because of the Byzantine reference and merging rules of WS-Policy,
// this will construct a single, completely resolved tree in its
// own, private document.
	public static CompoundAssertion resolvePolicy (Element oHostNode) {
		AppModel appModel = new AppModel (null);
		Document oDestDoc = appModel.getDocument();
		// Register wsu:Id as a valid ID attribute
		oDestDoc.declareGlobalXMLId(aWSU_NAMESPACE_URI, aWSU_ID);

		CompoundAssertion poMerged = null;

		String sPolicyURIs = null;
		int attrIndex = oHostNode.findAttr (aWSP_NAMESPACE_URI, aWSP_POLICYURIS);
		if (attrIndex >= 0) {
			sPolicyURIs = oHostNode.getAttrVal (attrIndex);
		}
		if (! StringUtils.isEmpty (sPolicyURIs)) {
			StringBuilder uriTokenizer = new StringBuilder (sPolicyURIs);
			String sURI = StringUtils.parseToken (uriTokenizer);
			while (sURI != null) {
				CompoundAssertion poPolicy = loadPolicyByReference (oDestDoc, oHostNode.getOwnerDocument(), sURI);
				if ((poPolicy != null) && (poMerged != null)) {
					poMerged = mergePolicies (poMerged, poPolicy);
				} else {
					poMerged = poPolicy;
				}
				sURI = StringUtils.parseToken (uriTokenizer);
			}
		}

		for (Node oChild = oHostNode.getFirstXMLChild(); oChild != null; oChild = oChild.getNextXMLSibling()) {
			if (oChild instanceof Element) {
				Element e = (Element) oChild;
				if (isPolicyOrRef (e)) {
					CompoundAssertion poPolicy = (CompoundAssertion) loadPolicy (oDestDoc, e);
					if ((poPolicy != null) && (poMerged != null)) {
						poMerged = mergePolicies (poMerged, poPolicy);
					} else {
						poMerged = poPolicy;
					}
				}
			}
		}

		if (poMerged != null) {
			Policy policy = poMerged;
			oDestDoc.appendChild (policy.moDomNode);
		}

		return poMerged;
	}

// Delete all policy references from a host node.  This includes
// those referenced by a wsp:PolicyURIs attribute and by children
// <wsp:Policy> and <wsp:PolicyReference> elements.
	public static void deletePolicyInfo (Element oHostNode) {
		String policyURIsAttr = null;
		int attrIndex = oHostNode.findAttr (aWSP_NAMESPACE_URI, aWSP_POLICYURIS);
		if (attrIndex >= 0) {
			policyURIsAttr = oHostNode.getAttrVal (attrIndex);
			if (! StringUtils.isEmpty (policyURIsAttr)) {
				oHostNode.removeAttr (attrIndex);
			}
		}

		Node oChild = oHostNode.getFirstXMLChild();
		while (oChild != null) {
			if (oChild instanceof Element) {
				if (isPolicyOrRef ((Element) oChild)) {
					oHostNode.removeChild (oChild);
				}
			}
			oChild = oChild.getNextXMLSibling();
		}
	}

// Load a policy node within a WS-Policy framework.
// This can work either within the same document as the source,
// or in a different document.
	static Policy loadPolicy (Document oDestDoc, Element oSrc) {
// Note: LoadPolicy deals with policy elements *within* a policy framework.  To
// determine an effective policy in place from a *host* node, use ResolvePolicy.
// Note: LoadPolicy deals with policy elements *within* a policy framework.  To
// determine an effective policy in place from a *host* node, use ResolvePolicy.
		String aNameSpaceURI = oSrc.getNS();
		String aLocalName = oSrc.getLocalName();
		Policy result = null;
		if ((aNameSpaceURI == aWSP_NAMESPACE_URI) && (aLocalName == aWSP_POLICYREFERENCE)) {
			int attrIndex = oSrc.findAttr (aWSP_NAMESPACE_URI, aWSP_URIATTR);
			if (attrIndex >= 0) {
				String uriAttr = oSrc.getAttrVal (attrIndex);
				if (! StringUtils.isEmpty (uriAttr)) {
					result = loadPolicyByReference (oDestDoc, oSrc.getOwnerDocument(), uriAttr);
				}
			}
		} else if (aNameSpaceURI == aWSP_NAMESPACE_URI && aLocalName == aWSP_POLICY) {
			result = new CompoundAssertion (oDestDoc, oSrc, WSP_ALL_ASSERTION);
		} else if (aNameSpaceURI == aWSP_NAMESPACE_URI && aLocalName == aWSP_POLICYREFERENCE) {
			result = new CompoundAssertion (oDestDoc, oSrc, WSP_ALL_ASSERTION);
		} else if (aNameSpaceURI == aWSP_NAMESPACE_URI && aLocalName == aWSP_ALL) {
			result = new CompoundAssertion (oDestDoc, oSrc, WSP_ALL_ASSERTION);
		} else if (aNameSpaceURI == aWSP_NAMESPACE_URI && aLocalName == aWSP_EXACTLYONE) {
			result = new CompoundAssertion (oDestDoc, oSrc, WSP_EXACTLY_ONE_ASSERTION);
		} else {
			result = new PrimitiveAssertion (oDestDoc, oSrc);
		}
		return result;
	}

// Create a new, unanchored, policy.
// Like LoadPolicy(), this works in a document supplied by the caller.
	public static CompoundAssertion newPolicy (Document oDestDoc, ObjectHolder<Element> oNewDomNodeOut) {
		CompoundAssertion poPolicy = new CompoundAssertion (oDestDoc, aWSP_NAMESPACE_URI, aWSP_POLICY);
		if (oNewDomNodeOut != null)
			oNewDomNodeOut.value = poPolicy.getDomNode();
		return poPolicy;
	}

	public int getAssertionType () {
		return meAssertionType;
	}

	private static CompoundAssertion loadPolicyByReference (Document oDestDoc, Document oSrcDoc, String sURI) {
		if ((sURI.length() > 0) && (sURI.charAt(0) == '#')) {
			String sID = sURI.substring (1);
			Element oSrc = oSrcDoc.getElementByXMLId (sID);
			if (oSrc != null) {
				return new CompoundAssertion (oDestDoc, oSrc, WSP_ALL_ASSERTION);
			}
		}
		else {
// JEY TODO: what about external references???
		}

		return null;
	}

// Merge policies:
	private static CompoundAssertion mergePolicies (Policy poLeft, Policy poRight) {
// Change component policies' <wsp:Policy> elements to <wsp:All>
//		assert (poLeft.getLocalName() == aWSP_POLICY && poRight.getLocalName() == aWSP_POLICY);
		String sPrefix = poLeft.getDomNode().getPrefix();
		String sLocal = aWSP_ALL;
		String aQName = (sPrefix + ':' + sLocal).intern();
		poLeft.getDomNode().setQName (aQName);
		poRight.getDomNode().setQName (aQName);

// Wrap in a new <wsp:Policy> element
		Document oDestDoc = poLeft.getDomNode().getOwnerDocument();
		CompoundAssertion poMerged = new CompoundAssertion (oDestDoc, aWSP_NAMESPACE_URI, aWSP_POLICY);
		poMerged.addPolicy (poLeft);
		poMerged.addPolicy (poRight);

		return poMerged;
	}

	public String getProperty (String sXPath) {
		org.w3c.dom.Node shadowNode = getShadowNode();
		XPath xpath = getXPathFactory().newXPath();
		
		NamespaceContextImpl namespaceContext = new NamespaceContextImpl(shadowNode);
		namespaceContext.addNamespaceMapping (aWSP_NAMESPACE_PREFIX, aWSP_NAMESPACE_URI);
		namespaceContext.addNamespaceMapping (aSP_NAMESPACE_PREFIX, aSP_NAMESPACE_URI);
		namespaceContext.addNamespaceMapping (aWSU_NAMESPACE_PREFIX, aWSU_NAMESPACE_URI);
		xpath.setNamespaceContext(namespaceContext);
		
		try {
			return xpath.evaluate(sXPath, shadowNode);
		} catch (XPathExpressionException e) {
			throw new ExFull(ResId.XPATH_ERROR, e.getMessage());
		}
	}

	public Policy getParentPolicy () {
		Element parent = getXMLParent();
		if (parent == null) {
			return null;
		}
		assert (parent instanceof Policy);
		return (Policy) parent;
	}

	void addPolicy (Policy poPolicy) {
		moDomNode.appendChild (poPolicy.getDomNode());
		appendChild (poPolicy);
	}

	private org.w3c.dom.Node getShadowNode () {
		if (mShadowNode == null) {
			org.w3c.dom.Document shadowDoc = getShadowDocument (moDomNode);
			mShadowNode = DOM.attach (shadowDoc, moDomNode);
		}
		return mShadowNode;
	}
	
	private static XPathFactory getXPathFactory() {
		if (mXPathFactory == null)
			mXPathFactory = XPathFactory.newInstance();
		
		return mXPathFactory;
	}

	final static boolean isPolicyOrRef (Element e) {
		String localName = e.getLocalName();
		return ((e.getNS() == aWSP_NAMESPACE_URI) && ((localName == aWSP_POLICY) || (localName == aWSP_POLICYREFERENCE)));
	}

	protected Policy (Document oDestDoc, Element oSrcNode) {
		init (oDestDoc, (Element) oDestDoc.importNode (oSrcNode, false));
	}

	@FindBugsSuppress(code="ES")
	protected Policy (Document oDestDoc, String aNameSpaceURI, String aLocalName) {
		String sPrefix = "";
		if (aNameSpaceURI == aWSP_NAMESPACE_URI) {
			sPrefix = aWSP_NAMESPACE_PREFIX + ":";
		} else if (aNameSpaceURI == aSP_NAMESPACE_URI) {
			sPrefix = aSP_NAMESPACE_PREFIX + ":";
		}
		init (oDestDoc, oDestDoc.createElementNS (aNameSpaceURI, sPrefix + aLocalName, null));
	}

	protected final void setAssertionType (int eAssertionType) {
		meAssertionType = eAssertionType;
	}

	protected final Element getDomNode () {
		return moDomNode;
	}

	private void init (Document doc, Element clonedNode) {
		moDomNode = clonedNode;
		moDomNode.setModel (doc.getModel());
		setNameSpaceURI (moDomNode.getNS(), true, false, false);
		setLocalName (moDomNode.getLocalName());
		meAssertionType = WSP_PRIMITIVE_ASSERTION;
	}

	private static Policy getNextPolicySibling (Node node) {
		while (node != null) {
			if (node instanceof Policy) {
				return (Policy) node;
			}
			node = node.getNextXMLSibling();
		}
		return null;
	}

	private org.w3c.dom.Document getShadowDocument (Node domNode) {
		if (mShadowDoc == null) {
			Policy parentPolicy = getParentPolicy();
			if (parentPolicy != null) {
				mShadowDoc = parentPolicy.getShadowDocument (domNode);
			} else {
				org.w3c.dom.Node shadowNode = DOM.attach (moDomNode);
				mShadowDoc = shadowNode.getOwnerDocument();
			}
		}
		return mShadowDoc;
	}
}
