// =================================================================================================
// ADOBE SYSTEMS INCORPORATED
// Copyright 2012 Adobe Systems Incorporated
// All Rights Reserved
//
// NOTICE:  Adobe permits you to use, modify, and distribute this file in accordance with the terms
// of the Adobe license agreement accompanying it.
// =================================================================================================
package com.adobe.xmp.core.parser;

import java.io.InputStream;
import java.util.Map;

import com.adobe.xmp.core.XMPException;
import com.adobe.xmp.core.XMPMetadata;


/**
 * Interface for all kinds of parsers that create an XMP data model.
 */
public interface XMPParser 
{
	/**
	 * Parse XMP in RDF/XML form from an InputStream. The input for parsing may be any valid Unicode
	 * encoding, it is recognized automatically.
	 * The client has to open/close the stream.
	 * The parser will simply ignore sub-trees that contain invalid XMP data. 
	 * It will only throw on serious errors.
	 *
	 * @param input the stream contains an XMP packet serialized as RDF/XML
	 * @param parseContext a map to provide options for the parser, 
	 * 		  or return additional information other than the XMP data model;
	 * 		  can be null if not needed
	 * @return an XMPMetadata object created from the input
	 * @throws XMPException If the data is invalid XML, invalid XMP or contains invalid data
	 */
	XMPMetadata parse( InputStream input, Map<String, Object> parseContext ) throws XMPException;
	
	/**
	 * Parse XMP in RDF/XML form from a String
	 * The parser will simply ignore sub-trees that contain invalid XMP data. 
	 * It will only throw on serious errors.
	 * @param input String that contains an XMP packet serialized as RDF/XML
	 * @param parseContext a map to provide options for the parser, 
	 * 		  or return additional information other than the XMP data model;
	 * 		  can be null if not needed
	 * @return an XMPMetadata object
	 * @throws XMPException
	 */
	XMPMetadata parse( String input, Map<String, Object> parseContext ) throws XMPException;
	
	/**
	 * Parse XMP in RDF/XML form from a byte buffer. The data must be encoded with UTF-8.
	 * The parser will simply ignore sub-trees that contain invalid XMP data. 
	 * It will only throw on serious errors.
	 * @param input buffer that contains an XMP packet serialized as RDF/XML
	 * @param parseContext a map to provide options for the parser, 
	 * 		  or return additional information other than the XMP data model;
	 * 		  can be null if not needed
	 * @return an XMPMetadata object
	 * @throws XMPException
	 */
	XMPMetadata parse( byte[] input, Map<String, Object> parseContext ) throws XMPException;
}
