// =================================================================================================
// ADOBE SYSTEMS INCORPORATED
// Copyright 2011 Adobe Systems Incorporated
// All Rights Reserved
//
// NOTICE:  Adobe permits you to use, modify, and distribute this file in accordance with the terms
// of the Adobe license agreement accompanying it.
// =================================================================================================

package com.adobe.xmp.schema.model;


/**
 * Describes a simple XMP type.
 * All simple types are based on five basic types:
 * <ul>
 * 		<li>Text</li>
 * 		<li>Integer</li>
 * 		<li>Real</li>
 * 		<li>Boolean</li>
 * 		<li>Date</li>
 * </ul>
 * Additionally, a rule can be added to contrain a simple property,
 * typically a regular expression for Text or an algebraic expression for Integer,
 * e.g. the MIMEType or Rating property.
 * 
 * @author Stefan Makswit
 */
public interface SimpleType extends PropertyType
{
	// Basic Value Types according to the XMP specification
	
	/** Name of Boolean */ 
	final String TYPE_BOOLEAN = "Boolean";
	/** Name of Date */ 
	final String TYPE_DATE = "Date";
	/** Name of Integer */ 
	final String TYPE_INTEGER = "Integer";
	/** Name of Real */ 
	final String TYPE_REAL = "Real";
	/** Name of Text */ 
	final String TYPE_TEXT = "Text";

	
	/** Enum of all five basic XMP simple types */
	public enum BasicType
	{
		/** literal type "Boolean" */
		BOOLEAN(TYPE_BOOLEAN),
		/** literal type "Date" */
		DATE(TYPE_DATE),
		/** literal type "Integer" */
		INTEGER(TYPE_INTEGER),
		/** literal type "Read" */
		REAL(TYPE_REAL),
		/** literal type "Text" */
		TEXT(TYPE_TEXT);

		/** stores the XMP type name for each enum item */
		private final String typeName;

		BasicType(String typeName)
		{
			this.typeName = typeName;
		}

		
		@Override
		public String toString()
		{
			return typeName;
		}
	}
	

	/**
	 * One of the basic types "Boolean", "Date", "Integer", "Real" or "Text".
	 * @return Returns the simple type this type is inherited from.
	 */ 
	BasicType getBasicType();
	
	
	/**
	 * @return Returns the inherited simple type. If it is a basic type,
	 * 		<code>null</code> is returned
	 */
	SimpleType getInherited();
}
