/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and may be covered by U.S. and Foreign Patents,
 * patents in process, and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.analytics.sitecatalyst;

import java.util.Collection;

import org.apache.sling.commons.json.JSONArray;
import org.apache.sling.commons.json.JSONObject;

import com.day.cq.wcm.webservicesupport.Configuration;

/**
 * Provides methods for Analytics Classification API related calls as specified
 * by the <a href=
 * "https://experiencecloud.adobe.com/developer/documentation/classifications-1-4-saint/classifications-api"
 * >Classifications API</a>.
 * 
 * @since 6.1
 */
public interface ClassificationsService {

    /**
     * Checks for the status of a job.
     * 
     * @param configuration Web service support configuration
     * @param jobId Classification job identifier
     * @return A JSONObject containing the job information. The object is of the
     *         following structure:
     *         <p>
     *         <code>
     *         {<br>
     *         &nbsp; id: (int),<br>
     *         &nbsp; type: (string: "Job"|"File"),<br>
     *         &nbsp; viewable_pages: (int),<br>
     *         &nbsp; status: (string: "Waiting For User Data"|"In Progress"|"Completed"|"Completed--With Errors")<br>
     *         }<br>
     *         </code>
     *         </p>
     * @throws SitecatalystException {@link SitecatalystException}
     */
    JSONObject getStatus(Configuration configuration, int jobId) throws SitecatalystException;

    /**
     * Returns the available elements for a given report suite, and compatibility across report suites.
     * 
     * @param configuration Web service support configuration
     * @param reportSuites Report suite identifiers
     * @return A JSONArray of JSONObjects containing compatible elements. The
     *         objects are of the following structure:
     *         <p>
     *         <code>
     *         {<br>
     *         &nbsp; id: (int),<br>
     *         &nbsp; name: (string)<br>
     *         }
     *         </code>
     *         </p>
     * @throws SitecatalystException {@link SitecatalystException}
     */
    JSONArray getCompatibilityElements(Configuration configuration, String... reportSuites) throws SitecatalystException;

    /**
     * Returns classification export filters. The returned data consists of
     * values that can be passed in as parameters of {@link #createExport},
     * including column numbers and names, and valid dates for filtering data
     * rows in a classifications export.
     * 
     * @param configuration Web service support configuration
     * @param element Element for which you want to get file information
     * @param reportSuites Array of report suite identifiers
     * @return A JSONArray of JSONObjects containing the information about valid
     *         data filters for a classification job. The objects are of the
     *         following structure:
     *         <p>
     *         <code>
     *         {<br>
     *         &nbsp; filter_dates: [{date: (string)},...]<br>
     *         &nbsp; columns: [{num: (int), name: (string)},...]<br>
     *         }
     *         </code>
     *         </p>
     * @throws SitecatalystException {@link SitecatalystException}
     */
    JSONArray getFilters(Configuration configuration, int element, String[] reportSuites)
            throws SitecatalystException;

    /**
     * Creates a classifications export job. This file represents
     * classifications for a given relation. Before calling this API, use
     * {@link #getFilters(Configuration, int, String[])} to obtain valid column numbers and names, and filter
     * dates. After creating an export job, use {@link #getStatus(Configuration, int)} to determine when the
     * export data is ready for download. When ready, use {@link #getExport} to retrieve
     * the data.
     * 
     * @param configuration Web service support configuration
     * @param campaignStartDate The campaign start date. Accepts any date string
     *            parseable by php:date(). Accepts the keywords ::all:: and
     *            ::active:: to filter by all campaigns, or by only active ones.
     * @param campaignEndDate The campaign end date. Accepts any date string
     *            parseable by php:date().
     * @param filterStartDate The start date for including rows in the export.
     *            Accepts any date string parseable by php:date().
     * @param filterEndDate The end date for including rows in the export.
     *            Accepts any date string parseable by php:date().
     * @param email E-Mail address to receive job notifications.
     * @param encoding The language encoding to use with the export file. For
     *            example, UTF-8.
     * @param element Relation identifier of classification the import should be
     *            done for. You get this ID as a return value from
     *            {@link #getCompatibilityElements(Configuration, String...)}.
     * @param reportSuites (Optional) The list of report suites on which
     *            classifications exports data.
     * @param rowFilterColumnName (Optional) The column name that
     *            classifications checks for cell values that match the value
     *            specified in the row_match_filter_match_column_value
     *            parameter.
     * @param rowFilterColumnValue (Optional) Include this parameter when using
     *            the row_match_filter_match_column_name parameter. If the cell
     *            value matches the value, classifications includes it in the
     *            export. Include ::empty:: to filter empty rows.
     * @param rowCount (Optional) Limits the number of data rows in the export
     *            file to the specified value. The default is 1000.
     * @param quoteOutput Whether to quote the output. In some cases, turning
     *            this on helps with duplicate keys containing whitespace.
     * @return Export job identifier
     * @throws SitecatalystException {@link SitecatalystException}
     */
    int createExport(Configuration configuration, 
                     String campaignStartDate,
                     String campaignEndDate,
                     String filterStartDate, 
                     String filterEndDate,
                     String email, 
                     String encoding, 
                     int element, 
                     String[] reportSuites,
                     String rowFilterColumnName, 
                     String rowFilterColumnValue, 
                     int rowCount, 
                     boolean quoteOutput) throws SitecatalystException;

    /**
     * This is a convenience method for {@link  ClassificationsService#createExport(Configuration, String, String, String, String, String, String, int, String[], String, String, int, boolean)}
     * @param configuration Web service support configuration
     * @param campaignStartDate The campaign start date. Accepts any date string
     *            parseable by php:date(). Accepts the keywords ::all:: and
     *            ::active:: to filter by all campaigns, or by only active ones.
     * @param campaignEndDate The campaign end date. Accepts any date string
     *            parseable by php:date().
     * @param filterStartDate The start date for including rows in the export.
     *            Accepts any date string parseable by php:date().
     * @param filterEndDate The end date for including rows in the export.
     *            Accepts any date string parseable by php:date().
     * @param email E-Mail address to receive job notifications.
     * @param encoding The language encoding to use with the export file. For
     *            example, UTF-8.
     * @param element Relation identifier of classification the import should be
     *            done for. You get this ID as a return value from
     *            {@link #getCompatibilityElements(Configuration, String...)}
     * @return result as provided by {@link  ClassificationsService#createExport(Configuration, String, String, String, String, String, String, int, String[], String, String, int, boolean)}
     * @throws SitecatalystException  #createExport(Configuration, String, String, String, String, String, String, int, String[], String, String, int, boolean).
     * 
     * @see ClassificationsService#createExport(Configuration, String, String, String, String, String, String, int, String[], String, String, int, boolean)
     */
    int createExport(Configuration configuration, 
            String campaignStartDate,
            String campaignEndDate,
            String filterStartDate, 
            String filterEndDate,
            String email, 
            String encoding, 
            int element) throws SitecatalystException;
    
    /**
     * Retrieves a page of data, known as a file segment, from a completed
     * classifications job.
     * 
     * @param configuration Web service support configuration
     * @param id The ID of the export file you want to view. You can get this
     *            value from {@link #getStatus(Configuration, int)}.
     * @param page The number of the data page you want to view from the export
     *            job. Default is 1. The viewable_pages value in
     *            {@link #getStatus(Configuration, int)} indicates the number of
     *            data pages in the job. Valid values for page are between 1 and
     *            the value of viewable_pages.
     * @return A JSONArray of JSONObjects containing information from a single
     *          data page of a classifications export job. The objects are of
     *          the following structure:
     *          <p>
     *          <code>
     *          {<br>
     *          &nbsp; warnings: [(string),...]<br>
     *          &nbsp; header: [(string),...]<br>
     *          &nbsp; data: [[(string)],...]<br>
     *          }
     *          </code>
     *          </p>
     * @throws SitecatalystException {@link SitecatalystException}
     */
    JSONArray getExport(Configuration configuration, int id, int page) throws SitecatalystException;
    
    /**
     * Retrieves a template to use for importing classifications data.
     * 
     * @param configuration Web service support configuration
     * @param reportSuites The report suite where you want to import data
     * @param element relation_id for which you want to get a template (e.g.
     *            101=evar1, 51=product, etc)
     * @param encoding The encoding to use for the template
     * @param classifications Optional array of text classification names to
     *            include as columns in the template
     * @return A JSONObject containing the template data for a classifications
     *         import. The objects are of the following structure:
     *         <p>
     *         <code>
     *         {<br>
     *         &nbsp; rsid: (string),<br>
     *         &nbsp; site_title: (string),<br>
     *         &nbsp; template: (string),<br>
     *         }
     *         </code>
     *         </p>
     * @throws SitecatalystException {@link SitecatalystException}
     */
    JSONObject getTemplate(Configuration configuration, 
                                String[] reportSuites, 
                                String element, 
                                String encoding,
                                String... classifications) throws SitecatalystException;

    /**
     * Creates an import job. To successfully submit an import, call
     * {@link ClassificationsService#createImport(Configuration, String, String, String[], String, boolean, boolean, boolean, String...)}
     * to specify the parameters for the Import job and the header columns to be
     * classified. After submitting the job, you can iteratively populate row
     * data by calling
     * {@link ClassificationsService#populateImport(Configuration, int, int, Collection)}.
     * After sending all data, call
     * {@link ClassificationsService#commitImport(Configuration, int)} to finalize the Import
     * job and submit it to the processing queue.
     * 
     * @param configuration Web service support configuration
     * @param description Description of the import job.
     * @param email E-Mail address to receive job notifications.
     * @param header Array of column values for classification.
     * @param element The report for which you want to perform a classifications
     *            import.
     * @param checkDivisions Specifies whether to check report suites for
     *            compatible divisions.
     * @param exportResults Specifies whether to automatically perform an export
     *            when the import job finishes processing.
     * @param overwrite Specifies whether to overwrite data when conflicts
     *            occur.
     * @param reportSuites Optional list of report suites to receive the
     *            import job.
     * @return Import job identifier
     * @throws SitecatalystException {@link SitecatalystException}
     */
    public int createImport(Configuration configuration, 
                            String description, 
                            String email, 
                            String[] header,
                            String element, 
                            boolean checkDivisions, 
                            boolean exportResults, 
                            boolean overwrite, 
                            String... reportSuites) throws SitecatalystException;

    /**
     * Appends row data to a previously created import job. Data can be
     * separated into pages, but the pages must be sent sequentially (either
     * ascending or descending) to avoid errors when jobs are processed.
     * <blockquote>Note: Individual job pages cannot exceed 25,000
     * rows.</blockquote>
     * 
     * @param configuration Web service support configuration
     * @param jobId Job identifier to append row data
     * @param page Page number that identifies the sequential position in the
     *            import job. For example, if you submit a spreadsheet
     *            with 1000 values, divided into 4 pages, page 1 includes values
     *            1 - 250, page 2 includes values 251 - 500, and so on.
     * @param rows Collection of row data that matches the column heading
     * @return <code>true</code> if population was successful,
     *         <code>false</code> otherwise.
     * @throws SitecatalystException {@link SitecatalystException}
     */
    boolean populateImport(Configuration configuration, int jobId, int page, Collection<String[]> rows)
            throws SitecatalystException;

    /**
     * Commits the import for processing.
     * 
     * @param configuration Web service support configuration
     * @param jobId Import job identifier
     * @return <code>true</code> if job has been processed successfully,
     *         <code>false</code> otherwise.
     * @throws SitecatalystException {@link SitecatalystException}
     */
    boolean commitImport(Configuration configuration, int jobId) throws SitecatalystException;

}
