/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2011 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and may be covered by U.S. and Foreign Patents,
 * patents in process, and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.analytics.sitecatalyst;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.jcr.Node;

import com.day.cq.analytics.sitecatalyst.impl.servlets.SitecatalystServlet;
import org.apache.sling.api.SlingHttpServletRequest;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;
import org.apache.sling.settings.SlingSettingsService;

import com.day.cq.commons.jcr.JcrConstants;
import com.day.cq.wcm.api.Page;
import com.day.cq.wcm.webservicesupport.Configuration;

public class SitecatalystUtil {

    private SitecatalystUtil() {
    }

    /**
     * Returns server URL according to request scheme. If the scheme is HTTP the
     * tracking server is returned otherwise the secure tracking server is
     * returned.
     * 
     * @param request  the request
     * @param config the configuration
     * @return server URL
     */
    public static String getServer(SlingHttpServletRequest request, Configuration config) {
        String server = "";
        if (request.getScheme().equals("http")) {
            server = "http://" + config.getInherited("cq:trackingServer", "");
        } else {
            server = "https://" + config.getInherited("cq:trackingServerSecure", "");
        }
        return server;
    }

    /**
     * Returns a comma separated list of report suites depending on the current
     * run mode of the server. If no report suite matches the current run mode
     * an empty string is returned.
     * 
     * @param settingsService  SlingSettingsService
     * @param configuration the configuratin
     * @return comma separated list of report suites for runmode of current server
     */
    public static String getReportSuites(SlingSettingsService settingsService, Configuration configuration) {
        String[] accounts = configuration.getInherited("reportsuites", new String[0]);
        Set<String> runModes = settingsService != null ? settingsService.getRunModes() : new HashSet<String>();

        String s_account = "";
        boolean first = true;
        for (int i = 0; i < accounts.length; i++) {
            String rsid, runMode;
            if (accounts[i].indexOf(";") != -1) {
                rsid = accounts[i].substring(0, accounts[i].indexOf(";"));
                runMode = accounts[i].substring(accounts[i].indexOf(";") + 1);
            } else {
                rsid = accounts[i];
                runMode = "";
            }
            if ((runMode.equals("")) || (runModes.contains(runMode))) {
                s_account += (first ? "" : ",") + rsid;
                first = false;
            }
        }

        return s_account;
    }

    /**
     * Returns number of domain name periods.
     * 
     * @param request the request
     * @param server the server url
     * @return number ofdomain name periods
     */
    public static Integer getCookieDomainNamePeriod(SlingHttpServletRequest request, String server) {
        Integer cookieDomainNamePeriod = 0;
        if (server.contains("2o7.net") || server.contains("omtrdc.net")) {
            cookieDomainNamePeriod = request.getServerName().split("\\.").length - 1;
        } else {
            cookieDomainNamePeriod = server.split("\\.").length - 1;
        }
        cookieDomainNamePeriod = (cookieDomainNamePeriod < 0) ? 0 : cookieDomainNamePeriod;
        return cookieDomainNamePeriod;
    }

    /**
     * Formats the resource path of a Resource <code>pageResource</code>
     * relative to the provided
     * {@link com.day.cq.wcm.webservicesupport.Configuration} path. A trailing
     * slash will be removed if present, all other slashes will be replaced by
     * colons.
     * <p>
     * <code>/content/geometrixx/en/support</code> will be formatted to
     * <code>content:geometrixx:en:support</code>
     * </p>
     * @param pageResource  page Resource
     * 
     * @param configuration SiteCatalyst configuration
     * @return Formatted path
     */
    public static String getFormattedPagePath(Resource pageResource, Configuration configuration) {
        String fmtPath = pageResource.getPath();
        String cfgPath = configuration.getPath();
        if (cfgPath != null && fmtPath.startsWith(cfgPath)) {
            fmtPath = fmtPath.replaceFirst(cfgPath, "");
        }
        if (fmtPath.startsWith("/")) {
            fmtPath = fmtPath.replaceFirst("/", "");
        }
        fmtPath = fmtPath.replaceAll("/", ":");
        return fmtPath;
    }

    /**
     * Ascends the {@link Resource} until a {@link Page} with a valid analytics
     * configuration is found. The configuration is considered valid if the node
     * has a property cq:services.
     * 
     * @param resolver {@link ResourceResolver}
     * @param resource {@link Resource}
     * @return Analytics resource or <code>null</code> if none could be found.
     */
    public static Resource findAnalyticsResource(ResourceResolver resolver, Resource resource) {
        try {
            String resourcePath = resource.getPath();
            while (resourcePath.lastIndexOf("/") > 0) {
                resource = resolver.getResource(resourcePath);
                if (resource != null) {
                    Resource res = resource.getChild(JcrConstants.JCR_CONTENT + "/analytics");
                    if (res != null) {
                        Node jcrContent = resource.getChild(JcrConstants.JCR_CONTENT).adaptTo(Node.class);
                        Node node = res.adaptTo(Node.class);
                        if (node.hasProperty("cq:services") || jcrContent.hasProperty("cq:cloudserviceconfigs")) {
                            return res;
                        }
                    }
                }
                resourcePath = resourcePath.substring(0, resourcePath.lastIndexOf("/"));
            }
        } catch (Exception ignored) {
        }
        return null;
    }

    /**
     * Returns one of the configured report suites based on the following rules:
     * return first publish report suite, if not, return first report suite for all
     * run modes and if still nothing available return first author report suite
     * 
     * @param config the Analytics configuration
     * @return the preferred report suite ID
     */
    public static String getPublishPreferredReportSuite(Configuration config) {
        String[] reportSuites = config.getInherited(SitecatalystServlet.PN_REPORTSUITES, new String[0]);
        Map<String, List<String>> reportingSuitesPerRunMode = new HashMap<String, List<String>>();
        for (String reportSuite : reportSuites) {
            String runMode = "all";
            
            String[] tmp = reportSuite.split(";");
            if (tmp.length == 2) {
                runMode = tmp[1];
            }
            
            List<String> suites = reportingSuitesPerRunMode.get(runMode);
            if (suites == null) {
                suites = new ArrayList<String>();
            }
            suites.add(tmp[0]);
            
            reportingSuitesPerRunMode.put(runMode, suites);
        }
        String reportSuite = null;
        if (reportingSuitesPerRunMode.get("publish") != null) {
            reportSuite = reportingSuitesPerRunMode.get("publish").get(0);
        } else if (reportingSuitesPerRunMode.get("all") != null) {
            reportSuite = reportingSuitesPerRunMode.get("all").get(0);
        } else if (reportingSuitesPerRunMode.get("author") != null) {
            reportSuite = reportingSuitesPerRunMode.get("author").get(0);
        }
        return reportSuite;
    }

}
